/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

/*!
 * @file	imageInfoUR.h
 * @brief	This file contains the ImageInfoUR class.
 * @author 	Info-Dev
 * @author	Diego CACERES (diego.caceres[AT]creatis.insa-lyon.fr)
 * @date 	2011-11-15
 */

#ifndef IMAGEINFO_H_
#define IMAGEINFO_H_

#include <iostream>
#include <string>
#include <cstdio>
#include <vtkVersionMacros.h>
#include <vtkImageData.h>
#include <vtkMetaImageWriter.h>
#include <vtkMetaImageReader.h>
#include <vtkSmartPointer.h>
#include "regionStructUR.h"
#include "imageModificationManager.h"

/*! @class ImageInfoUR imageInfoUR.h "imageInfoUR.cxx"
 *	@brief This class contains the information of the undo/redo
 *	@details This class contains the two images (undo and redo) and the ImageMManager.
 */
class ImageInfoUR {

public:
	//typedef definition
	// ----------------------------------------------------------------------------------
	/*!	@typedef vtkSmartPointer<vtkImageData> VTKImageDataPointerType;
	 *	@brief Defines the pointer of vtkImageDate type.
	 */
	typedef vtkSmartPointer<vtkImageData> VTKImageDataPointerType;
	// ----------------------------------------------------------------------------------
	/*!	@typedef vtkSmartPointer<vtkMetaImageReader> VTKMetaImageReaderPointerType;
	 *	@brief Defines the pointer of vtkMetaImageReader type.
	 */
	typedef vtkSmartPointer<vtkMetaImageReader> VTKMetaImageReaderPointerType;
	// ----------------------------------------------------------------------------------
	/*!	@typedef vtkSmartPointer<vtkMetaImageWriter> VTKMetaImageWriterPointerType;
	 *	@brief Defines the pointer of vtkMetaImageWriter type.
	 */
	typedef vtkSmartPointer<vtkMetaImageWriter> VTKMetaImageWriterPointerType;
	// ----------------------------------------------------------------------------------
	/*!	@typedef std::string StringType;
	 *	@brief Redefinition of the string type.
	 */
	typedef std::string StringType;
	// ----------------------------------------------------------------------------------
	/*!	@typedef ImageMManager ImageMManagerType;
	 *	@brief Definition of the ImageMManager type.
	 */
	typedef ImageMManager ImageMManagerType;
	// ----------------------------------------------------------------------------------
public:
	// ----------------------------------------------------------------------------------
	/*! @fn ImageInfoUR();
	 * @brief This is the default constructor.
	 */
	ImageInfoUR();
	// ----------------------------------------------------------------------------------
	/*! @fn virtual ~ImageInfoUR();
	 * @brief This is the destructor.
	 */
	virtual ~ImageInfoUR();
	// ----------------------------------------------------------------------------------
	/*! @fn void SetImageName(const StringType &imgName);
	 * @brief This method sets the image name for the undo/redo.
	 * @param imgName The imageName.
	 */
	void SetImageName(const StringType &imgName);
	// ----------------------------------------------------------------------------------
	/*! @fn void SetImageMManager(ImageMManagerType *imMManager);
	 * @brief This method set the image modification manager.
	 * @param imMManager The pointer of the manager.
	 */
	void SetImageMManager(ImageMManagerType *imMManager);
	// ----------------------------------------------------------------------------------
	/*! @fn void SetStatus(const bool& onMemory);
	 * @brief This method sets the memory status of the images.
	 * @param onMemory This is true if the images are in the memory.
	 */
	void SetStatus(const bool& onMemory);
	// ----------------------------------------------------------------------------------
	/*! @fn void SetImages(VTKImageDataPointerType imgUndo,
	 VTKImageDataPointerType imgRedo);
	 * @brief This method sets the undo/redo Images.
	 * @param imgUndo The undo image.
	 * @param imgRedo The redo image.
	 */
	void SetImages(VTKImageDataPointerType imgUndo,
			VTKImageDataPointerType imgRedo);
	// ----------------------------------------------------------------------------------
	/*! @fn void LoadImagesToMemory(const StringType& gPath);
	 * @brief This method takes the images from disk and load them to the principal memory.
	 * @param gPath The global path where the images are stored.
	 */
	void LoadImagesToMemory(const StringType& gPath);
	// ----------------------------------------------------------------------------------
	/*! @fn void RemoveImagesFromMemory(const StringType& gPath);
	 * @brief This method removes the images from the principal memory.
	 * @param gPath The global path where the images are stored.
	 */
	void RemoveImagesFromMemory(const StringType& gPath);
	// ----------------------------------------------------------------------------------
	/*! @fn void SaveImagesOnDisk(const StringType& gPath);
	 * @brief This method stores the images in the hard disk.
	 * @param  gPath The global path where the images are stored.
	 */
	void SaveImagesOnDisk(const StringType& gPath);
	// ----------------------------------------------------------------------------------
	/*! @fn void RemoveImagesFromDisk(const StringType& gPath);
	 * @brief This method removes the images from the hard disk (permanently)
	 * @param gPath The global path where the images are stored.
	 */
	void RemoveImagesFromDisk(const StringType& gPath);
	// ----------------------------------------------------------------------------------
	/*! @fn VTKImageDataPointerType GetUndoImage();
	 * @brief This method returns the undo image.
	 * @return
	 */
	VTKImageDataPointerType GetUndoImage();
	// ----------------------------------------------------------------------------------
	/*! @fn VTKImageDataPointerType GetRedoImage();
	 * @brief This method returns the redo image.
	 * @return
	 */
	VTKImageDataPointerType GetRedoImage();
	// ----------------------------------------------------------------------------------
	/*! @fn StringType GetImageName();
	 * @brief This method returns the ImagesName.
	 * @return
	 */
	StringType GetImageName();
	// ----------------------------------------------------------------------------------
	/*! @fn ImageMManagerType* GetImageMManager();
	 * @brief This method returns the Image Modification Manager.
	 * @return
	 */
	ImageMManagerType* GetImageMManager();
	// ----------------------------------------------------------------------------------
	/*! @fn void SaveImageAsMHD(const StringType& filename,
	 VTKImageDataPointerType image);
	 * @brief This method saves an image in a MHD format.
	 * @param filename The filename.
	 * @param VTKImageDataPointerType image.
	 */
	void SaveImageAsMHD(const StringType& filename,
			VTKImageDataPointerType image);
	// ----------------------------------------------------------------------------------
	/*! @fn bool GetStatus();
	 * @brief This method returns true if the images are located in the principal memory.
	 * @return
	 */
	bool GetStatus();
	// ----------------------------------------------------------------------------------
private:
	VTKImageDataPointerType m_UndoImage; 		//!<This is the undo image.
	VTKImageDataPointerType m_RedoImage; 		//!<This is the redo image
	StringType 				m_ImageName; 		//!<This is the base name of the images.
	ImageMManagerType		*m_ImageMManager; 	//!<This is the modification manager of the images.
	bool 					m_OnMemory; 		//!<This is true if the images are in memory.
	bool 					m_OnDisk; 			//!<This is true if the images are in disk.
};

#endif /* IMAGEINFO_H_ */
