/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#include "vtkObjectFactory.h"

#include "vtkActor.h"
#include "vtkCylinderSource.h"
#include "vtkPolyDataMapper.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkProperty.h"
#include "vtkCamera.h"
#include "vtkPoints.h"
#include "vtkCellArray.h"
#include "vtkPolyData.h"
#include "vtkPolyDataMapper.h"
#include "vtkPolyDataWriter.h"
#include "vtkPolyDataReader.h"
#include "vtkImageReader.h"
#include "vtkImageViewer.h"
#include "vtkImageViewer2.h"
#include "vtkImageToStructuredPoints.h"
#include <vtkImageThreshold.h> 

#include "vtkExtractVOI.h"
#include "vtkImageClip.h"
#include "vtkImageResample.h"
#include "vtkImageThreshold.h"
#include "vtkImageCast.h"
#include "vtkImageSeedConnectivity.h"
#include "vtkImageData.h"
#include "vtkMarchingCubes.h"
#include "vtkImageReslice.h"
#include "vtkTransform.h"
#include "vtkSphereSource.h"
#include "vtkDoubleArray.h"
#include "vtkPointData.h"
#include "vtkCommand.h"
#include "vtkCallbackCommand.h"
#include "vtkImageResample.h"
#include "vtkMath.h"
#include "vtkStripper.h"

#include "wxEmptyPanelWidget_2.h"

#include <wx/splitter.h>



//-------------------------------------------------------------------
//-------------------------------------------------------------------
//-------------------------------------------------------------------
vtkStandardNewMacro(TreeExtraction_MH_JFC);
//-------------------------------------------------------------------

TreeExtraction_MH_JFC::TreeExtraction_MH_JFC(){
	_branchLevel		= 2;
	_imagedataResult	= NULL;
	_cleanLevel			= 400;
}
//-------------------------------------------------------------------
TreeExtraction_MH_JFC::~TreeExtraction_MH_JFC(){
	Reset();
}
//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::Reset(){
	if (_imagedataResult!=NULL){
		_imagedataResult->Delete();
	}
}
//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::SetInput(marExperiment *input){
	_marExperiment = input;
}

//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::BruleVolume(vtkImageData *imagedata, int x,int y,int z,int r , int value)
{
	int ext[6];
	int i, j, k, ii,jj,kk;	
	imagedata->GetExtent(ext);
	unsigned short *ptr;
	int x1=x-r;
	int x2=x+r;
	int y1=y-r;
	int y2=y+r;
	int z1=z-r;
	int z2=z+r;
	double tmpsqrt;
	for(i=x1;i<=x2;i++){
		for(j=y1;j<=y2;j++){
			for(k=z1;k<=z2;k++){
				ptr=(unsigned short *)	imagedata->GetScalarPointer(i,j,k);
				ii=i-x;
				jj=j-y;
				kk=k-z;
				tmpsqrt=ii*ii + jj*jj + kk*kk;
				if ((ptr!=NULL) && ( sqrt(tmpsqrt)<=r))
				{
					if (value==0){
						*ptr=value;
					} else {
						if (*ptr==0)  {
							*ptr=value;
						}
					}
				} 
			} // for z
		}  // for y
	} // fox x
}

//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::BruleVolume(vtkImageData *imagedata,vtkPolyData *inputAxisRadio, int value)
{
	int x,y,z,r;
	double point[3],radio;
	double fractpart, intpart;
	int i,maxPoints=inputAxisRadio->GetPoints()->GetNumberOfPoints();
	vtkDoubleArray *allRadios = (vtkDoubleArray*)(inputAxisRadio->GetPointData()->GetScalars("radio"));
	for (i=0; i<maxPoints; i++){	
		inputAxisRadio->GetPoints()->GetPoint(i,point);
		radio=allRadios->GetValue(i);
		x = (int)(point[0]);
		y = (int)(point[1]);	
		z = (int)(point[2]);
		fractpart = modf (radio , &intpart);
		if (fractpart>=0.5){
		 intpart++;
		}
		r = (int)( intpart );
		BruleVolume(imagedata,x,y,z,r,value);
	}
}


//-------------------------------------------------------------------

void TreeExtraction_MH_JFC::FindVit( marExperiment *newExperiment , int actualAxis , double *pointBif , double *vit )
{
	double dist,min=-1;
	double point[3];
	double dx,dy,dz;
	int jPoint,maxPoints;
	int iAxis,maxAxis;

	maxAxis=newExperiment->getNumberOfAxes();
	for (iAxis=0;iAxis<maxAxis;iAxis++){
		if (iAxis!=actualAxis){
			newExperiment->setAxis(iAxis);
			marAxis *maraxis = newExperiment->getAxis( );
			if (maraxis!=NULL) {
				vtkPolyData  *axisVtk	= maraxis->GetAxisData();
				maxPoints=axisVtk->GetPoints()->GetNumberOfPoints();
				for (jPoint=0; jPoint<maxPoints; jPoint++){	
					axisVtk->GetPoints()->GetPoint(jPoint,point);
					dx = pointBif[0]-point[0];
					dy = pointBif[1]-point[1];
					dz = pointBif[2]-point[2];
					dist = sqrt( dx*dx + dy*dy + dz*dz );
					if ((min==-1) || (dist<min)) {
						min=dist;
						vit[0] = dx;
						vit[1] = dy;
						vit[2] = dz;
					} //if 
				} // for
			} // if maraxis!=NULL
		} // i!=actualAxis
	} //for
}

//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::Execute(int count, int sens, double *vit, int cleanLevel)
{
 	//------------Arbre Extraction --------------
	// Call Maracas axis extraction
	count--;
/*
time_t start,end;
double dif;
time (&start);
*/
		int idActualAxis_Before = _marExperiment->getNumberOfAxes();
		_marExperiment->extractVascularTree( sens , vit, cleanLevel );
/*
time (&end);
dif = difftime (end,start);
FILE *ff;
ff=fopen("c:/temp/MaracasTIME.txt","a+");
fprintf(ff,"\n TreeExtraction_MH_JFC::Execute %d \n",count);
fprintf(ff,"evt %.2lf \n",dif);
fclose(ff);
*/
		int idActualAxis = _marExperiment->getNumberOfAxes();
		// Bifurcation Detection JFC
		marAxis *maraxis = _marExperiment->getAxis( );
		if ((maraxis!=NULL) && (idActualAxis_Before!=idActualAxis)) {
			vtkImageData *imagedata			= _marExperiment->getDynData( )->getVolume( )->castVtk();
			vtkPolyData  *inputAxisRadio	= maraxis->GetAxisData();

			if (count>0){
/*
time (&start);
*/
				_prgov = carotidaBifurcacion::New();
				_prgov->SetMaxPropRadio(5);
				_prgov->SetHumbral(0.45);
				_prgov->SetMaxPropMasa(10);
				_prgov->SetMinPropMasa(0.01);
				_prgov->SetInput( inputAxisRadio );
				_prgov->SetInput2( imagedata );
				_prgov->Update();
/*
time (&end);
dif = difftime (end,start);
ff=fopen("c:/temp/MaracasTIME.txt","a+");
fprintf(ff,"\tcB  %.2lf \n",dif);
fclose(ff);
*/
			}

			// Brule Volume
			BruleVolume(imagedata		 , inputAxisRadio , 0			 );
			BruleVolume(_imagedataResult , inputAxisRadio , idActualAxis );
			inputAxisRadio->Delete();


			if (count>0){
				// LOOP for branch extraction
				vtkPolyData *bifurcations =  _prgov->GetOutput();
				int i,maxPoints = bifurcations->GetPoints()->GetNumberOfPoints();
				int x,y,z;
				double point[3], vit[3];
				for (i=0; i<maxPoints; i++){
					bifurcations->GetPoints()->GetPoint(i,point);
					x = (int)(point[0]);
					y = (int)(point[1]);
					z = (int)(point[2]);
					FindVit( _marExperiment , idActualAxis , point , vit );
					_marExperiment->setStartPoint( x , y , z );
					//Branch extraction;
					Execute(count, 0 , vit , (int)_cleanLevel );
				}
				bifurcations->Delete();
//				_prgov->Delete();
//				_prgov = NULL;
			} // of count>0

		} // maraxi=NULL
}

//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::InitImagaDataResult(){
	Reset();
	vtkImageData *imagedata = _marExperiment->getDynData( )->getVolume( )->castVtk();
	imagedata->Update();
	_imagedataResult = vtkImageData::New();
	int dim[3];
	imagedata->GetDimensions(dim);
	_imagedataResult->SetScalarType(imagedata->GetScalarType());

	_imagedataResult->SetDimensions(dim[0],dim[1],dim[2]); 
	_imagedataResult->SetExtent( 0 , dim[0]-1 , 0 , dim[1]-1 , 0 , dim[2]-1 ); 
	_imagedataResult->SetWholeExtent(0 , dim[0]-1 , 0 , dim[1]-1 , 0 , dim[2]-1 );
	_imagedataResult->SetSpacing( imagedata->GetSpacing()  ); 
	_imagedataResult->SetOrigin( 0,0,0  ); 
	_imagedataResult->Update();
	_imagedataResult->UpdateData();
	unsigned short *ptr;
	int i,j,k;
	for ( i=0 ; i<dim[0] ; i++ ){
  		for ( j=0 ; j<dim[1] ; j++ ){
			for ( k=0 ; k<dim[2] ; k++ ){
				ptr = (unsigned short *)	_imagedataResult->GetScalarPointer(i,j,k);
				*ptr = 0;
			}
		}
	}
}
//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::Update(){
/*
FILE *ff;
ff=fopen("c:/temp/MaracasTIME.txt","w");
fclose(ff);
*/
	InitImagaDataResult();
	Execute(_branchLevel);
}
//-------------------------------------------------------------------
marExperiment* TreeExtraction_MH_JFC::GetInput(){
	return _marExperiment;
}
//-------------------------------------------------------------------
vtkImageData* TreeExtraction_MH_JFC::GetVolume(){
	return _imagedataResult;
}
//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::SetBranchLevel(int branchLevel){
	_branchLevel = branchLevel;
}
//-------------------------------------------------------------------
void TreeExtraction_MH_JFC::SetCleanTreeLevel( int cleanLevel ){
	_cleanLevel = (double)cleanLevel;
}

//-------------------------------------------------------------------
//-------------------------------------------------------------------
//-------------------------------------------------------------------
wxEmptyPanelWidget_2::wxEmptyPanelWidget_2(wxWindow *parent, marInterface* mar)
: wxPanel( parent, -1) 
{

	_sensibility_JF		= 0.5;

	_mar								= mar;
	wxBoxSizer			*sizer			= new wxBoxSizer(wxVERTICAL  );
    wxSplitterWindow	*pnlSplitter	= new wxSplitterWindow( this , -1);
	wxPanel				*viewPanel		= CreateViewPanel(pnlSplitter);
	wxPanel				*controlPanel	= CreateControlPanel(pnlSplitter);

	sizer		-> Add( pnlSplitter ,1,wxGROW  ,0);
	pnlSplitter	-> SetMinimumPaneSize( 150 );
    pnlSplitter	-> SplitVertically( viewPanel, controlPanel, 600 );
	this		-> SetSizer(sizer);

	// Original Volume
	_mCubes				= NULL;
	_surfMapper			= NULL;
	_surfActor			= NULL;

	// Result 1: Volume + Axis
	_1_prgov			= NULL;
	_1_mapfinal			= NULL;
	_1_stripfinal		= NULL;
	_1_isoMapperMC6		= NULL;
	_1_isoActorMC6		= NULL;
	_1_isoMC6			= NULL;

	// Result 2: Volume + Axis
	_2_prgov			= NULL;
	_2_mapfinal			= NULL;
	_2_stripfinal		= NULL;
	_2_isoMapperMC6		= NULL;
	_2_isoActorMC6		= NULL;
	_2_isoMC6			= NULL;

	//
	_isoMapperMC1		= NULL;
	_isoActorMC1		= NULL;
	_isoMC1				= NULL;
	_imageThresholdMC1	= NULL;

	//
	_treeExtraction		= NULL; 

}
//-------------------------------------------------------------------
wxEmptyPanelWidget_2::~wxEmptyPanelWidget_2(){
	// Erase Original Volule
	if (_mCubes			) { _mCubes			-> Delete(); }
	if (_surfMapper		) {	_surfMapper		-> Delete(); }
	if (_surfActor		) {	_surfActor		-> Delete(); }

	// Erase result: Volume + Axis
	ResetTree1_JF();
	ResetTree2_JF();
	ResetTree_MH_JFC();
}
//-------------------------------------------------------------------
wxPanel* wxEmptyPanelWidget_2::CreateViewPanel(wxWindow *parent)
{
 	wxPanel *panel		=	new wxPanel(parent,-1);
    wxBoxSizer *sizer	=	new wxBoxSizer(wxVERTICAL);
	_maracasSurfaceWidget = new wxSurfaceWidget(panel);
	sizer->Add(_maracasSurfaceWidget	, 1, wxEXPAND, 0);
	panel->SetSizer(sizer);
	panel->SetAutoLayout(true);
	panel->SetSize(400,400);
	panel->Layout();
	return panel;
}
//-------------------------------------------------------------------
wxPanel* wxEmptyPanelWidget_2::CreateControlPanel(wxWindow *parent)
{
	wxPanel *panel		= new wxPanel(parent,-1);
	_opacity_OrigVol	= new wxSlider( panel, -1, 30, 0, 100 , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_isoValue			= new wxSlider( panel, -1, 0 , 0, 3000, wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_opacity_Res1VolJF	= new wxSlider( panel, -1, 25, 0, 100 , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_opacity_Res2VolJF	= new wxSlider( panel, -1, 25, 0, 100 , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_sl_sensibility_JF	= new wxSlider( panel, -1, 0 , 0, 100 , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_branchLevel		= new wxSlider( panel, -1, 2 , 1, 10  , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_opacity_ResVolMHJF	= new wxSlider( panel, -1, 25, 0, 100 , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_isoValue_MH_JFC	= new wxSlider( panel, -1, 0 , 0, 100 , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );
	_cleanTree_MH_JFC	= new wxSlider( panel, -1, 400 , 0, 1000 , wxDefaultPosition, wxDefaultSize, wxSL_HORIZONTAL | wxSL_LABELS );

	int tmp_sensibility= (int)(_sensibility_JF*100.0);
	_sl_sensibility_JF->SetValue( tmp_sensibility );


	_opacity_OrigVol	->SetSize(250,20);   // Original Volume
	_isoValue			->SetSize(250,20);
	_opacity_Res1VolJF	->SetSize(250,20);  // Result 1 Volume JF
	_opacity_Res2VolJF	->SetSize(250,20);  // Result 2 Volume JF
	_branchLevel		->SetSize(250,20);
	_opacity_ResVolMHJF	->SetSize(250,20);  // Result Volume MH+JFC

	//Extract Tree 1 JF
	wxButton *btnExtractTree1_JF	= new wxButton(panel,-1,_T("Extract Tree 1 JF"));
	wxButton *btnEraseTree1_JF		= new wxButton(panel,-1,_T("Erase Tree 1 JF"));

	//Extract Tree 2 JF
	wxButton *btnExtractTree2_JF	= new wxButton(panel,-1,_T("Extract Tree 2 JF"));
	wxButton *btnEraseTree2_JF		= new wxButton(panel,-1,_T("Erase Tree 2 JF"));

	//Extract Tree MH + JF
	wxButton *btnExtractTree_MH_JFC = new wxButton(panel,-1,_T("Extract Tree MH + JFC"));
	wxButton *btnEraseTree_MH_JFC	= new wxButton(panel,-1,_T("Erase Tree MH + JFC"));


    wxFlexGridSizer *sizer = new wxFlexGridSizer(1);
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("Isovalue - Original Volume")));	
	sizer->Add(_isoValue);
	sizer->Add(new wxStaticText(panel,-1,_T("Opacity - Original Volume")));
	sizer->Add(_opacity_OrigVol);

	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  - - -  Juan Francisco Module 1 - - - ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(btnExtractTree1_JF);
	sizer->Add(btnEraseTree1_JF);
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("Opacity - Result Volume")));
	sizer->Add(_opacity_Res1VolJF);

	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  - - -  Juan Francisco Module 2 - - - ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T(" % Sensibility")));
	sizer->Add(_sl_sensibility_JF);
	sizer->Add(btnExtractTree2_JF);
	sizer->Add(btnEraseTree2_JF);
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("Opacity - Result Volume")));
	sizer->Add(_opacity_Res2VolJF);


	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  - - -  MH + JFC  - - - ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(btnExtractTree_MH_JFC);
	sizer->Add(btnEraseTree_MH_JFC);
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("Branch Level MH+JFC")));
	sizer->Add(_branchLevel);
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("Opacity - Result Volume MH+JFC")));
	sizer->Add(_opacity_ResVolMHJF);
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("IsoValue - Result Volume MH+JFC")));
	sizer->Add(_isoValue_MH_JFC);
	sizer->Add(new wxStaticText(panel,-1,_T("  ")));
	sizer->Add(new wxStaticText(panel,-1,_T("signal level (Clean Tree)")));
	sizer->Add(_cleanTree_MH_JFC);

	panel->SetSizer(sizer);
	panel->SetAutoLayout(true);
	panel->SetSize(400,400);
	panel->Layout();
	Connect(_opacity_OrigVol->GetId()		, wxEVT_COMMAND_SLIDER_UPDATED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnOpacity_OrigVol		); 
	Connect(_isoValue->GetId()				, wxEVT_SCROLL_THUMBRELEASE		, (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnIsoValue				); 
	Connect(_opacity_Res1VolJF->GetId()		, wxEVT_COMMAND_SLIDER_UPDATED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnOpacity_Res1VolJF	); 
	Connect(_opacity_Res2VolJF->GetId()		, wxEVT_COMMAND_SLIDER_UPDATED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnOpacity_Res2VolJF	); 
	Connect(_sl_sensibility_JF->GetId()		, wxEVT_COMMAND_SLIDER_UPDATED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnSensibility			); 

	Connect(btnExtractTree1_JF->GetId()		, wxEVT_COMMAND_BUTTON_CLICKED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnBtnExtractTree1_JF	); 
	Connect(btnEraseTree1_JF->GetId()		, wxEVT_COMMAND_BUTTON_CLICKED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnBtnEraseTree1_JF 	); 
	Connect(btnExtractTree2_JF->GetId()		, wxEVT_COMMAND_BUTTON_CLICKED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnBtnExtractTree2_JF	); 
	Connect(btnEraseTree2_JF->GetId()		, wxEVT_COMMAND_BUTTON_CLICKED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnBtnEraseTree2_JF 	); 

	Connect(btnExtractTree_MH_JFC->GetId()	, wxEVT_COMMAND_BUTTON_CLICKED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnBtnExtractTree_MH_JFC);	
	Connect(btnEraseTree_MH_JFC->GetId()	, wxEVT_COMMAND_BUTTON_CLICKED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnBtnEraseTree_MH_JFC 	);	
	Connect(_branchLevel->GetId()			, wxEVT_SCROLL_THUMBRELEASE		, (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnBranchLevel			); 
	Connect(_opacity_ResVolMHJF->GetId()	, wxEVT_COMMAND_SLIDER_UPDATED  , (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnOpacity_ResVolMHJF	); 
	Connect(_isoValue_MH_JFC->GetId()		, wxEVT_COMMAND_SLIDER_UPDATED	, (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnIsoValue_MH_JFC		); 
	Connect(_cleanTree_MH_JFC->GetId()		, wxEVT_COMMAND_SLIDER_UPDATED	, (wxObjectEventFunction) &wxEmptyPanelWidget_2::OnCleanTree_MH_JFC		); 

	return panel;
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::Refresh()
{
	_maracasSurfaceWidget->GetVtk3DSurfaceWidget()->Render();
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::ConfigureVTK()
{
    wxBusyCursor wait;
//	_imageviewer3D->Configure();
	vtkImageData	*imagedata		= _mar->_experiment->getDynData( )->getVolume( )->castVtk();
	double			opacity_OrigVol	= ((double)(_opacity_OrigVol->GetValue()))/100;
	double			isoValue; 

	// Visualisation - original volume
    double range[2];
    imagedata->GetScalarRange( range );
	isoValue=range[1] / 8;

    _mCubes = vtkMarchingCubes::New( );
    _mCubes->SetInput( imagedata );
    _mCubes->SetValue( 0, isoValue );

    vtkStripper *stripper = vtkStripper::New();
    stripper->SetInput( _mCubes->GetOutput( ) );

    _surfMapper = vtkPolyDataMapper::New( );
	_surfMapper->SetInput( stripper->GetOutput() );
    _surfMapper->ScalarVisibilityOff( );
    stripper->Delete();

    _surfActor = vtkActor::New( );
    _surfActor->SetMapper( _surfMapper );
    _surfActor->PickableOff( );
    _surfActor->GetProperty( )->SetColor( 0, 0, 0.8 );
    _surfActor->GetProperty( )->SetOpacity( opacity_OrigVol );

	// Update Controls
	_isoValue->SetRange( (int)(range[0]), (int)(range[1]) );
	_isoValue->SetValue( (int)isoValue );
	
	// Interface Update
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();
	ren->AddActor(_surfActor);

	//-Maracas-
	_maracasSurfaceWidget->ShowMARACASData( _mar );
}


//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::ResetTree1_JF()
{
	// Remove 
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();
    if (_1_isoActorMC6){
		ren->RemoveActor(_1_isoActorMC6);
	}
    if (_1_stripfinal){
		ren->RemoveActor(_1_stripfinal);
	}

	// Delete 
	if (_1_prgov		) {	_1_prgov		-> Delete(); }
	if (_1_mapfinal		) {	_1_mapfinal		-> Delete(); }
	if (_1_stripfinal	) {	_1_stripfinal	-> Delete(); }
	if (_1_isoMapperMC6	) {	_1_isoMapperMC6	-> Delete(); }
	if (_1_isoActorMC6	) {	_1_isoActorMC6	-> Delete(); }
	if (_1_isoMC6		) {	_1_isoMC6		-> Delete(); }

	// Init
	_1_prgov		= NULL;
	_1_mapfinal		= NULL;
	_1_stripfinal	= NULL;
	_1_isoMapperMC6	= NULL;
	_1_isoActorMC6	= NULL;
	_1_isoMC6		= NULL;
}

//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::ResetTree2_JF()
{
	// Remove 
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();
    if (_2_isoActorMC6){
		ren->RemoveActor(_2_isoActorMC6);
	}
    if (_2_stripfinal){
		ren->RemoveActor(_2_stripfinal);
	}

	// Delete 
	if (_2_prgov		) {	_2_prgov		-> Delete(); }
	if (_2_mapfinal		) {	_2_mapfinal		-> Delete(); }
	if (_2_stripfinal	) {	_2_stripfinal	-> Delete(); }
	if (_2_isoMapperMC6	) {	_2_isoMapperMC6	-> Delete(); }
	if (_2_isoActorMC6	) {	_2_isoActorMC6	-> Delete(); }
	if (_2_isoMC6		) {	_2_isoMC6		-> Delete(); }

	// Init
	_2_prgov		= NULL;
	_2_mapfinal		= NULL;
	_2_stripfinal	= NULL;
	_2_isoMapperMC6	= NULL;
	_2_isoActorMC6	= NULL;
	_2_isoMC6		= NULL;
}



//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::ExtractTree1_JF(int x, int y, int z)
{
	double opacity_Res1VolJF = ((double)_opacity_Res1VolJF->GetValue())/100;
	vtkImageData *imagedata = _mar->_experiment->getDynData( )->getVolume( )->castVtk();

	double	puntoactualprov[3];
	double	espprin[3];
	int		extprin[6];				

	puntoactualprov[0]=x;
	puntoactualprov[1]=y;
	puntoactualprov[2]=z;
	
	imagedata->GetSpacing(espprin);
	imagedata->GetExtent(extprin);

	puntoactualprov[0]=puntoactualprov[0]*espprin[0];
	puntoactualprov[1]=puntoactualprov[1]*espprin[1];
	puntoactualprov[2]=puntoactualprov[2]*espprin[2];
	
	// Juan Francisco Carrillo Filter 
	_1_prgov= axisExtractor::New();
	_1_prgov->SetHumbral(0.45);
	_1_prgov->SetInput(imagedata);
	_1_prgov->SetPoint(puntoactualprov);
	_1_prgov->Update();

	// Visualisation - result vascular tree
	_1_mapfinal = vtkPolyDataMapper::New();
	_1_mapfinal->SetInput(_1_prgov->GetOutput());

	_1_stripfinal = vtkActor::New();
	_1_stripfinal->SetMapper(_1_mapfinal);
	_1_stripfinal->GetProperty()->SetColor(1, 1, 1);
	_1_stripfinal->GetProperty()->SetLineWidth(2);
	_1_stripfinal->GetProperty()->BackfaceCullingOff();

	// Visualisation - result volume
	vtkImageData *imageOut = _1_prgov->GetVolumen();
	imageOut->Update();

    _1_isoMC6 = vtkMarchingCubes::New();
    _1_isoMC6->SetInput(imageOut);
    _1_isoMC6->SetValue(0, 1);
	_1_isoMC6->Update();

	_1_isoMapperMC6 = vtkPolyDataMapper::New();
    _1_isoMapperMC6->SetInput(_1_isoMC6->GetOutput());
    _1_isoMapperMC6->ScalarVisibilityOff();
    _1_isoMapperMC6->ImmediateModeRenderingOn();

	_1_isoActorMC6 = vtkActor::New();
    _1_isoActorMC6->SetMapper(_1_isoMapperMC6);
    _1_isoActorMC6->GetProperty()->SetColor( 0, 1, 0);
    _1_isoActorMC6->GetProperty()->SetOpacity( opacity_Res1VolJF );
 
   
	// Interface Update
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();
	ren->AddActor(_1_isoActorMC6);
	ren->AddActor(_1_stripfinal);
}


//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::ExtractTree2_JF(int x, int y, int z)
{
	double opacity_Res2VolJF = ((double)_opacity_Res2VolJF->GetValue())/100;
	vtkImageData *imagedata = _mar->_experiment->getDynData( )->getVolume( )->castVtk();

	double	puntoactualprov[3];
	double	espprin[3];
	int		extprin[6];				

	puntoactualprov[0]=x;
	puntoactualprov[1]=y;
	puntoactualprov[2]=z;
	
	imagedata->GetSpacing(espprin);
	imagedata->GetExtent(extprin);

	puntoactualprov[0]=puntoactualprov[0]*espprin[0];
	puntoactualprov[1]=puntoactualprov[1]*espprin[1];
	puntoactualprov[2]=puntoactualprov[2]*espprin[2];
	
	_2_prgov= axisExtractor02::New();
	_2_prgov->SetParam(1);
	_2_prgov->SetParam2(1);
	_2_prgov->SetParam3(_sensibility_JF);
	_2_prgov->SetMaxant(20);
	_2_prgov->SetMinant(5);
	_2_prgov->SetInput(imagedata);
	_2_prgov->SetPoint(puntoactualprov);
	_2_prgov->Update();

	// Visualisation - result vascular tree
	_2_mapfinal = vtkPolyDataMapper::New();
	_2_mapfinal->SetInput(_2_prgov->GetOutput());

	_2_stripfinal = vtkActor::New();
	_2_stripfinal->SetMapper(_2_mapfinal);
	_2_stripfinal->GetProperty()->SetColor(0.3, 0.3, 0.3);
	_2_stripfinal->GetProperty()->SetLineWidth(2);
	_2_stripfinal->GetProperty()->BackfaceCullingOff();

	// Visualisation - result volume
	vtkImageData *imageOut = _2_prgov->GetVolumen();
	imageOut->Update();

    _2_isoMC6 = vtkMarchingCubes::New();
    _2_isoMC6->SetInput(imageOut);
    _2_isoMC6->SetValue(0, 1);
	_2_isoMC6->Update();

	_2_isoMapperMC6 = vtkPolyDataMapper::New();
    _2_isoMapperMC6->SetInput(_2_isoMC6->GetOutput());
    _2_isoMapperMC6->ScalarVisibilityOff();
    _2_isoMapperMC6->ImmediateModeRenderingOn();

	_2_isoActorMC6 = vtkActor::New();
    _2_isoActorMC6->SetMapper(_2_isoMapperMC6);
    _2_isoActorMC6->GetProperty()->SetColor( 1, 1, 0);
    _2_isoActorMC6->GetProperty()->SetOpacity( opacity_Res2VolJF );
    
	// Interface Update
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();
	ren->AddActor(_2_isoActorMC6);
	ren->AddActor(_2_stripfinal);
}

//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnOpacity_OrigVol(wxScrollEvent& event)
{
	double value = ((double)_opacity_OrigVol->GetValue())/100;
    _surfActor->GetProperty( )->SetOpacity( value );
	Refresh();
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnOpacity_Res1VolJF(wxScrollEvent& event)
{
	if (_1_isoActorMC6!=NULL){
		double value = ((double)_opacity_Res1VolJF->GetValue())/100;
		_1_isoActorMC6->GetProperty( )->SetOpacity( value );
		Refresh();
	}
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnOpacity_Res2VolJF(wxScrollEvent& event)
{
	if (_2_isoActorMC6!=NULL){
		double value = ((double)_opacity_Res2VolJF->GetValue())/100;
		_2_isoActorMC6->GetProperty( )->SetOpacity( value );
		Refresh();
	}
}

//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnIsoValue(wxScrollEvent& event)
{
    wxBusyCursor wait;
	_mCubes->SetValue(0, _isoValue->GetValue());
	Refresh();
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnBtnExtractTree1_JF(wxCommandEvent& event)
{
	if (_1_prgov==NULL){
		wxBusyCursor wait;
		vtk3DSurfaceWidget *world3D = _maracasSurfaceWidget->GetVtk3DSurfaceWidget();
		if( world3D->GetInitialSphere() ){
			double point[3];
			world3D->GetSphereCenter( point );
			ExtractTree1_JF((int)point[0],(int)point[1],(int)point[2]);
		}
		Refresh();
	}
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnBtnExtractTree2_JF(wxCommandEvent& event)
{
	if (_2_prgov==NULL){
		wxBusyCursor wait;
		vtk3DSurfaceWidget *world3D = _maracasSurfaceWidget->GetVtk3DSurfaceWidget();
		if( world3D->GetInitialSphere() ){
			double point[3];
			world3D->GetSphereCenter( point );
			ExtractTree2_JF((int)point[0],(int)point[1],(int)point[2]);
		}
		Refresh();
	}
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnBtnEraseTree1_JF(wxCommandEvent& event)
{
	ResetTree1_JF();
	Refresh();
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnBtnEraseTree2_JF(wxCommandEvent& event)
{
	ResetTree2_JF();
	Refresh();
}

//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::PaintVascularTree_MH_JFC(marExperiment * newExperiment)
{
	vtkActor			*branchActor;
	vtkPolyData			*axisVtk;
	vtkPolyDataMapper	*branchMapper;
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();

	int i,maxAxis=newExperiment->getNumberOfAxes();
	for (i=0;i<maxAxis;i++){
		newExperiment->setAxis(i);
		marAxis *maraxis =newExperiment->getAxis( );
		if (maraxis!=NULL) {
			axisVtk = maraxis->Draw();
			branchMapper = vtkPolyDataMapper::New( );
			branchMapper->SetInput( axisVtk );
			branchMapper->Update();
			branchActor = vtkActor::New( );
			branchActor->SetMapper( branchMapper );
			branchActor->GetProperty()->SetColor( 1, 1, 1 );
			if (i== 0) { branchActor->GetProperty()->SetColor( 0  , 0	, 0.3 ); }
			if (i== 1) { branchActor->GetProperty()->SetColor( 0  , 0.3	, 0.3 ); }
			if (i== 2) { branchActor->GetProperty()->SetColor( 0.3, 0.3	, 0.3 ); }
			if (i== 3) { branchActor->GetProperty()->SetColor( 0.0, 0.0	, 0.6 ); }
			if (i== 4) { branchActor->GetProperty()->SetColor( 0.0, 0.6	, 0.6 ); }
			if (i== 5) { branchActor->GetProperty()->SetColor( 0.6, 0.6	, 0.6 ); }
			if (i== 7) { branchActor->GetProperty()->SetColor( 0.0, 0.0	, 0.9 ); }
			if (i== 8) { branchActor->GetProperty()->SetColor( 0.0, 0.9	, 0.9 ); }
			if (i== 9) { branchActor->GetProperty()->SetColor( 0.9, 0.9	, 0.9 ); }
			if (i==10) { branchActor->GetProperty()->SetColor( 0.9, 0.0	, 0.0 ); }
			if (i==11) { branchActor->GetProperty()->SetColor( 0.6, 0.0	, 0.0 ); }
			if (i==12) { branchActor->GetProperty()->SetColor( 0.3, 0.0	, 0.0 ); }
			if (i==13) { branchActor->GetProperty()->SetColor( 0.6, 0.6	, 0.0 ); }
			if (i==14) { branchActor->GetProperty()->SetColor( 0.3, 0.3	, 0.0 ); }
			if (i==15) { branchActor->GetProperty()->SetColor( 0.9, 0.9	, 0.0 ); }
			branchActor->GetProperty()->SetLineWidth( 2.0 );
			ren->AddActor( branchActor );

			_lstAxisVtk.push_back( axisVtk );
			_lstBranchActor.push_back( branchActor );
			_lstBranchMapper.push_back( branchMapper );
		}
	}
}


//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::PaintVascularVolume_MH_JFC(vtkImageData * image)
{

	// Visualisation - result volume
	image->Update();

	_imageThresholdMC1=vtkImageThreshold::New();
	_imageThresholdMC1->SetInput(image);
	_imageThresholdMC1->ThresholdBetween(0,100);
	_imageThresholdMC1->SetOutValue  (  0   ); 
	_imageThresholdMC1->Update();

    _isoMC1 = vtkMarchingCubes::New();
    _isoMC1->SetInput(_imageThresholdMC1->GetOutput()	);
    _isoMC1->SetValue(0, 1);
	_isoMC1->Update();

	_isoMapperMC1 = vtkPolyDataMapper::New();
    _isoMapperMC1->SetInput(_isoMC1->GetOutput());
    _isoMapperMC1->ScalarVisibilityOff();
    _isoMapperMC1->ImmediateModeRenderingOn();
	_isoMapperMC1->Update();

	_isoActorMC1 = vtkActor::New();
    _isoActorMC1->SetMapper(_isoMapperMC1);
    _isoActorMC1->GetProperty()->SetColor( 1, 0, 0);
    _isoActorMC1->GetProperty()->SetOpacity( 0.4 );

	// Interface Update
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();
	ren->AddActor(_isoActorMC1);
}

//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::WriteSignals(marExperiment *newExperiment){
	int  i       = 0;
	int  j       = 0;
	bool ok      = true;
	int  value   = 0;
	int  maxAxis = newExperiment->getNumberOfAxes();

	FILE *ff;
	ff=fopen("c:/Temp/MaracasTEMP.TXT", "w");

	while (ok==true){
		ok=false;
		for (i=0;i<maxAxis;i++){
			newExperiment->setAxis(i);
			marAxis *maraxis =newExperiment->getAxis( );
			if (maraxis!=NULL) {
				if ( j < maraxis->getNumberOfControlPoints()  ){
					ok=true;
					value= (int)(maraxis->getSignal((uint32_t)j) );
					fprintf(ff,"%d\t", value );
				} else {
					fprintf(ff,"\t");
				}
			}
		}
		fprintf(ff,"\n" );
		j++;
	}
	fclose(ff);
}


//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnBtnExtractTree_MH_JFC(wxCommandEvent& event)
{
  if ((_lstBranchActor.size()==0) 
	  && 
	  (_maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetInitialSphere() ) )
  {
   wxBusyCursor wait;

	// Copie Volume
	int voiA[6];
	int voiB[6];
	_mar->_experiment->getVOI(voiA);
	voiB[0]=0;
	voiB[1]=voiA[1]-voiA[0];
	voiB[2]=0;
	voiB[3]=voiA[3]-voiA[2];
	voiB[4]=0;
	voiB[5]=voiA[5]-voiA[4];
	marExperiment *newExperiment = new marExperiment( _mar->_experiment->getParameters() );
	newExperiment->setVOI( voiB );
	newExperiment->initExperiment( _mar->_experiment->getDynData( )->getVolume() );
	
	double	point[3];
	int		extent[ 6 ];
    _mar->_experiment->getDynData( )->getVolume( )->castVtk()->GetExtent( extent );
    _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetSphereCenter( point );
	int x = (int)point[0] - extent[0];
	int y = (int)point[1] - extent[2];
	int z = (int)point[2] - extent[4];
	newExperiment->setStartPoint( x, y, z);

	// Vascular tree extraction
	_treeExtraction = TreeExtraction_MH_JFC::New();
	_treeExtraction->SetInput(newExperiment);
	_treeExtraction->SetCleanTreeLevel( _cleanTree_MH_JFC->GetValue() );
	_treeExtraction->SetBranchLevel( _branchLevel->GetValue() );
	_treeExtraction->Update();

	//Paint Vascular Tree
	PaintVascularTree_MH_JFC(newExperiment);
	//Paint Vascular Volume
	PaintVascularVolume_MH_JFC( _treeExtraction->GetVolume() );
	Refresh( );

	WriteSignals(newExperiment);

	// Clening memory
	delete newExperiment;
  }
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnBtnEraseTree_MH_JFC(wxCommandEvent& event)
{
	ResetTree_MH_JFC();
	Refresh();
}

//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::ResetTree_MH_JFC()
{	
	// Remove
	vtkRenderer *ren = _maracasSurfaceWidget->GetVtk3DSurfaceWidget()->GetRenderer();
	int i,size;
	size=_lstAxisVtk.size();
	size=_lstBranchMapper.size();
	size=_lstBranchActor.size();
	for (i=0;i<size;i++){
		if (_lstBranchActor[i]!=NULL){
			ren->RemoveActor( _lstBranchActor[i] );
		}
	}
	for (i=0;i<size;i++){
	// Delete
		if (_lstAxisVtk[i]		!= NULL	) {	_lstAxisVtk[i]		-> Delete();	}
//		if (_lstBranchMapper[i]	!= NULL	) {	_lstBranchMapper[i] -> Delete();	}
		if (_lstBranchActor[i]	!= NULL	) {	_lstBranchActor[i]	-> Delete();	}
	// Init
		_lstAxisVtk[i]		= NULL;
		_lstBranchActor[i]	= NULL;
		_lstBranchMapper[i]	= NULL;
	}
	_lstAxisVtk.clear();
	_lstBranchActor.clear();
	_lstBranchMapper.clear();

	// Remove 
    if (_isoActorMC1){
		ren->RemoveActor(_isoActorMC1);
	}

	// Delete 
	if (_isoMapperMC1		) {	_isoMapperMC1		-> Delete(); }
	if (_isoActorMC1		) {	_isoActorMC1		-> Delete(); }
	if (_isoMC1				) {	_isoMC1				-> Delete(); }
	if (_imageThresholdMC1	) {	_imageThresholdMC1	-> Delete(); }


	// Init
	_isoMapperMC1		= NULL;
	_isoActorMC1		= NULL;
	_isoMC1				= NULL;
	_imageThresholdMC1	= NULL;

	//
//	if (_treeExtraction	) {	_treeExtraction	-> Delete(); }     // EED
//	_treeExtraction=NULL; 

}

//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnOpacity_ResVolMHJF(wxScrollEvent& event)
{
	if (_isoActorMC1!=NULL){
		double value = ((double)_opacity_ResVolMHJF->GetValue())/100;
		_isoActorMC1->GetProperty( )->SetOpacity( value );
	}
	Refresh();
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnBranchLevel(wxScrollEvent& event)
{
	//
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnIsoValue_MH_JFC(wxScrollEvent& event)
{
	if (_imageThresholdMC1!=NULL){
		wxBusyCursor wait;
		double threshold = _isoValue_MH_JFC->GetValue();
		if (threshold!=0){
			_imageThresholdMC1->ThresholdBetween(threshold, threshold );
		} else {
			_imageThresholdMC1->ThresholdBetween(0, 1000 );
		}
		Refresh();
	}
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnCleanTree_MH_JFC(wxScrollEvent& event)
{
//
}
//------------------------------------------------------------------------
void wxEmptyPanelWidget_2::OnSensibility(wxScrollEvent& event)
{
	_sensibility_JF = (double)(_sl_sensibility_JF->GetValue())/100;
}

