/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

//----------------------------------------------------------------------------
#include <wx/brush.h>
#include <wx/gdicmn.h>

#include "mBarRange.h"

//const wxEventType wxEVT_TSBAR = wxNewEventType();

DEFINE_EVENT_TYPE(wxEVT_TSBAR)
DEFINE_EVENT_TYPE(wxEVT_TSBAR_ACTUAL)
DEFINE_EVENT_TYPE(wxEVT_TSBAR_START)
DEFINE_EVENT_TYPE(wxEVT_TSBAR_END)
DEFINE_EVENT_TYPE(wxEVT_TSBAR_MOVED)
DEFINE_EVENT_TYPE(wxEVT_SELECTION_END)

//----------------------------------------------------------------------------
//EVENT TABLE
//----------------------------------------------------------------------------

IMPLEMENT_CLASS(mBarRange, wxScrolledWindow)

BEGIN_EVENT_TABLE(mBarRange, wxScrolledWindow)
	EVT_PAINT (mBarRange::OnPaint)
	EVT_SIZE  (mBarRange::OnSize)
	EVT_MOTION (mBarRange::OnMouseMove)
	EVT_RIGHT_DOWN (mBarRange :: onShowPopupMenu)
	EVT_MENU(cntID_CHANGE_COLOR, mBarRange :: onChangePartColor)
	EVT_MENU(cntID_ENABLE_ACTUAL, mBarRange :: onEnableRange_Actual)
	EVT_MENU(cntID_MOVABLE_ACTUAL_BAR, mBarRange :: onMovable_ActualWithBar)

	//
	EVT_LEFT_DOWN( mBarRange :: onLeftClicDown)
	EVT_LEFT_UP( mBarRange :: onLeftClickUp)

	EVT_CHAR( mBarRange :: onKey )

	//how to catch the new event (our event)
	//EVT_COMMAND  (ID_MY_WINDOW, wxEVT_MY_EVENT, MyFrame::OnMyEvent)
END_EVENT_TABLE()


//----------------------------------------------------------------------------
//CONSTRUCTOR
//----------------------------------------------------------------------------

//wxDefaultSize
mBarRange::mBarRange(wxWindow *parent, int w, int h)
//EED-2019-10-25 :wxScrolledWindow(parent,-1,wxDefaultPosition, wxSize(60,60) ,wxTAB_TRAVERSAL)
 :wxPanel(parent,-1,wxDefaultPosition, wxSize(60,60) ,wxTAB_TRAVERSAL)
{
	acceptedClick		= 	true;
	_bitmap_bar			= 	NULL;
	SetWidth (w);
	SetHeight(h);
	_initialPoint		= 	0;
	trianglesHalfWidth	= 	5;
	// Setting the default parts colors
	start_Colour		=	wxColour(1,0,255,254);
	actual_Colour		=	wxColour(255,255,202);
	end_Colour			=	wxColour(0,0,255);
	bar_Colour			=	wxColour(255,0,255);
	backgroundColor     =   parent ->GetBackgroundColour();
	guideLineColor		=   wxColour(255,0,0);
	//actual is in _start and end
	//false means that it could be anywhere
	_moveActualWithBar	=	false;
	_in_rangeProperty	=	false;
	_selectionMoveId	=	-1;
	realX_vertical_line =	-1;
	activeState			=	false;
	_actual				=	0;
	deviceEndMargin 	= 	0;
	SetOrientation(true);
	setIfWithActualDrawed(true);
	b_popmenu.Append (cntID_CHANGE_COLOR, _("Change Color"), _("Changes the color of the selected part"));
	b_popmenu.Append (cntID_ENABLE_ACTUAL, _("Enable actual in range"), _("Enables/Disables the actual triangle to be or not in range"));
	b_popmenu.Append (cntID_MOVABLE_ACTUAL_BAR, _("Move actual-bar simultaneously"), _("Disables the actual triangle to move with the bar"));
	SetSize(w,h);
}

//----------------------------------------------------------------------------
//DESTRUCTOR
//----------------------------------------------------------------------------

mBarRange::~mBarRange()
{
	DeletePendingEvents ();
	RemoveEventHandler ( GetEventHandler() ); 	
}

//---------------------------------------------------------------------------
//Draw bar: vertical or Horizontal
//---------------------------------------------------------------------------
void mBarRange::DrawBar()
{
	//Horizontal
	if(_orientation)
	{
		SetWindowStyle(wxNO_FULL_REPAINT_ON_RESIZE);
		_bitmap_bar			= new wxBitmap(_w+1280,_h+100);
		//SIL//_bitmap_info	= new wxBitmap(_w+100+1280, _h+100);
	} else {
		SetWindowStyle(wxNO_FULL_REPAINT_ON_RESIZE);
		_bitmap_bar 		= new wxBitmap(_h+deviceStart_y+100,_w+1280);
		_bitmap_info 		= new wxBitmap(_h+deviceStart_y+100, _w+1280);
	}
}
//----------------------------------------------------------------------------
//Getters & Setters
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//the property condition on actual triangle
//----------------------------------------------------------------------------
bool mBarRange::GetInRangeProperty()
{
	return _in_rangeProperty;
}
//----------------------------------------------------------------------------
void mBarRange::SetInRangeProperty(bool in)
{
	_in_rangeProperty = in;
}
//----------------------------------------------------------------------------
//the information about the actual triangle in range or not, true if is between start and end
//----------------------------------------------------------------------------
bool mBarRange::IsActualInRange()
{
	return ( _actual <= _end && _actual >= _start );
}

//----------------------------------------------------------------------------
// the position of the rectangle, vertical or horizontal
//----------------------------------------------------------------------------
bool mBarRange::GetOrientation()
{
	return _orientation;
}
//-----------------------------------------------------------------------------
void mBarRange::SetOrientation(bool orientation)
{
	if(_orientation)
	{
		SetSize(_h,_w);
	}
	_orientation=orientation;

}
//----------------------------------------------------------------------------
// _start of the pixel rectangle
//----------------------------------------------------------------------------

int mBarRange::GetPixelStart()
{
	return ((_start - _min)*(_w-deviceEndMargin))/(_max - _min);	
}

//----------------------------------------------------------------------------
// param i: value in pixels
//----------------------------------------------------------------------------
void mBarRange::SetPixelStart(int i)
{
	_start = _min+((i - deviceStart_x)*( _max - _min))/(_w-deviceEndMargin);
	
}

//----------------------------------------------------------------------------
// _actual of the pixel rectangle
//----------------------------------------------------------------------------
int mBarRange::GetPixelActual()
{
	return ((_actual - _min)*(_w-deviceEndMargin))/(_max - _min);
}

//----------------------------------------------------------------------------
// param i: value in pixels
//----------------------------------------------------------------------------
void mBarRange::SetPixelActual(int i)
{
	_actual = _min + (i-deviceStart_x)*(_max-_min)/(_w-deviceEndMargin);
}

//----------------------------------------------------------------------------
// _end of the pixel rectangle
//----------------------------------------------------------------------------
int mBarRange::GetPixelEnd()
{
	return ((_end - _min)*(_w-deviceEndMargin))/(_max - _min);
}

//----------------------------------------------------------------------------
// param i: value in pixels to be converted to real logical value
//----------------------------------------------------------------------------
void mBarRange::SetPixelEnd(int i)
{
	_end = _min + (i-deviceStart_x)*(_max-_min)/(_w-deviceEndMargin);
}

//----------------------------------------------------------------------------
// Logical max of the triangle
//----------------------------------------------------------------------------
double mBarRange::GetMax()
{
	return _max;
}

//----------------------------------------------------------------------------
void mBarRange::SetMax(double i)
{
	_max=i;
}
//----------------------------------------------------------------------------
// Logical min of the triangle
//----------------------------------------------------------------------------

double mBarRange::GetMin()
{
	return _min;
}

//----------------------------------------------------------------------------
void mBarRange::SetMin(double i)
{
	_min=i;
}

//----------------------------------------------------------------------------
// pixel dimensions of the rectangle
//----------------------------------------------------------------------------

int mBarRange::GetWidth()
{
	return _w;
}
//----------------------------------------------------------------------------
void mBarRange::SetWidth(int w)
{
	_w=w;
}
//----------------------------------------------------------------------------
int mBarRange::GetHeight()
{
	return _h;
}

//----------------------------------------------------------------------------
void mBarRange::SetHeight(int h)
{
	_h=h;	
}

//----------------------------------------------------------------------------
// Logical  Start of the rectangle
//----------------------------------------------------------------------------

int mBarRange::filtreValue(int value)
{
	if(value<_min)
	{
		value = _min;
	} else if (value>_max) {
		value = _max;
	}
	return value;
}

//----------------------------------------------------------------------------
int mBarRange::GetStart()
{
	return _start;
}
//----------------------------------------------------------------------------
// param start: value real units
//----------------------------------------------------------------------------
void mBarRange::SetStart(int newstart)
{
	_start = filtreValue(newstart);

	if (_start>_end) 
	{ 
		_start=_end; 
	}

	if (_in_rangeProperty==true)
	{
		if (_start>_actual)   { _start=_actual; }
	}

	RefreshForce();	
}
//----------------------------------------------------------------------------
// Logical End of the rectangle
//----------------------------------------------------------------------------

int mBarRange::GetEnd()
{
	return _end;
}
//----------------------------------------------------------------------------
// param end: value pixel units
//----------------------------------------------------------------------------
void mBarRange::SetEnd(int newend)
{
	_end = filtreValue(newend);
	if (_end<_start) { _end=_start; }
	if (_in_rangeProperty==true)
	{
		if (_end<_actual)   
		{
		 _end=_actual; 
		} // _end
	}
	RefreshForce();	
}
//----------------------------------------------------------------------------
// logical  Actual of the rectangle
//----------------------------------------------------------------------------
int mBarRange::GetActual()
{
	return _actual;
}
//----------------------------------------------------------------------------
void mBarRange::SetActual(int newactual)
{
	_actual = filtreValue(newactual);
	if (_in_rangeProperty==true)
	{
		if (_actual<_start) { _actual=_start; }		
		if (_actual>_end)   { _actual=_end; }
	}
	RefreshForce();
}

//----------------------------------------------------------------------------
//
//----------------------------------------------------------------------------
int mBarRange::GetTrianglesHalfWidth()
{
	return trianglesHalfWidth;
}

//----------------------------------------------------------------------------
void mBarRange::SetTrianglesHalfWidth(int nwTriHalfWidth)
{
	trianglesHalfWidth = nwTriHalfWidth;
}

void mBarRange::OnSize( wxSizeEvent &WXUNUSED(event) )
{
	wxRect rectTotal = GetClientRect(); 
	if(_orientation)
	{		
		SetWidth( rectTotal.GetWidth() - deviceEndMargin );			
	} 	else 	{
		SetWidth( rectTotal.GetHeight() - deviceEndMargin);				
	}
	_selectionMoveId = -1;
	Refresh();		
}

//----------------------------------------------------------------------------

void mBarRange::Refresh(bool eraseBackground, const wxRect* rect)
{
//EED2019-10-25	wxScrolledWindow::Refresh(false);
	wxPanel::Refresh(false);
}


//----------------------------------------------------------------------------
//Bar Methods
//----------------------------------------------------------------------------
void mBarRange::OnPaint( wxPaintEvent &WXUNUSED(event) )
{
	if (_bitmap_bar!=NULL)
	{
		//repaint rectangle
		if(_orientation)
		{
			RefreshHorizontalView();
			wxMemoryDC temp_dc;
			temp_dc.SelectObject( *_bitmap_bar );
			wxPaintDC dc( this );
			dc.Blit(deviceStart_x-(trianglesHalfWidth+2), deviceStart_y, _w-deviceEndMargin+2*(trianglesHalfWidth+2), _h, &temp_dc, 0, 0);
			//repaint info
//			if (_visibleLables)
//			{
//				temp_dc.SelectObject( *_bitmap_info );				
//				dc.Blit(deviceStart_x,deviceStart_y+_h, _w+deviceStart_x-deviceEndMargin, _h+deviceStart_y+200, &temp_dc, deviceStart_x, deviceStart_y);
//				//dc.Blit(deviceStart_x,deviceStart_y+_h, _w+deviceStart_x-deviceEndMargin, _h+deviceStart_y+60, &temp_dc, 0, 0);
//			}
	
		} else {
			RefreshVerticalView();
			wxMemoryDC temp_dc;
			temp_dc.SelectObject( *_bitmap_bar );
			wxPaintDC dc( this );			
//			dc.Blit(deviceStart_y,deviceStart_x, _h+deviceStart_y-deviceEndMargin,_w+deviceStart_x-deviceEndMargin, &temp_dc, 0, 0);	
			dc.Blit(deviceStart_y,deviceStart_x-(trianglesHalfWidth+2), _h,_w-deviceEndMargin+2*(trianglesHalfWidth+2), &temp_dc, 0, 0);	
			
			//repaint info
//			if (_visibleLables)
//			{
//				temp_dc.SelectObject( *_bitmap_info );
//				dc.Blit(0,_w, _h+deviceStart_y+200, _w+deviceStart_x+200-deviceEndMargin, &temp_dc, deviceStart_y,_w+deviceStart_x);
//			}

		} 
	} // _bitmap_bar
}


//----------------------------------------------------------------------------
//Repaint the bar if it is horizontal
//----------------------------------------------------------------------------
void mBarRange::RefreshHorizontalView()
{
	wxPoint points[3];

	//int largestNumberWidthInPixels = 15; // JPRx
	int pxStart		= GetPixelStart();
	int pxEnd		= GetPixelEnd();
	int pxActual	= GetPixelActual();

	int letterHeight= 9;
	int barHeight	= 2*letterHeight;
	int tempHeight	= _h-(6*letterHeight);	
	
	if (_visibleLables)
	{
		barHeight	= (tempHeight>0)  ? tempHeight : (int) _h/2;
	}
	else
		barHeight	= _h;	

	wxMemoryDC temp_dc;
	temp_dc.SelectObject( *_bitmap_bar );

	// Background of this widget
	temp_dc.SetPen(wxPen( backgroundColor ));
	temp_dc.SetBrush(wxBrush( backgroundColor ,wxSOLID  ));
	temp_dc.DrawRectangle(0,0,_w+2*trianglesHalfWidth,_h);
	temp_dc.SetPen(wxPen( wxColour(167,165,191) ,1,wxSOLID  ));
	temp_dc.DrawLine(trianglesHalfWidth+2, 0, _w-deviceEndMargin, 0);
	temp_dc.DrawLine(trianglesHalfWidth+2, barHeight, (_w-deviceEndMargin-trianglesHalfWidth-2), barHeight);
	temp_dc.SetDeviceOrigin(trianglesHalfWidth+2,0);

	// Filling the bar
	temp_dc.SetBrush(wxBrush( wxColour(104,104,104),wxSOLID  ));
	temp_dc.SetPen(wxPen( wxColour(104,104,104),1,wxSOLID  ));
	temp_dc.DrawRectangle( pxStart , 0, pxEnd-pxStart, barHeight);

	//  The Bar
	if( _selectionMoveId==4 )
	{
		temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
		temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
	} else {
		temp_dc.SetBrush(wxBrush( bar_Colour,wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(164,0,164),1,wxSOLID  ));
	}
	temp_dc.DrawRectangle( pxStart,1, pxEnd-pxStart, barHeight );

	// 2 Shadow Triangles: Start and End 
	temp_dc.SetBrush(wxBrush( wxColour(104,104,104),wxSOLID  ));
	temp_dc.SetPen(wxPen( wxColour(104,104,104),1,wxSOLID  ));
	points[0].x= 0;
	points[0].y= barHeight;
	points[1].x= -trianglesHalfWidth-1;
	points[1].y= 0;
	points[2].x= trianglesHalfWidth+2;
	points[2].y= 0;
	temp_dc.DrawPolygon(3,points,pxStart,0);
	temp_dc.DrawPolygon(3,points,pxEnd,0);

	// 2 Triangles: Start and End 
	points[1].x = -trianglesHalfWidth;	
	points[2].x = trianglesHalfWidth;
	
	//first triangle (start)
	if( _selectionMoveId == 1 )
	{
		temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
		temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
	} else {
		temp_dc.SetBrush(wxBrush( start_Colour,wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(0,51,204),1,wxSOLID  ));
	}
	temp_dc.DrawPolygon(3,points,pxStart,0);
	//second triangle (end)
	if( _selectionMoveId == 2 )
	{
		temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
		temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
	} else {
		temp_dc.SetBrush(wxBrush( end_Colour,wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(0,0,255),1,wxSOLID  ));
	}
	temp_dc.DrawPolygon(3,points,pxEnd,0);

	if( withActualDrawed )
	{
		// 1 Shadow Triangle: Actual
		temp_dc.SetBrush(wxBrush( wxColour(104,104,104),wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(104,104,104),1,wxSOLID  ));
		points[1].x = -trianglesHalfWidth-1;
		points[2].x = trianglesHalfWidth+2;
		
		temp_dc.DrawPolygon(3,points,pxActual,0);

		// 1 Triangle: Actual (red)
		if( _selectionMoveId==3 )
		{
			temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
			temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
		} else {
			temp_dc.SetBrush(wxBrush( actual_Colour,wxSOLID  ));
			temp_dc.SetPen(wxPen( wxColour(205,160,5),1,wxSOLID  ));
		}
		points[1].x = -trianglesHalfWidth;
		points[2].x = trianglesHalfWidth;
		temp_dc.DrawPolygon(3,points,pxActual,0);
	}

	if (realX_vertical_line!=-1)
	{
		temp_dc.SetPen(wxPen(  guideLineColor,1,wxDOT ));
		int pixelX_guide = ((realX_vertical_line - _min)*(_w-deviceEndMargin))/(_max - _min) ; 
		temp_dc.DrawLine(pixelX_guide, 0, pixelX_guide, barHeight);
	}

	//Information Device drawing

	if (_visibleLables)
	{
		//temp_dc.SelectObject( *_bitmap_info );
		/*temp_dc.SetBrush(wxBrush( colourParent ,wxSOLID  ));
		temp_dc.SetPen(wxPen( colourParent ,1,wxSOLID  ));*/
		//temp_dc.DrawRectangle(deviceStart_x,_h+deviceStart_y,_w+deviceStart_x+40,_h+deviceStart_y+40);
		//temp_dc.DrawRectangle(0,_h,_w+40-deviceEndMargin,_h+40);

		wxFont font(letterHeight-1, wxFONTFAMILY_SWISS, wxNORMAL, wxNORMAL);
		temp_dc.SetFont(font);
		temp_dc.SetTextForeground(*wxBLACK);

		//the **MIN** value, always at the same y level that corresponds to barHeight+1
		wxString text_min;
//		text_min<< GetMin();
		text_min.Printf(_T("%d"), (int)GetMin() );
		
		temp_dc.DrawText(text_min,0,barHeight+1);

		//the **MAX** value always at the same place
		wxString text_max;
//		text_max << GetMax();
		text_max.Printf(_T("%d"), (int)GetMax() );

		//As there is a margin of 40 extra most numbers (max) should be visibles
//		stringSize = temp_dc.GetTextExtent(text_max);
        wxCoord tmpX,tmpY;
		temp_dc.GetTextExtent(text_max,&tmpX,&tmpY);
		wxSize stringSize(tmpX,tmpY);
		
		temp_dc.DrawText(text_max,_w-deviceEndMargin -(stringSize.GetWidth())/*2*trianglesHalfWidth*/,barHeight+1);	
		
		//show logical values
		//show the **START TRIANGLE** value 
		wxString text_start;
//		text_start << GetStart();		
		text_start.Printf(_T("%d"), (int)GetStart() );

		temp_dc.DrawText(text_start, pxStart,barHeight+2*letterHeight);
		//show the **END TRIANGLE** value
		wxString text_end;
//		text_end << GetEnd();
		text_end.Printf(_T("%d"), (int)GetEnd() );

//		stringSize = temp_dc.GetTextExtent(text_end);
		temp_dc.GetTextExtent(text_end,&tmpX,&tmpY);
		stringSize.SetHeight(tmpY);
		stringSize.SetWidth(tmpX);
		temp_dc.DrawText(text_end, pxEnd-stringSize.GetWidth(),barHeight+3*letterHeight);
		if( withActualDrawed )
		{
			//show the actual value of actual
			wxString text_actual;
//			text_actual << GetActual();
			text_actual.Printf(_T("%d"), (int)GetActual() );
//			stringSize = temp_dc.GetTextExtent(text_actual);
			temp_dc.GetTextExtent(text_actual,&tmpX,&tmpY);
  		    stringSize.SetHeight(tmpY);
		    stringSize.SetWidth(tmpX);
			temp_dc.DrawText(text_actual, pxActual-(stringSize.GetWidth()/2),barHeight+letterHeight);			
		}			
	}
}

//----------------------------------------------------------------------------
//Repaint the bar if it is vertical
//----------------------------------------------------------------------------

void mBarRange::RefreshVerticalView()
{
	wxPoint points[3];

	int px1=GetPixelStart();
	int px2=GetPixelEnd();
	int px3=GetPixelActual();
	int letterHeight = 9;
	int panelHeight = 9*3+_w;

	int barWidth;
	if (_visibleLables)
	{
		barWidth = (_w-30)>0 ? _w-30 : (int) _w/2;
	}
	else
		barWidth = _w;	

	wxMemoryDC temp_dc;
	temp_dc.SelectObject( *_bitmap_bar );

	// Background
	temp_dc.SetBrush(wxBrush( backgroundColor ,wxSOLID  ));
	temp_dc.SetPen(wxPen( backgroundColor ));

	temp_dc.DrawRectangle(0,0,_h,_w+2*trianglesHalfWidth);
	

	temp_dc.SetPen(wxPen( wxColour(167,165,191) ,1,wxSOLID  ));
	temp_dc.DrawLine(0,trianglesHalfWidth+2, 0, _w-deviceEndMargin);
	temp_dc.DrawLine(barWidth, trianglesHalfWidth+2, barWidth, (_w-deviceEndMargin-trianglesHalfWidth-2));
	temp_dc.SetDeviceOrigin(0,trianglesHalfWidth+2);

	// Filling the bar
	temp_dc.SetBrush(wxBrush( wxColour(104,104,104),wxSOLID  ));
	temp_dc.SetPen(wxPen( wxColour(104,104,104),1,wxSOLID  ));
	temp_dc.DrawRectangle( 0,px1 ,_h, px2-px1 );


	//  The Bar
		if( _selectionMoveId==4 )
	{
		temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
		temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
	}
	else
	{
		temp_dc.SetBrush(wxBrush( bar_Colour,wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(164,0,164),1,wxSOLID  ));
	}
	temp_dc.DrawRectangle( 1,px1,_h, px2-px1);


	// 2 Shadow Triangles: Start and End 
	points[0].x	= _h;
	points[0].y	= 0;
	points[1].x	= 0;
	points[1].y	= -trianglesHalfWidth-1;
	points[2].x	= 0;
	points[2].y	= trianglesHalfWidth+2;
	temp_dc.SetBrush(wxBrush( wxColour(104,104,104),wxSOLID  ));
	temp_dc.SetPen(wxPen( wxColour(104,104,104),1,wxSOLID  ));
	temp_dc.DrawPolygon(3,points,0,px1);
	temp_dc.DrawPolygon(3,points,0,px2);

	// 2 Triangles: Start and End 
	points[0].x	= _h;
	points[0].y	= 0;
	points[1].x	= 0;
	points[1].y	= -trianglesHalfWidth;
	points[2].x	= 0;
	points[2].y	= trianglesHalfWidth;
	//first triangle (start)
	if( _selectionMoveId==1 )
	{
		temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
		temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
	}
	else
	{
		temp_dc.SetBrush(wxBrush( start_Colour,wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(0,51,204),1,wxSOLID  ));
	}
	temp_dc.DrawPolygon(3,points,0,px1);
	//second triangle (end)
	if( _selectionMoveId==2 )
	{
		temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
		temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
	}
	else
	{
		temp_dc.SetBrush(wxBrush( end_Colour,wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(0,0,255),1,wxSOLID  ));
	}
	temp_dc.DrawPolygon(3,points,0,px2);

	if( withActualDrawed )
	{
		// 1 Shadow Triangle: Actual
		temp_dc.SetBrush(wxBrush( wxColour(104,104,104),wxSOLID  ));
		temp_dc.SetPen(wxPen( wxColour(104,104,104),1,wxSOLID  ));
		points[0].x=_h;
		points[0].y=0;
		points[1].x=0;
		points[1].y=-trianglesHalfWidth-1;
		points[2].x=0;
		points[2].y=trianglesHalfWidth+2;
		temp_dc.DrawPolygon(3,points,0,px3);

		// 1 Triangle: Actual (red)
		points[0].x = _h;
		points[0].y = 0;
		points[1].x = 0;
		points[1].y = -trianglesHalfWidth;
		points[2].x = 0;
		points[2].y = trianglesHalfWidth;
		if( _selectionMoveId==3 )
		{
			temp_dc.SetBrush(wxBrush( wxColour(128,255,0),wxSOLID  ));
			temp_dc.SetPen(wxPen(  wxColour(0,128,0),1,wxSOLID  ));
		}
		else
		{
			temp_dc.SetBrush(wxBrush( actual_Colour,wxSOLID  ));
			temp_dc.SetPen(wxPen( wxColour(205,160,5),1,wxSOLID  ));
		}
		temp_dc.DrawPolygon(3,points,0,px3);
	}

	if (realX_vertical_line!=-1)
	{
		temp_dc.SetPen(wxPen(  guideLineColor,1,wxDOT  ));
		int pixelX_guide = realX_vertical_line*_w/(_max-_min)+deviceStart_x; 
		temp_dc.DrawLine(0,pixelX_guide, _h, pixelX_guide);
	}

	//Information Device drawing
	if (_visibleLables)
	{
		/*temp_dc.SelectObject( *_bitmap_info );

		temp_dc.SetBrush(wxBrush( backgroundColor ,wxSOLID  ));
		temp_dc.SetPen(wxPen( backgroundColor ,1,wxSOLID  ));
		temp_dc.DrawRectangle(deviceStart_y,_w+deviceStart_x,_h+deviceStart_y+200,_w+deviceStart_x+200);
*/

		temp_dc.SetBackgroundMode(wxTRANSPARENT);
		wxFont font(letterHeight-1, wxFONTFAMILY_SWISS, wxNORMAL, wxNORMAL);
		temp_dc.SetFont(font);
		temp_dc.SetTextForeground(*wxBLACK);

		//show logical values
		//show the actual value of start
		wxString text_start;
//		text_start<<"Start:"<< GetStart();
		text_start.Printf(_T("%s %d"),_T("Start: "), (int)GetStart() );
		temp_dc.DrawText( text_start ,deviceStart_y, _w+deviceStart_x+letterHeight+1);
		//show the actual value of end
		wxString text_end;
//		text_end <<"End: "<<GetEnd();
		text_end.Printf(_T("%s %d"),_T("End: "), (int)GetEnd() );
		temp_dc.DrawText( text_end ,deviceStart_y,_w+deviceStart_x+letterHeight*2 );
		if( withActualDrawed )
		{
			//show the actual value of actual
			wxString text_actual;
//			text_actual <<"Actual: " <<GetActual();
			text_actual.Printf(_T("%s %d"),_T("Actual: "), (int)GetActual() );
			temp_dc.DrawText( text_actual ,deviceStart_y,_w+deviceStart_x+letterHeight*3);
		}
		//the min value, always at the same place
		wxString text_min;
//		text_min<<"Min: " << GetMin();
		text_min.Printf(_T("%s %d"),_T("Min: "), (int)GetMin() );
		temp_dc.DrawText( text_min ,deviceStart_y,_w+deviceStart_x+3);
		//the max value always at the samen place
		wxString text_max;
//		text_max <<"Max: "<< GetMax();
		text_max.Printf(_T("%s %d"),_T("Max: "), (int)GetMax() );
		//toca calcular cuantol lo corremos
		temp_dc.DrawText(text_max,deviceStart_y,_w+deviceStart_x+43);		
	}

}

//----------------------------------------------------------------------------
void mBarRange::RefreshForce()
{
	Refresh();
	Update();
}
//----------------------------------------------------------------------------
void mBarRange::OnMouseMove(wxMouseEvent& event )
{
	if (activeState)
	{
		wxPoint point = event.GetPosition();
		int barHeight;
		if (_orientation)
		{
			setClickedX(point.x);
			barHeight = point.y;
		}
		else
		{
			setClickedX(point.y);
			barHeight = point.x;
		}
		int logicClick = getLogicValueofPixel(clickedX);
			
		if( _selectionMoveId==-1 )
		{
			if (barHeight <=_h)
			{
				bool in_StartTri = (clickedX>=GetPixelStart()-5+ deviceStart_x) && (clickedX<=GetPixelStart()+5+ deviceStart_x);
				bool in_EndTri 	 = (clickedX>=GetPixelEnd()-5+ deviceStart_x) && (clickedX<=GetPixelEnd()+5+ deviceStart_x);
				bool in_actualT	 = withActualDrawed && (clickedX>=GetPixelActual()-5+ deviceStart_x) && (clickedX<=GetPixelActual()+5+ deviceStart_x);
				bool in_movingBar= (clickedX>GetPixelStart()+5+ deviceStart_x) && (clickedX<GetPixelEnd()-5+ deviceStart_x);

				if( in_actualT )
					_selectionMoveId = 3;
				else if(in_StartTri)
					_selectionMoveId = 1;  
				else if( in_EndTri )
					_selectionMoveId = 2;
				else if( in_movingBar )
					_selectionMoveId = 4;
			}
		} else {
			if(acceptedClick)
			{
				//is in start triagle
				if( _selectionMoveId ==1 && event.LeftIsDown())
				{
					bool validPos_StartTri = (logicClick<GetEnd() && logicClick >=_min);
					if( validPos_StartTri && !_in_rangeProperty)
					{	
						SetPixelStart(clickedX);
						RefreshForce();
						RefreshHorizontalView();
						//-------------------------------------------
						// Sending the event of start triangle moved
						//-------------------------------------------
						createAndSendEvent( wxEVT_TSBAR_START );
					}
					//start has to be less than actual
					else if (validPos_StartTri && _in_rangeProperty)
					{
						if(logicClick<=GetActual())
						{
							SetPixelStart(clickedX);
							RefreshForce();
						//	RefreshHorizontalView();
							//-------------------------------------------
							// Sending the event of start triangle moved
							//-------------------------------------------
						createAndSendEvent( wxEVT_TSBAR_START );
						}
					}
				} // _selectionMoveId == 1
				//is in end triangle
				else if( _selectionMoveId == 2 && event.LeftIsDown() )
				{
					bool validPos_EndTri = logicClick>GetStart()&& logicClick<=_max;  
					if( validPos_EndTri && !_in_rangeProperty )
					{					
						SetPixelEnd(clickedX);
						RefreshForce();
	//					RefreshHorizontalView();	
						//-------------------------------------------
						//Sending the event of end triangle moved
						//-------------------------------------------
						createAndSendEvent( wxEVT_TSBAR_END );
					}
					//the end triangle cant be less than actual
					else if( validPos_EndTri && _in_rangeProperty )
					{
						if(logicClick>=GetActual())
						{
							SetPixelEnd(clickedX);
							RefreshForce();
						//	RefreshHorizontalView();
							//-------------------------------------------
							//Sending the event of end triangle moved
							//-------------------------------------------
							createAndSendEvent( wxEVT_TSBAR_END );
						}
					}
				} 
				//is the actual triangle
				else if( _selectionMoveId == 3 && event.LeftIsDown())
				{
					bool validPos_ActualTri=(logicClick<=_max) && (logicClick>=_min);
					//is in actual triangle but it could be anywhere
					if( validPos_ActualTri && !_in_rangeProperty )
					{
						SetPixelActual(clickedX);
						RefreshForce();
						RefreshHorizontalView();
						//-------------------------------------------
						//Sending the event of actual triangle moved
						//-------------------------------------------
						createAndSendEvent( wxEVT_TSBAR_ACTUAL );	
//						createAndSendEvent( 98765 );
					}
					else if( validPos_ActualTri && _in_rangeProperty )
					// the tringle in between start and end
					{
						if( logicClick>=GetStart() && logicClick<=GetEnd())
						{
							SetPixelActual(clickedX);
							RefreshForce();
							RefreshHorizontalView();
							//-------------------------------------------
							//Sending the event of actual triangle moved
							//-------------------------------------------
							createAndSendEvent( wxEVT_TSBAR_ACTUAL );
						}
					} 
				} 
				//is the bar
				else if ( _selectionMoveId == 4 &&  event.LeftIsDown() )
				{	
					//FILE * f=fopen("E:/borrar/file.txt","a+");
					if(_initialPoint == 0)
					{
						_initialPoint = logicClick;
						logicInitial_start = GetStart(); 
						logicInitial_end = GetEnd();
						logicInitial_actual = GetActual();
						//SIL//fprintf(f,"\n\n---- Inicia draggin:\n  logicInitial_start:%d, logicInitial_end:%d,logicInitial_actual:%d \n", _initialPoint,logicInitial_start,logicInitial_end,logicInitial_actual);
					}
					int difference = logicClick -_initialPoint;
					int next_end = difference + logicInitial_end;
					int next_start = difference + logicInitial_start;
					int next_actual = difference + logicInitial_actual;
					
					/*SIL//fprintf(f,"diff:%d, next_end%d, next_start%d, next_actual%d \n", difference,next_end,next_start,next_actual);
					fclose(f);*/
					
					//if actual is not fixed to be in the middle
					if( ((logicClick>next_start) && (logicClick<next_end)&& (next_end<=_max)&& (next_start>=_min)) && !_in_rangeProperty)
					{
						SetStart(next_start);
						SetEnd(next_end);
						if( _moveActualWithBar )
						{
							SetActual (next_actual);
							//-------------------------------------------
							//Sending the event of actual triangle moved
							//-------------------------------------------
							createAndSendEvent( wxEVT_TSBAR_ACTUAL );
						}
						RefreshForce();
						RefreshHorizontalView();	
											
						//-------------------------------------------
						// Sending the event that the bar ahs being moved
						//-------------------------------------------
						createAndSendEvent( wxEVT_TSBAR_MOVED );
						//EED
						createAndSendEvent( wxEVT_TSBAR_ACTUAL );
						createAndSendEvent( wxEVT_TSBAR_END );
						createAndSendEvent( wxEVT_TSBAR_START );
					}
					//if actual has to be between start and end
					else if(_in_rangeProperty && ((next_start<=GetActual()) && (next_end>=GetActual()) && (next_end<=_max)&& (next_start>=_min)) )
					{
						SetStart(next_start);
						SetEnd(next_end);
						if( _moveActualWithBar )
						{
							SetActual (next_actual);
							//-------------------------------------------
							//Sending the event of actual triangle moved
							//-------------------------------------------
							createAndSendEvent( wxEVT_TSBAR_ACTUAL );
						}
						RefreshForce();
						RefreshHorizontalView();	
						
						//-------------------------------------------
						// Sending the event that the bar ahs being moved
						//-------------------------------------------
						createAndSendEvent( wxEVT_TSBAR_MOVED );
						//EED
						createAndSendEvent( wxEVT_TSBAR_ACTUAL );
						createAndSendEvent( wxEVT_TSBAR_END );
						createAndSendEvent( wxEVT_TSBAR_START );
					}
				}			
			}
			if( !event.LeftIsDown())
			{
				_initialPoint=0;
				_selectionMoveId = -1;
				RefreshForce();
				//-------------------------------------------
				//Sending a general event just because
				//-------------------------------------------
				//SIL//createAndSendEvent( wxEVT_TSBAR );
				createAndSendEvent(wxEVT_SELECTION_END);
			}
		}				
	}	
}


/*
* Sets the represented minimum and maximunm values
* param minRealValue The minimum represented value (real value)
* param maxRealValue The maximum represented value (real value)
*/
void mBarRange :: setRepresentedValues ( double minRealValue, double maxRealValue)
{
	_min 	= minRealValue;
	_max 	= maxRealValue;
	_start	=_min;
	_end	=_max;
}

/*
* Sets the property for viewing or not the bar labels information
*/
void mBarRange :: setVisibleLabels ( bool setVisibleLB )
{
	_visibleLables = setVisibleLB;
}

/*
	* Sets the property for viewing or not the bar labels information
	* return _visibleLables The state of visible labels or not 
	*/
	bool mBarRange ::getIfVisibleLabels ()
	{
		return _visibleLables;
	}

	/**
	* Sets the device start drawing left-superior (pixel) start point 
	* param deviceStart_x Pixel start for x-coord
	* param deviceStart_y Pixel start for y-coord
	*/
	void mBarRange :: setDeviceBlitStart ( wxCoord devStart_x, wxCoord devStart_y )
	{
		deviceStart_x = devStart_x;
		deviceStart_y = devStart_y;
		// For the initialization case
		if (GetPixelEnd()<0)
		{
			if (_orientation)
			{
				SetPixelStart(deviceStart_x);
				SetPixelEnd(_w+deviceStart_x);
				SetPixelActual(deviceStart_x);
			} else {
				SetPixelStart(deviceStart_x);
				SetPixelEnd(_h+deviceStart_x);
				SetPixelActual(deviceStart_x);
			}
		}
		DrawBar();
	}
	
	/**
	* Shows the popup menu 
	*/
	void mBarRange :: onShowPopupMenu (wxMouseEvent& event)
	{
		if (activeState)
		{
			bool validClic = false;
			if (_orientation)
			{
				validClic = event.GetX() >= deviceStart_x && event.GetY()<= (_h + deviceStart_y);
			}
			else
			{
				validClic = event.GetX()>=deviceStart_y && event.GetX()<= (_h+deviceStart_y) && event.GetY()>deviceStart_x;
			}
			if (validClic)
			{
				if(_orientation)
					setClickedX(event.GetX());
				else
					setClickedX(event.GetY());

				if (getClickedX()<=_h)
				{						
					bool in_StartTri 	= (clickedX>=GetPixelStart()-5+ deviceStart_x) && (clickedX<=GetPixelStart()+5+ deviceStart_x);
					bool in_EndTri 		= (clickedX>=GetPixelEnd()-5+ deviceStart_x) && (clickedX<=GetPixelEnd()+5+ deviceStart_x);
					bool in_actualT		= (clickedX>=GetPixelActual()-5+ deviceStart_x) && (clickedX<=GetPixelActual()+5+ deviceStart_x);
					bool in_movingBar 	= (clickedX>GetPixelStart()+5+ deviceStart_x) && (clickedX<GetPixelEnd()-5+ deviceStart_x);

					if(in_StartTri)
						_selectionMoveId = 1;
					else if( in_EndTri )
						_selectionMoveId = 2;
					else if( in_actualT )
						_selectionMoveId = 3;
					else if( in_movingBar )
						_selectionMoveId = 4;
				}				
				PopupMenu( &b_popmenu, event.GetX(), event.GetY());
			}		
		}
	}
	
	/**
	* Reacts to the cntID_ADD_COLOR_POINT wxCommandEvent and adds a color degrade point to the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void mBarRange :: onChangePartColor ( wxCommandEvent& anEvent )
	{
		bool okSelectedColor = false;
		wxColour selectedColour;
		wxColourData data;
		wxColourDialog dialog( GetParent(), &data);

		if ( dialog.ShowModal() == wxID_OK )
		{
			selectedColour = dialog.GetColourData().GetColour();
			okSelectedColor = true;
		}
		if( okSelectedColor )
		{
			if (_selectionMoveId==1 )
				start_Colour = selectedColour;
			else if (_selectionMoveId==2 )
				end_Colour = selectedColour;
			else if( _selectionMoveId==3 )
				actual_Colour = selectedColour;
			else if( _selectionMoveId==4 )
				bar_Colour = selectedColour;		
		}
		_selectionMoveId = -1;
        RefreshForce();
		
	}
	
	/**
	* Reacts to the cntID_ENABLE_ACTUAL (false) wxCommandEvent enables the actual to be between the the range.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void mBarRange :: onEnableRange_Actual ( wxCommandEvent& anEvent )
	{
		if (!_in_rangeProperty)
		{
			if(IsActualInRange())
			{
				SetInRangeProperty (true);
				b_popmenu.SetLabel (cntID_ENABLE_ACTUAL, _T("Disable actual in range"));
			}
		}
		else
		{
			SetInRangeProperty (false);
			b_popmenu.SetLabel (cntID_ENABLE_ACTUAL, _T("Enable actual in range"));			
		}
	}

	/**
	* Reacts to the cntID_MOVABLE_ACTUAL_BAR wxCommandEvent by enabling or disabling the property of moving the actual triangle with the bar, just when it is inside of it.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void  mBarRange :: onMovable_ActualWithBar ( wxCommandEvent& anEvent )
	{
		if (_moveActualWithBar )
		{
			_moveActualWithBar = false;
			b_popmenu.SetLabel (cntID_MOVABLE_ACTUAL_BAR, _T("Move actual+bar simultaneously"));
		}
		else
		{
			if(IsActualInRange())
			{
				_moveActualWithBar = true;
				b_popmenu.SetLabel (cntID_MOVABLE_ACTUAL_BAR, _T("Move actual-bar independent"));
			}
		}
	}

		/*
	* Set active state 
	* param activeNow The new state
	*/
	void mBarRange :: setActiveStateTo (bool activeNow)
	{
		activeState = activeNow;
	}
	
	/*
	* Gets the active state of the bar
	*  return activeState The actual state
	*/
	bool mBarRange :: isActive()
	{
		return activeState;
	}

	/*
	* Gets the real-x value to draw a vertical line
	* return realX_vertical_line The real x value for the vertical line
	*/
	int	mBarRange :: getRealX_vertical_line()
	{
		return realX_vertical_line;
	}

	/*
	* Sets the real-x value to draw a vertical line
	* param newReal_x The new real x value for the vertical line
	*/
	void mBarRange :: setRealX_vertical_line(int newReal_x)
	{
		realX_vertical_line = newReal_x;
	}

	/*
	* Gets the device value form the end of this panel to the end of the drawing area in the device in pixels
	* return deviceEndMargin The value asigned to the right margin
	*/
	int	mBarRange :: getDeviceEndX()
	{
		return deviceEndMargin;
	}

	/*
	* Sets the new device (deviceEndMargin) value form the end of this panel to the end of the drawing area in the device
	* param newDeviceEnd_pixels The new pixel value to asign to the right(horizontal view), underneath(vertical view) margin in pixels
	*/
	void mBarRange :: setDeviceEndMargin(int newDeviceEnd_pixels)
	{
		deviceEndMargin = newDeviceEnd_pixels;
	}

	/*
	* Gets the last clickedX pixel coord inside the bar with respect to the container panel.
	* return clickedX The x-coord pixel value
	*/
	int mBarRange :: getClickedX()
	{
		return clickedX;
	}

	/*
	* Sets the last clickedX pixel coord inside the bar with respect to the container panel.
	* param nwClickX The x-coord pixel value
	*/
	void mBarRange :: setClickedX(int nwClickX)
	{
		clickedX = nwClickX;
	}


		/*
	* Gets the start porcentage with respect to the represented values of the bar
	* return The porcentage represented by the start  showing point
	*/
	float mBarRange :: getStartShowPorcentage()
	{
		return (float)( 1+(_start - _max)/(_max-_min));
	}

	/*
	* Gets the end porcentage with respect to the represented values of the bar
	* return The porcentage represented by the end showing point
	*/
	float mBarRange :: getEndShowPorcentage()
	{
		return (float) (1+(_end - _max)/(_max-_min));
	}

	/*
	* Gets the actual porcentage with respect to the represented values of the bar
	* return The porcentage represented by the actual  showing point
	*/
	float mBarRange :: getActualShowPorcentage()
	{
		return (float) (1+(_actual - _max)/(_max-_min));
	}

	int mBarRange :: getLogicValueofPixel(int thePixel)
	{
		return _min+((thePixel - deviceStart_x)*( _max - _min))/(_w-deviceEndMargin);
	}

	/*
	* Sets the condition for knowing if the actual triangle is being drawed or not
	* param drawActual The condition to set for drawing or not the actual control (true for drawing)
	*/
	void mBarRange :: setIfWithActualDrawed(bool drawActual)
	{
		if(!withActualDrawed && drawActual)
		{	
			b_popmenu.Append (cntID_ENABLE_ACTUAL, _("Enable actual in range"), _("Enables/Disables the actual triangle to be or not in range"));
			b_popmenu.Append (cntID_MOVABLE_ACTUAL_BAR, _("Move actual-bar simultaneously"), _("Disables the actual triangle to move with the bar"));		
		}
		else if (withActualDrawed && !drawActual)
		{
			b_popmenu.Remove(cntID_ENABLE_ACTUAL);
			b_popmenu.Remove(cntID_MOVABLE_ACTUAL_BAR);
		}
		withActualDrawed = drawActual;
		Refresh();	
	}

	/*
	* Gets the condition for knowing if the actual triangle is being drawed or not
	* return withActualDrawed The condition for drawing or not the actual control
	*/
	bool mBarRange :: getIfWithActualDrawed()
	{
		return withActualDrawed;
	}

	void mBarRange::createAndSendEvent(WXTYPE theEventType)
	{
		wxCommandEvent cevent( theEventType, GetId() );
		cevent.SetEventObject( this );
		GetEventHandler()->ProcessEvent( cevent );
	}

	/*
	* Sets the background color od the bar
	* theColor The color to set to the backgroundColor
	*/
	void mBarRange :: setBackgroundColor(wxColour theColor)
	{
		backgroundColor = theColor;
	}

	/*
	* Sets the guide line color
	* param theNwGuideLineColor The color to set to the guideLineColor
	*/
	void mBarRange :: setGuideLineColour(wxColour theNwGuideLineColor)
	{
		guideLineColor = theNwGuideLineColor;
	}

	/*
	* Gets the guide line color
	* return guideLineColor The color of the guideLine
	*/
	wxColour mBarRange :: getGuideLineColour()
	{
		return guideLineColor;
	}

	void  mBarRange ::onLeftClicDown(wxMouseEvent& event )
	{
		acceptedClick = true;		
		SetFocus();
	}

	void  mBarRange::onLeftClickUp(wxMouseEvent& event )
	{	
		acceptedClick = false;
	}

//EED 20 Juillet 2011
	void  mBarRange::onKey(wxKeyEvent& event)
	{
		int step=0;
		if ((event.GetKeyCode()==314) || (event.GetKeyCode()==317))
		{
			step=-1;
		}
		
		if ((event.GetKeyCode()==315) || (event.GetKeyCode()==316))
		{
			step=1;
		}
		
		if (step!=0)
		{
			if (_selectionMoveId == 1) // start
			{
				SetStart(GetStart()+step);
				createAndSendEvent( wxEVT_TSBAR_START );
			}
			
			if (_selectionMoveId == 2) // end
			{
				SetEnd(GetEnd()+step);
				createAndSendEvent( wxEVT_TSBAR_END );
			}
		
			if (_selectionMoveId == 3) // actual
			{
				SetActual(GetActual()+step);
				createAndSendEvent( wxEVT_TSBAR_ACTUAL );
			}
		
			if (_selectionMoveId == 4) // bar
			{
				if (( GetStart()+step>=_min ) && ( GetEnd()+step<=_max ))
				{
					SetStart(GetStart()+step);
					SetEnd(GetEnd()+step);
					if (_moveActualWithBar) 
					{ 
						SetActual(GetActual()+step); 
					}
					createAndSendEvent( wxEVT_TSBAR_START );
					createAndSendEvent( wxEVT_TSBAR_END );
					createAndSendEvent( wxEVT_TSBAR_ACTUAL );
					createAndSendEvent( wxEVT_TSBAR_MOVED );
				} // Start>_min  &&  End<_max
			}// _selectionMoveId == 4
		} // step
	}


