/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

//------------------------------------------------------------------------------------------------------------
// Class definition include
//------------------------------------------------------------------------------------------------------------
#include "pColorBar.h"
#include <math.h>

//------------------------------------------------------------------------------------------------------------
// Generated events declaration and definition
//------------------------------------------------------------------------------------------------------------
/*
* How to catch generated event: EVT_COMMAND(ID_CONTAINER_WINDOW_LISTENER, wxEVT_TYPE_EVENT, ContainerClass :: determinedEventFor)
*/
BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_EVENT_TYPE( wxEVT_ADDED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_REMOVED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_MOVED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_CHANGED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_ON_COLOR_BAR, -1 )
END_DECLARE_EVENT_TYPES()


DEFINE_EVENT_TYPE( wxEVT_ADDED_POINT )
DEFINE_EVENT_TYPE( wxEVT_REMOVED_POINT )
DEFINE_EVENT_TYPE( wxEVT_MOVED_POINT )
DEFINE_EVENT_TYPE( wxEVT_CHANGED_POINT )
DEFINE_EVENT_TYPE( wxEVT_ON_COLOR_BAR )

//------------------------------------------------------------------------------------------------------------
// Class implementation
//------------------------------------------------------------------------------------------------------------
/** file pColorBar.cxx */
IMPLEMENT_CLASS(pColorBar, wxScrolledWindow)

//------------------------------------------------------------------------------------------------------------
// Handled events table
//------------------------------------------------------------------------------------------------------------
BEGIN_EVENT_TABLE(pColorBar, wxScrolledWindow)
	EVT_PAINT( pColorBar :: onPaint)
	EVT_SIZE ( pColorBar :: OnSize )
	EVT_MOTION ( pColorBar :: onMouseMove)
	EVT_RIGHT_DOWN( pColorBar :: onShowPopupMenu)
	EVT_LEFT_DCLICK( pColorBar :: onLeftButtonDClick)
	EVT_LEFT_DOWN( pColorBar :: onLeftClicDown)
	EVT_LEFT_UP( pColorBar :: onLeftClickUp)
	EVT_MENU( cntID_DEL_COLOR_POINT, pColorBar :: onDeleteColorPoint)
	EVT_MENU( cntID_ADD_COLOR_POINT, pColorBar :: onAddColorPoint)
	EVT_MENU( cntID_CHANGE_COLOR_POINT, pColorBar :: onChangeColourSelectedPoint)
	EVT_MENU( cntID_TRIANGLES_UP, pColorBar :: onTrianglesUp_Figure)
	EVT_MENU( cntID_TRIANGLES_DOWN, pColorBar :: onTrianglesDown_Figure)
	EVT_MENU( cntID_TRIANGLES_LEFT, pColorBar :: onTrianglesLeft_Figure)
	EVT_MENU( cntID_TRIANGLES_RIGHT, pColorBar :: onTrianglesRight_Figure)
	EVT_MENU( cntID_RECTANGLES, pColorBar :: onRectangles_Figure)
	EVT_MENU( cntID_DEGRADE_CONTROL, pColorBar :: onDegradeControl)
	EVT_MIDDLE_UP( pColorBar :: onShowPopupMenu )
	EVT_RIGHT_UP ( pColorBar :: onShowPopupMenu )
END_EVENT_TABLE()

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------

/*
* Creates a colorBar instance 
* param *parent Container event listener window
* param _w Width of the color bar drawing area
* param _h Height of the color bar drawing area
* param _bar_orientation VERTICAL (false) or HORIZONTAL (true) direction to set 
*/
pColorBar :: pColorBar (wxWindow *parent, int _w, int _h, bool _bar_orientation)
:wxScrolledWindow(parent, -1, wxDefaultPosition, wxDefaultSize, wxTAB_TRAVERSAL)
{
		_logicalBar = new LogicalColorBar();
		colourParent =  GetParent()->GetBackgroundColour();
		deviceEndMargin = 0;
		acceptedClick = true;
		setGapValues (0,6);
		setWidth(_w);
		setHeight(_h);
		
		// Setting the default represented values 
		setDegradeState(true);	
		setStaticLimitsTo(false);
		viewingRange = false;
		accumTemporalPoints = 0;
		deviceEndMargin = 0;
		activeState = false;
		avaliablePointToMove = -1;
						
		movingPointIndex = -1;
		movingNodePoint = NULL;
		realX_vertical_line = -1;
		guideLineColor		=   wxColour(255,0,0);

		//TRIANGLES UP
		figure = new pFigure( RECTANGLE, FIGURE_WIDTH, bar_height, -2, _bar_orientation );
		setOrientation ( _bar_orientation );
		
		// Appending menu items to the pop-menu
		c_popmenu.Append( cntID_ADD_COLOR_POINT, _T("Add color point"), _T("Adds a color point in the clicked point"));
		c_popmenu.Append( cntID_DEL_COLOR_POINT, _T("Delete color point"), _T("Deletes the color point at the clicked point"));
		c_popmenu.Append( cntID_CHANGE_COLOR_POINT, _T("Change color"), _T("Changes the color at the clicked point"));
		c_popmenu.Append ( cntID_DEGRADE_CONTROL, _T("Turn off degrade"), _T("Turns on/off the degrade"),wxITEM_SEPARATOR);
		
		changesMenu = new wxMenu;
		changesMenu->AppendCheckItem ( cntID_TRIANGLES_UP, _T("Triangles up"),_T("Sets triangles up figure"));
		changesMenu->AppendCheckItem ( cntID_TRIANGLES_DOWN, _T("Triangles down"),_T("Sets triangles down figure"));
		changesMenu->AppendCheckItem ( cntID_TRIANGLES_LEFT, _T("Triangles left"),_T("Sets triangles left figure"));
		changesMenu->AppendCheckItem ( cntID_TRIANGLES_RIGHT, _T("Triangles right"),_T("Sets triangles right figure"));
		changesMenu->AppendCheckItem ( cntID_RECTANGLES, _T("Rectangles"),_T("Sets rectangle figure"));
		c_popmenu.Append(cntID_CHANGE_FIGURE, _T("Change figure"),changesMenu, _T("Changes the figure of the color points"));
		changesMenu->Check (cntID_TRIANGLES_UP, true);
}

pColorBar :: ~pColorBar()
{

}
	//------------------------------------------------------------------------------------------------------------
	// Color bar proccessed events methods implementation
	//------------------------------------------------------------------------------------------------------------
	
	void pColorBar :: OnSize ( wxSizeEvent &WXUNUSED(event) )
	{
		wxRect rectTotal = GetClientRect();			
		if(getOrientation())
		{		
			setWidth( rectTotal.GetWidth() - deviceEndMargin );			
		} 
		else 
		{
			setWidth( rectTotal.GetHeight() - deviceEndMargin);				
		}
		Refresh();
	}
	/**
	* Reacts to the mouse movement inside the color bar for moving the clicked color point in the x-axis.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_MOVED_COL_POINT if a point was moved.
	* param event The mouse actioned event
	*/
	void pColorBar ::  onMouseMove ( wxMouseEvent& event )
	{	
		if(!event.LeftIsDown())
			avaliablePointToMove = -1;

		if (acceptedClick && activeState && _logicalBar->getCount()>0)
		{
			bool movedPoint = false;
			wxPoint point = event.GetPosition();
			double realClicked = getOrientation() ? convertToRealValue(point.x) : convertToRealValue(point.y);
			bool rangeToMove = realClicked<=maxX_represented_Tshow && realClicked>=minX_represented_Tshow;//getOrientation() ? (point.x >=deviceStart_x && point.x<=bar_width+deviceStart_x-deviceEndMargin ) : ( point.y>=deviceStart_y && point.y<=bar_width+deviceStart_x-deviceEndMargin);
			if ( rangeToMove && acceptedClick && showedColorPoints.size()>0 )
			{	
				if ( movingPointIndex == -1 )
				{
					startNode_show = showedColorPoints.front();
					lastNode_show = showedColorPoints.back();
									
					//int numberEdges = figure->getNumberEdges(); // JPRx
					//wxPoint points [4]; // JPRx
					//int i=0; // JPRx
					int a;
					for (a=0; a<showedColorPoints.size() && movingPointIndex==-1 ; a++ )
					{
						pColorPoint * nodePoint = showedColorPoints[a];
						int point_pixelX = convertToPixelValue(nodePoint -> getRealX());
						point_pixelX+= deviceStart_x;
						if ( getOrientation() )
							movingPointIndex  =  (figure ->isPointInside (point_pixelX,point.x))? a : -1;
						else
							movingPointIndex  =  (figure ->isPointInside (point_pixelX,point.y))? a : -1;

						if(movingPointIndex != -1)
						{
							movingNodePoint = nodePoint;		
							if(avaliablePointToMove ==-1)
								avaliablePointToMove = movingPointIndex;
						}												
					}						
				}
				if (event.LeftIsDown() && movingNodePoint && movingPointIndex!=-1 && avaliablePointToMove==movingPointIndex)
				{					
					bool hasPrevPoint = movingPointIndex > 0;
					bool hasNextPoint = movingPointIndex < showedColorPoints.size()-1;

					pColorPoint * prevPoint = hasPrevPoint? showedColorPoints[movingPointIndex-1]: NULL; 
					pColorPoint * nextPoint = hasNextPoint ? showedColorPoints[movingPointIndex+1]: NULL;

					if ( hasPrevPoint && prevPoint!=NULL )
					{							
						if ( hasNextPoint && nextPoint!=NULL )
						{								
							double previousValue = prevPoint->getRealX();
							double nextValue = nextPoint->getRealX();
							movedPoint = prevPoint->isTemporalColor() ? 
								( nextPoint->isTemporalColor() ? (realClicked>=previousValue)&&(realClicked<=nextValue) : (realClicked>=previousValue)&&(realClicked<nextValue))
								:(realClicked>previousValue)&&(realClicked<nextValue);	
							if(!movedPoint)
								avaliablePointToMove = movingPointIndex;
							if(!movedPoint && realClicked<previousValue) 
							{
								realClicked = previousValue+1;
								movedPoint = true;
							}
							else if(!movedPoint && realClicked>nextValue) 
							{
								realClicked = nextValue-1;
								movedPoint = true;
							}														
						}
						else 
						{
							if ( !staticLimits )
							{
								// Is the last point
								double previousValue = prevPoint->getRealX();
								movedPoint = prevPoint->isTemporalColor() ? 
									( realClicked>=previousValue ) && ( realClicked<= maxX_represented_Tshow ) 
									: ( realClicked>previousValue ) && ( realClicked<= maxX_represented_Tshow );
								if(!movedPoint)
									avaliablePointToMove = movingPointIndex;
								if(!movedPoint && realClicked<previousValue)
								{
									realClicked = prevPoint->isTemporalColor()? previousValue: previousValue+1;
									movedPoint = true;
								}								
								else if (!movedPoint && realClicked>=maxX_represented_Tshow) 
								{
									realClicked = maxX_represented_Tshow;
									movedPoint = true;
								}								
							}
						}
					}
					else
					{
						if ( hasNextPoint && nextPoint!=NULL )
						{
							if ( !staticLimits )
							{
								// Is the first point								
								double nextValue = nextPoint->getRealX();
								movedPoint = ( nextPoint->isTemporalColor() ? (realClicked<=nextValue) && ( realClicked>= minX_represented_Tshow )
									:( realClicked<nextValue ) && ( realClicked>= minX_represented_Tshow ));										
									
								if(!movedPoint)
									avaliablePointToMove = movingPointIndex;
								if(!movedPoint && realClicked>nextValue) 
								{
									realClicked = nextPoint->isTemporalColor() ? nextValue : nextValue-1;
									movedPoint = true;
								}	
								else if (!movedPoint && realClicked<=minX_represented_Tshow) 
								{
									realClicked = minX_represented_Tshow;
									movedPoint = true;
								}								
							}
						}
						else
						{
							//Is the only point in the color bar
							movedPoint = ( realClicked<= maxX_represented_Tshow ) && ( realClicked>= minX_represented_Tshow );	
							if(!movedPoint)
									avaliablePointToMove = movingPointIndex;
							if (!movedPoint && realClicked<=minX_represented_Tshow) 
							{
								realClicked = minX_represented_Tshow;
								movedPoint = true;
							}
							else if(!movedPoint && realClicked>=maxX_represented_Tshow) 
							{
								realClicked = maxX_represented_Tshow;
								movedPoint = true;
							}
						}
					}
					if(viewingRange)
					{  
						bool movingTempStart = movingNodePoint->getRealX() == startNode_show->getRealX() && movingNodePoint->isTemporalColor();
						bool movingTempEnd = movingNodePoint->getRealX()== lastNode_show->getRealX() && movingNodePoint->isTemporalColor();
						if( movingTempStart || movingTempEnd )
							movedPoint = false;
					}
				}	
					// Inform movedPoint event
				if ( movedPoint )
				{					
					movingNodePoint->setRealX ( realClicked );
					RefreshForce();//Refresh to show the point is moved
					createAndSendEvent(wxEVT_MOVED_POINT);						
				}
				else
				{
					movingPointIndex = -1;						
					//Creating a general event of mouse move to actualize views
					createAndSendEvent( wxEVT_ON_COLOR_BAR );
				}			
			}
			else
			{
				movingPointIndex = -1;	
				//Creating a general event of mouse move to actualize views
				createAndSendEvent( wxEVT_ON_COLOR_BAR );
			}			
			if ( movingPointIndex ==-1 )
			{
				RefreshForce();//Refresh to show the point is not selected anymore		
			}
		}		
	}

	void pColorBar ::  Refresh(bool eraseBackground , const wxRect* rect )
	{
		wxScrolledWindow::Refresh(false);
	}


	/*
	* Reacts to the event of drawing the bar and draws it
	* param &WXUNUSED(event) The correspondig wxPaintEvent actioned event
	*/
	void pColorBar ::  onPaint ( wxPaintEvent &WXUNUSED(event) )
	{
		repaintView();
		if ( getOrientation() ) // HORIZONTAL
		{
			wxMemoryDC temp_dc;
			temp_dc.SelectObject( * colorBar_Bitmap );
			wxPaintDC dc( this );
					
//			dc.Blit(deviceStart_x, deviceStart_y, bar_width+deviceStart_x, bar_height+deviceStart_y, &temp_dc, deviceStart_x, deviceStart_y);

			dc.Blit(deviceStart_x, deviceStart_y, bar_width/*-deviceEndMargin*/, bar_height, &temp_dc, 0, 0);

			// Bitmap for posible needed information to be shown
			/*
			temp_dc.SelectObject( * information_Bitmap );
			dc.Blit(0,bar_height, width, height, &temp_dc, 0, height);
			*/
		}
		else
		{
			wxMemoryDC temp_dc;
			temp_dc.SelectObject( * colorBar_Bitmap );
			wxPaintDC dc( this );
//			dc.Blit(deviceStart_y,deviceStart_x, bar_height+deviceStart_y, bar_width+deviceStart_x, &temp_dc, deviceStart_y, deviceStart_x);	
			
			dc.Blit(deviceStart_y,deviceStart_x, bar_height, bar_width-deviceEndMargin, &temp_dc, 0, 0);				
			// Bitmap for posible needed information to be shown
			/*
			temp_dc.SelectObject( * information_Bitmap );
			dc.Blit(0,bar_width, height, width, &temp_dc, 0, width );
			*/
			temp_dc.SelectObject(wxNullBitmap);
		}
	}

	/**
	* Reacts to the cntID_ADD_COLOR_POINT wxCommandEvent and adds a color degrade point to the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_ADDED_COL_POINT if the point was inserted succesfully.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar ::  onAddColorPoint ( wxCommandEvent& anEvent )
	{
		bool addedPoint = false;
		double real_x = getOrientation() ? convertToRealValue( clickedX ) : convertToRealValue( clickedY );
		wxColour selectedColor = getSelectedColour();
		if (okSelectedColor)
		{
			addedPoint = addColorPoint (real_x, selectedColor);			
		}
		// Inform addedPoint event
		if ( addedPoint )
		{
			//RefreshForce();
			createAndSendEvent( wxEVT_ADDED_POINT );			
		}
	}

	/**
	* Reacts to the cntID_DEL_COLOR_POINT wxCommandEvent and deletes a color degrade point of the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_REMOVED_COL_POINT if the point was succesfully removed.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar :: onDeleteColorPoint ( wxCommandEvent& anEvent )
	{
		bool 		deletedPoint 		= false;
		double 	real_x 				= getOrientation() ? convertToRealValue( clickedX ) : convertToRealValue( clickedY );
					deletedPoint 		= deleteColorPoint(real_x);
		// Inform deletedPoint event
		if ( deletedPoint )
		{
			//RefreshForce();
			createAndSendEvent( wxEVT_REMOVED_POINT );			
		} //if
	}

		/**
	* Reacts to the cntID_CHANGE_COLOR_POINT wxCommandEvent and changes the assigned color to the selected color degrade point of the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_CHANGED_POINT if the point was succesfully removed.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar :: onChangeColourSelectedPoint ( wxCommandEvent& anEvent )
	{
		bool changedColour = false;
		changedColour = getOrientation() ? changeColor( clickedX ) : changeColor( clickedY );
		// Inform deletedPoint event
		if ( changedColour )
		{
			//RefreshForce();
			createAndSendEvent( wxEVT_CHANGED_POINT );			
		} // if
	}


	/**
	* Reacts to the cntID_TRIANGLES_UP wxCommandEvent and changes the assigned figure to -triangles up- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar :: onTrianglesUp_Figure ( wxCommandEvent& anEvent  )
	{
		if( !changesMenu->IsChecked(cntID_TRIANGLES_UP) )
		{
			changeFigure (-8);
			changesMenu->Check (cntID_TRIANGLES_UP, true);
			changesMenu->Check (cntID_TRIANGLES_DOWN, false);
			changesMenu->Check (cntID_TRIANGLES_LEFT, false);
			changesMenu->Check (cntID_TRIANGLES_RIGHT, false);
			changesMenu->Check (cntID_RECTANGLES, false);
		}
	}
	/**
	* Reacts to the cntID_TRIANGLES_DOWN wxCommandEvent and changes the assigned figure to -triangles down- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar :: onTrianglesDown_Figure ( wxCommandEvent& anEvent  )
	{
		if( !changesMenu->IsChecked(cntID_TRIANGLES_DOWN) )
		{
			changeFigure (-2);
			changesMenu->Check (cntID_TRIANGLES_UP, false);
			changesMenu->Check (cntID_TRIANGLES_DOWN, true);
			changesMenu->Check (cntID_TRIANGLES_LEFT, false);
			changesMenu->Check (cntID_TRIANGLES_RIGHT, false);
			changesMenu->Check (cntID_RECTANGLES, false);
		}
	}
	/**
	* Reacts to the cntID_TRIANGLES_LEFT wxCommandEvent and changes the assigned figure to -triangles left- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar :: onTrianglesLeft_Figure ( wxCommandEvent& anEvent  )
	{
		if( !changesMenu->IsChecked(cntID_TRIANGLES_LEFT) )
		{
			changeFigure (-4);
			changesMenu->Check (cntID_TRIANGLES_UP, false);
			changesMenu->Check (cntID_TRIANGLES_DOWN, false);
			changesMenu->Check (cntID_TRIANGLES_LEFT, true);
			changesMenu->Check (cntID_TRIANGLES_RIGHT, false);
			changesMenu->Check (cntID_RECTANGLES, false);
		}
	}
	/**
	* Reacts to the cntID_TRIANGLES_RIGHT wxCommandEvent and changes the assigned figure to -triangles right- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar :: onTrianglesRight_Figure ( wxCommandEvent& anEvent  )
	{
		if( !changesMenu->IsChecked(cntID_TRIANGLES_RIGHT) )
		{
			changeFigure (-6);
			changesMenu->Check (cntID_TRIANGLES_UP, false);
			changesMenu->Check (cntID_TRIANGLES_DOWN, false);
			changesMenu->Check (cntID_TRIANGLES_LEFT, false);
			changesMenu->Check (cntID_TRIANGLES_RIGHT, true);
			changesMenu->Check (cntID_RECTANGLES, false);
		}
	}
	/**
	* Reacts to the cntID_RECTANGLES wxCommandEvent and changes the assigned figure to -rectangles- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar :: onRectangles_Figure ( wxCommandEvent& anEvent  )
	{
		if( !changesMenu->IsChecked(cntID_TRIANGLES_RIGHT) )
		{
			changeFigure (RECTANGLE);// RECTANGLE = 4
			changesMenu->Check (cntID_TRIANGLES_UP, false);
			changesMenu->Check (cntID_TRIANGLES_DOWN, false);
			changesMenu->Check (cntID_TRIANGLES_LEFT, false);
			changesMenu->Check (cntID_TRIANGLES_RIGHT, false);
			changesMenu->Check (cntID_RECTANGLES, true);
		}
	}

	/**
	* Reacts to the cntID_DEGRADE_CONTROL wxCommandEvent and turns on/off the degrade in the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void pColorBar ::  onDegradeControl ( wxCommandEvent& anEvent )
	{
		bool degrade = getDegradeState();
		wxString nextAction_text;
		if(degrade)
		{
			nextAction_text = _T("Turn on degrade");	
			setDegradeState (false);
		}
		else
		{
			nextAction_text = _T("Turn off degrade");
			setDegradeState (true);
		}
		c_popmenu.SetLabel ( cntID_DEGRADE_CONTROL, nextAction_text ); 
	}

	/**
	* Reacts to the wxEVT_RIGHT_DCLICK wxMouseEvent and adds a color degrade point to the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_ADDED_COL_POINT if the point was succesfully inserted.
	* param & event The wxMouseEvent actioned event 
	*/
	void pColorBar ::  onLeftButtonDClick (wxMouseEvent& event)
	{
		if (activeState)
		{
			bool addedPoint = false;
			wxPoint point = event.GetPosition();
			bool posiblePoint = false;
			if( getOrientation() )
			{
				posiblePoint = point.x>=deviceStart_x && point.y<= (bar_height + deviceStart_y);
			}
			else
			{
				posiblePoint = point.x>=deviceStart_y && point.x<= (bar_height+deviceStart_y) && point.y>deviceStart_x;
			}

			if (posiblePoint)
			{
				double real_x = getOrientation() ? convertToRealValue( point.x ) : convertToRealValue( point.y );
				wxColour selectedColor = getSelectedColour();
				if (okSelectedColor)
				{
					addedPoint = addColorPoint (real_x, selectedColor);					
				}
				// Inform addedPoint event
				if ( addedPoint )
				{
					//RefreshForce(); Is all ready refreshed on the adition of the point method
					createAndSendEvent( wxEVT_ADDED_POINT );			
				}
			}
		}
	}
		
	void  pColorBar ::onLeftClicDown(wxMouseEvent& event )
	{
		acceptedClick = true;		
	}

	void  pColorBar ::onLeftClickUp(wxMouseEvent& event )
	{	
		movingPointIndex = -1;
		acceptedClick = false;
	}
	/*
	* Shows the popup menu 
	*/
	void pColorBar ::  onShowPopupMenu (wxMouseEvent &event)
	{
		if (activeState)
		{
			if(showedColorPoints.empty())
			{
				c_popmenu.Enable(cntID_DEL_COLOR_POINT, false);			
			}
			else
			{
				c_popmenu.Enable(cntID_DEL_COLOR_POINT, true);
			}
			bool setClickedValues = false;
			if( getOrientation() )
			{
				setClickedValues = event.GetX()>=deviceStart_x && event.GetY()<= (bar_height + deviceStart_y);
			}
			else
			{
				setClickedValues = event.GetX()>=deviceStart_y && event.GetX()<= (bar_height+deviceStart_y) && event.GetY()>deviceStart_x;
			}
			if (setClickedValues)
			{
				setClickedX ( event.GetX() );
				setClickedY ( event.GetY() );
				RefreshForce();
				PopupMenu( &c_popmenu, event.GetX(), event.GetY());
			}
		}
	}
	//------------------------------------------------------------------------------------------------------------
	// Other methods implementation
	//------------------------------------------------------------------------------------------------------------

	/*
	* Method that reinitiates attributes of the color bar to the given points
	* param minRealValue The minimum represented value (real value)
	* param maxRealValue The maximum represented value (real value)
	*/
	void pColorBar :: reinitiateColorBar(double minRealValue, double maxRealValue)
	{

		_logicalBar -> clearPoints();
		showedColorPoints.clear();
				
		movingPointIndex = -1;
		realX_vertical_line = -1;
		activeState = true;
		accumTemporalPoints = 0;
		viewingRange = false;
		setDegradeState(true);			
		setRepresentedValues(minRealValue, maxRealValue);
	}

	/*
	* Method that reinitiates  attributes of the color bar and set the points given by parameter
	* param pointsVector The the vector of the new points, the color points must have at least two points.
	*/
	void pColorBar :: reinitiateColorBarTo (std::vector<pColorPoint *> pointsVector)
	{
		_logicalBar -> clearPoints();
		_logicalBar -> setColorPoints(pointsVector);
		showedColorPoints.clear();
		
		movingPointIndex = -1;
		realX_vertical_line = -1;
		activeState = true;
		accumTemporalPoints = 0;		
		viewingRange = false;
		setDegradeState(true);					
		
		double minRealValue = _logicalBar -> getMinValue();
		double maxRealvalue = _logicalBar -> getMaxValue();

		setRepresentedValues(minRealValue, maxRealvalue);
	}

	/*
	* Method that sets a copy of the points of the color bar to the given pointer parameter
	* param pointsList The list for getting the points
	*/
	void pColorBar :: getPointsListWithTemps (std::vector<pColorPoint*> &pointsVector)
	{		
		pColorPoint* cPoint;
		for (int a =0; a<showedColorPoints.size() ; a++)	
		{
			cPoint = showedColorPoints[a];
			pColorPoint* copyPoint = new pColorPoint(cPoint->getRealX(), cPoint->getColor( ), (cPoint->isTemporalColor()));
			pointsVector.push_back(copyPoint);			
		}
	}

	/*
	* Method that sets a copy of the points of the color bar to the given pointer parameter, and sets the default values to the needeed attributes, it doesn't
	* includes the teporal points created
	* param pointsList The list for getting the points
	*/
	
	void pColorBar :: getAddedColorsPointsList (std::vector<pColorPoint*> &pointsVector)
	{
		pColorPoint* cPoint;
		for (int a =0; a<_logicalBar->getCount(); a++)	
		{
			cPoint = _logicalBar ->getPointAtIndex(a);			
			pColorPoint* copyPoint = new pColorPoint(cPoint->getRealX(), cPoint->getColor( ), (cPoint->isTemporalColor()));
			pointsVector.push_back(copyPoint);			
		}
	}
	

	/**
	* Adds a color degrade point to the color bar.
	* param xRealValue The real xValue of the point
	* param theColour The assigned color for the point
	* param temporalStart Indicates if the inserted point is the temporal startShowing point
	* param temporalEnd Indicates if the inserted point is the temporal startShowing point
	* return Returns true if the point was succesfully inserted.
	*/
	bool pColorBar :: addColorPoint (double xRealValue, wxColour theColour/*, bool temporalStart, bool temporalEnd*/)
	{
		bool addedPoint = false;
		addedPoint = _logicalBar->addColorPoint(xRealValue,theColour);
		if (addedPoint) 
			RefreshForce();
		return addedPoint;
	}


	/**
	* Delets a color degrade point to the color bar.
	* param xRealValue The real xValue of the point to delete	
	* return Returns true if the point was succesfully inserted.
	*/
	bool pColorBar :: deleteColorPoint (double xRealValue)
	{	
		pColorPoint* actualPoint;		
		bool deletedPoint = false;
		for (int a =0; a<showedColorPoints.size() && !deletedPoint; a++)	
		{
			actualPoint = showedColorPoints[a];
			int actualX = (int)actualPoint -> getRealX();
			bool isInsideActual = figure -> isPointInside (convertToPixelValue(actualX), convertToPixelValue(xRealValue));
			if( actualX == xRealValue || isInsideActual)
			{
				movingPointIndex=-1;								
				deletedPoint = _logicalBar->deleteColorPoint(actualPoint -> getRealX());					
			}			
		}			
		if (deletedPoint) 
			RefreshForce();
		return deletedPoint;			
	}

	/**
	* Changes the color degrade point color value.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_CHANGED_COLOR_POINT if the point changed its colour.
	* param clickedValue The x-coord pixel value of the point to which the color change interests	
	* return Returns true if the color point was succesfully updated.
	*/
	bool pColorBar :: changeColor ( int clickedValue )
	{		
		bool colourUpdated = false;
		double xRealValue = convertToRealValue( clickedValue );
		pColorPoint* actualPoint;
		for (int a =0; a<showedColorPoints.size() && !colourUpdated; a++)	
		{
			actualPoint = showedColorPoints[a];
			int actualX = (int)actualPoint -> getRealX();
			bool isInsideActual = figure -> isPointInside (convertToPixelValue(actualX), clickedValue);
			if( actualX == xRealValue || isInsideActual)
			{
				wxColour newColour = getSelectedColour();
				if (okSelectedColor)
				{
					actualPoint -> setColor(newColour);
					colourUpdated = true;					
				}				
			}			
		}		
		if(colourUpdated)
			RefreshForce();
		return colourUpdated;
	}

	/*
	* Repaints the color bar in horizontal direction mode
	*/
	void pColorBar ::  repaintView(  )
	{
		wxMemoryDC temp_dc;
		temp_dc.SelectObject( * colorBar_Bitmap );		
		temp_dc.SetBrush(wxBrush( colourParent ,wxSOLID  ));
		temp_dc.SetPen(wxPen( colourParent,1,wxSOLID  ));
		if (getOrientation())
		{
			temp_dc.DrawRectangle(0, 0, bar_width, bar_height);
			temp_dc.SetPen(wxPen( wxColour(167,165,191),1,wxSOLID  ));
			temp_dc.DrawRectangle(0, 0, bar_width-deviceEndMargin, bar_height);
		}
		else
		{
			temp_dc.DrawRectangle(0, 0, bar_height, bar_width);
			temp_dc.SetPen(wxPen( wxColour(167,165,191),1,wxSOLID  ));
			temp_dc.DrawRectangle(0, 0, bar_height, bar_width-deviceEndMargin);
		}
		//wxPoint figPoints[4]; // JPRx
		
		if( _logicalBar -> getCount() >0)
		{
			updatePointsToDraw();
			
			int numberEdges = figure->getNumberEdges();
			wxPoint points [4];
			figure->getVertexPoints(points);
			figure ->setVertexPoints (points);

			int maxValue = showedColorPoints.size();
			pColorPoint * iniPoint;
			pColorPoint * endPoint;
			wxColor intialColor, finalColor;
			for (int a=0; a<maxValue; a++)
			{
				iniPoint = showedColorPoints[a];
				int ini_pixelX = convertToPixelValue(iniPoint -> getRealX());				
				intialColor = iniPoint -> getColor();
								
				if( a<maxValue-1 )
				{
					if( doingDegrade )
					{
						endPoint = showedColorPoints[a+1];
						int end_pixelX = convertToPixelValue(endPoint -> getRealX());
						finalColor = endPoint -> getColor();					
					
						int rectangle_width = end_pixelX - ini_pixelX;
						
						//*************************************************************************************
						int initial_r = intialColor.Red();
						int final_r = finalColor.Red()  ;
						int initial_g = intialColor.Green();
						int final_g= finalColor.Green();
						int initial_b = intialColor.Blue();
						int final_b= finalColor.Blue();
						
						float m_scope_r = (float)(final_r-initial_r)/rectangle_width;
						float m_scope_g = (float)(final_g-initial_g)/rectangle_width;
						float m_scope_b = (float)(final_b-initial_b)/rectangle_width;				

						int next_r = initial_r;
						int next_g = initial_g;
						int next_b = initial_b;

						float nxt_r = (float)next_r;
						float nxt_g = (float)next_g;
						float nxt_b = (float)next_b;

						for (int Xi =ini_pixelX; Xi<= end_pixelX; Xi++) 					
						{
							temp_dc.SetBrush(wxBrush( wxColour(next_r, next_g, next_b),wxSOLID  ));
							temp_dc.SetPen(wxPen( wxColour(next_r, next_g, next_b),1,wxSOLID  ));
							if( getOrientation() )
								temp_dc.DrawLine(Xi, 0, Xi, bar_height);
							else
								temp_dc.DrawLine(0, Xi, bar_height, Xi);

							nxt_r = (m_scope_r + nxt_r);
							nxt_g = (m_scope_g + nxt_g);
							nxt_b = (m_scope_b + nxt_b);
							
							next_r = (int)(nxt_r);
							next_g = (int)(nxt_g);
							next_b = (int)(nxt_b);								
						}									
					}				
				}
				if( movingPointIndex!=-1 && movingNodePoint!=NULL )
				{					
					if ( !(movingNodePoint -> isTemporalColor()) )
					{
						if( movingNodePoint==iniPoint && movingPointIndex!=-1 )
						{
							temp_dc.SetBrush(wxBrush(  wxColour(5,36,180), wxSOLID  ));
							temp_dc.SetPen(wxPen( wxColour(239,239,239), 1, wxSOLID  ));
							if (getOrientation())
								temp_dc.DrawPolygon(numberEdges, points, ini_pixelX, 0);
							else
								temp_dc.DrawPolygon(numberEdges, points, 0, ini_pixelX);
						}
						else
						{						
								temp_dc.SetBrush(wxBrush( intialColor, wxSOLID  ));	
								if(iniPoint->isTemporalColor())
									temp_dc.SetPen(wxPen( intialColor, 1, wxSOLID  ));			
								else
									temp_dc.SetPen(wxPen( wxColour(0,0,0), 1, wxSOLID  ));
								if (getOrientation())
									temp_dc.DrawPolygon(numberEdges, points, ini_pixelX, 0);
								else
									temp_dc.DrawPolygon(numberEdges, points, 0, ini_pixelX);					
						}
					}
				}
				else
				{
					if ( !(iniPoint -> isTemporalColor()) )
					{
						temp_dc.SetBrush(wxBrush( intialColor, wxSOLID  ));
						temp_dc.SetPen(wxPen( wxColour(0,0,0), 1, wxSOLID  ));
						if (getOrientation())
							temp_dc.DrawPolygon(numberEdges, points, ini_pixelX, 0);
						else
							temp_dc.DrawPolygon(numberEdges, points, 0, ini_pixelX);
					}							
				}		
			}
			if (realX_vertical_line!=-1)
			{
				temp_dc.SetPen(wxPen(  guideLineColor,1,wxDOT  ));
				int pixelX_guide = convertToPixelValue(realX_vertical_line);
				if (getOrientation())
					temp_dc.DrawLine(pixelX_guide, 0, pixelX_guide, bar_height);
				else
					temp_dc.DrawLine(0, pixelX_guide, bar_height, pixelX_guide);
			}
		}
	}
	
	/*
	* Forces the refresh view of the color bar
	*/
	void pColorBar ::  RefreshForce ()
	{
		Refresh(false);		
	}
	
	/*
	* Draws the color bar with its degrades
	* param recentlyChangedOrientation indicates if the orientation has been changed before calling this method
	*/
	void pColorBar ::  drawColorBar (bool recentlyChangedOrientation)
	{
		wxRect rectTotal = GetClientRect(); 
		if (recentlyChangedOrientation)
		{
			if ( getOrientation() ) // HORIZONTAL
			{
				//SetSize(width, height );
				wxWindow::SetSize( width, height );
				//SetSize( rectTotal.GetWidth(), rectTotal.GetHeight() );
				setWidth(GetClientRect().GetWidth()- deviceEndMargin);
				//setHeight(GetClientRect().GetHeight());
				
				colorBar_Bitmap = new wxBitmap( bar_width+1280, bar_height+1280 );
				//information_Bitmap = new wxBitmap( width-bar_width, height-bar_height );
			}
			else // VERTICAL
			{
				wxWindow::SetSize( height, width );
				//SetSize( rectTotal.GetHeight(), rectTotal.GetWidth() );
				setWidth(GetClientRect().GetHeight()- deviceEndMargin);
				//setHeight(GetClientRect().GetWidth());
				
				//SetWindowStyle( wxNO_FULL_REPAINT_ON_RESIZE );
				colorBar_Bitmap = new wxBitmap( bar_height+1280, bar_width+1280 );
				//information_Bitmap = new wxBitmap( height-bar_height, width-bar_width );
			}		
		}
	}
		
	/*
	* Changes the figure of the color points according to the parameter
	* param figureOrientation Is the corresponding number for identifying the figure 
	* RECTANGLE = 4,TRIANGLE LEFT = -4, TRIANGLE RIGHT = -6, TRIANGLE UP = -8, TRIANGLE DOWN = -2
	*/
	void pColorBar :: changeFigure(int figureOrientation)
	{		
		if ( figureOrientation<1 ) //The figure is a  a rotated triangle 
		{
			changePointsFigure_Edges(3);
			figure -> setFigureOrientation ( figureOrientation );
			
		}
		else //The figure is a rectangle
		{
			changePointsFigure_Edges(4);
		}	
		RefreshForce();
	}

	/**
	* Changes the figure used for the degreade color points to the indicated one by parameter
	* param figEdges Is the constant that represents the figure number of edges (TRIANGLE | RECTANGLE)
	*/
	void pColorBar :: changePointsFigure_Edges(int figEdges)
	{
		figure->setNumberEdges( figEdges);
	}
	
	/**
	* Gets the constant that represents the figure used for the degreade color points
	* return figureEdges Is the constant that represents the figure number of edges (TRIANGLE | RECTANGLE)
	*/
	int pColorBar :: getPointsFigure_Edges ()
	{
		return (figure -> getNumberEdges());
	}
	
	/**
	* Sets the height of the drawing bar area
	* param _h The height to set
	*/
	void pColorBar ::  setHeight (int _h)
	{
		height = _h;
		bar_height = height-gapY;
	}

	/**
	* Sets the height of the drawing bar area
	* return _h The height to get
	*/
	int pColorBar :: getHeight()
	{
		 return height;
	}

	/**
	* Sets the width of the drawing bar area
	* param _w The width to set
	*/
	void pColorBar ::  setWidth (int _w)
	{
		width = _w;
		bar_width = width-gapX;
	}

	/**
	* Gets the width of the drawing bar area
	* param width The width to get
	*/
	int pColorBar :: getWidth ()
	{
		return width;			
	}

	/**
	* Sets the height of the containing rectangle bar
	* param _h The height to set
	*/
	void pColorBar ::  setBarHeight (int _h)
	{
		bar_height = _h;
	}

	/**
	* Gets the height of the containing rectangle bar
	* return bar_height The height to get
	*/
	int pColorBar :: getBarHeight()
	{
		return bar_height;
	}

	/**
	* Sets the width of the containing rectangle bar
	* param _w The width to set
	*/
	void pColorBar ::  setBarWidth (int _w)
	{
		bar_width = _w;
	}

	/**
	* Gets the width of the containing rectangle bar
	* return bar_width The width to get
	*/
	int pColorBar :: getBarWidth ()
	{
		return bar_width;
	}

	/**
	* Sets the orientation of the color bar
	* param _orientation The orientation to set VERTICAL = false, HORIZONTAL = true
	*/
	void pColorBar ::  setOrientation (bool _orientation)
	{
		_logicalBar -> setOrientation(_orientation);
		figure -> setBarOrientation (_orientation);
	}

	/**
	* Gets the orientation of the color bar
	* return bar_orientation The bar orientation assigned
	*/
	bool pColorBar :: getOrientation ()
	{
		return _logicalBar->getOrientation();
	}

	/**
	* Sets the collection of color points
	* param _points The new points to set, each one of data type pColorPoint ( xValue, wxColour_assigned)
	*/
	void pColorBar ::  setPoints (std::vector<pColorPoint *> _points)
	{
		_logicalBar->clearPoints();
		showedColorPoints.clear();
		_logicalBar->setColorPoints(_points);
		//posible needed to update
	}

	/**
	* Gets the last clickedX pixel coord inside the bar.
	* return clickedX The x-coord pixel value
	*/
	int pColorBar :: getClickedX()
	{
		return clickedX;
	}

	/**
	* Sets the last clickedX pixel coord inside the bar.
	* param xCoordPixel The x-coord value to set
	*/
	void pColorBar :: setClickedX(int xCoordPixel)
	{
		clickedX = xCoordPixel;
	}

	/**
	* Gets the last clickedY pixel coord inside the bar.
	* return clickedY The y-coord pixel value
	*/
	int pColorBar :: getClickedY()
	{
		return clickedY;
	}

	/**
	* Sets the last clickedY pixel coord inside the bar.
	* param yCoordPixel The y-coord pixel value to set
	*/
	void pColorBar :: setClickedY(int yCoordPixel)
	{
		clickedY = yCoordPixel;
	}

	/**
	* Gets the real x value for a given x-pixel value using the rule real_x = minX_represented_Tshow + ((x_Pixel * (maxX_represented_Tshow - minX_represented_Tshow) ) / bar_width)
	* param x_Pixel The pixel value to convert into real value with respect to the x scale
	* return realX The real-x value corresponding to the xPixel
	*/
	double pColorBar :: convertToRealValue ( int x_Pixel )
	{
		/**
		int newPixel = x_Pixel*(bar_width - deviceEndMargin - deviceStart_x)/(bar_width) ;
		return newPixel*(maxX_represented_Tshow - minX_represented_Tshow)/(bar_width);
*/
		return minX_represented_Tshow + ((x_Pixel-deviceStart_x) * (maxX_represented_Tshow - minX_represented_Tshow) ) /( bar_width-deviceEndMargin);		
	}

	/**
	* Gets the x-pixel value for a given x_real value using the rule x_pixel = ((x_real - minX_represented_Tshow) * bar_width)/(maxX_represented_Tshow - minX_represented_Tshow)
	* param x_Pixel The pixel value to convert into real value with respect to the x scale
	* return realX The real-x value corresponding to the xPixel
	*/
	int pColorBar :: convertToPixelValue (double x_real )
	{
		return (int)( ((x_real - minX_represented_Tshow) * (bar_width-deviceEndMargin))/(maxX_represented_Tshow - minX_represented_Tshow) );		
	}

	/**
	* Gets the selected color and updates the state of the okSelectedColor
	* return selectedColor Is the selected rbg color
	*/
	wxColour pColorBar :: getSelectedColour()
	{
		okSelectedColor = false;
		wxColour col;
		wxColourData data;
		wxColourDialog dialog( GetParent(), &data);

		if ( dialog.ShowModal() == wxID_OK )
		{
			col = dialog.GetColourData().GetColour();
			okSelectedColor = true;
		}
		return col;
	}

	/**
	* Sets the represented minimum and maximunm values
	* param minRealValue The minimum represented value (real value)
	* param maxRealValue The maximum represented value (real value)
	*/
	void pColorBar :: setRepresentedValues ( double minRealValue, double maxRealValue )
	{
		temporalMinXToShow = (int)minRealValue;
		temporalMaxXToShow = (int)maxRealValue;

		minX_represented_Tshow = (int)minRealValue;		
		maxX_represented_Tshow = (int)maxRealValue;
		_logicalBar->setMinValue(minX_represented_Tshow);
		_logicalBar->setMaxValue(maxX_represented_Tshow);
	}

	/**
	* Gets the data of the last point moving in a pColorPoint copy
	* return pointData Is a pColorPoint with the data of las moved color
	*/
	pColorPoint * pColorBar :: getLastMovedColorPoint()
	{
		pColorPoint * dataPoint = new pColorPoint ( 0, wxColor (0,0,0), false);
		if (movingNodePoint)
		{		
			dataPoint ->setColor(movingNodePoint->getColor());
			dataPoint ->setRealX(movingNodePoint->getRealX());
		}
		return dataPoint;
	}

	/**
	* Sets the gap values for the color bar bitmap
	* param gap_x Gap in x
	* param gap_y Gap in y
	*/
	void pColorBar :: setGapValues (int gap_x, int gap_y)
	{
		gapX = gap_x;
		gapY = gap_y;
	}

	/**
	* Sets the degrade state of the color bar
	* param newState The degrade stare to set 
	*/
	void pColorBar :: setDegradeState(bool newState)
	{
		doingDegrade = newState;
	}

	/**
	* Gets the degrade state of the color bar
	* return doingDegrade is the actual degrade state of the bar
	*/
	bool pColorBar :: getDegradeState()
	{
		return doingDegrade;
	}

	/**
	* Sets the visible range of the bar and repaints the bar according to it, the min value must be less than the max value.
	* param minToShow Is the minimum value to show in the colorbar
	* param maxToShow Is the maximum value to show in the colorbar
	*/
	void pColorBar :: setVisibleRange(int minToShow, int maxToShow)
	{
		minX_represented_Tshow = minToShow;
		maxX_represented_Tshow = maxToShow; 
		viewingRange = true;
		RefreshForce();
	}

	/**
	* Sets the state of static or not for the limit color points
	* pamar areStatic Is the state to set for the limits
	*/
	void pColorBar :: setStaticLimitsTo(bool areStatic)
	{
		staticLimits = areStatic;
	}

	/**
	* Gets the state of static or not for the limit color points
	* return staticLimits Is the state for limits
	*/
	bool pColorBar :: getStaticLimits()
	{
		return staticLimits;
	}

	/**
	* Sets the device start drawing left-superior (pixel) start point and draws automatically the color bar
	* param deviceStart_x Pixel start for x-coord
	* param deviceStart_y Pixel start for y-coord
	*/
	void pColorBar :: setDeviceBlitStart ( wxCoord devStart_x, wxCoord devStart_y )
	{
		deviceStart_x = devStart_x;
		deviceStart_y = devStart_y;
		if (getOrientation())
			width += deviceStart_x;
		else
			height += deviceStart_y;
		drawColorBar(true);
	}

	/**
	* Clears the temporal color points of the list
	*/
	void pColorBar :: clearTemporalColors()
	{
		
	}

	/**
	* Set active state 
	* param activeNow The new state
	*/
	void pColorBar :: setActiveStateTo (bool activeNow)
	{
		activeState = activeNow;
	}
	
	/**
	* Gets the active state of the bar
	*  return isActive The actual state
	*/
	bool pColorBar :: isActive()
	{
		return activeState;
	}

	/**
	* Gets the real-x value to draw a vertical line
	* return realX_vertical_line The real x value for the vertical line
	*/
	int	pColorBar :: getRealX_vertical_line()
	{
		return realX_vertical_line;
	}

	/**
	* Sets the real-x value to draw a vertical line
	* param realX_vertical_line The new real x value for the vertical line
	*/
	void pColorBar :: setRealX_vertical_line(int newReal_x)
	{
		realX_vertical_line = newReal_x;
	}

	/**
	* Sets the new device (deviceEndMargin) value form the end of this panel to the end of the drawing area in the device
	* param newDeviceEnd_pixels The new pixel value to asign to the right(horizontal view), underneath(vertical view) margin in pixels
	*/
	void pColorBar :: setDeviceEndMargin(int newDeviceEnd_pixels)
	{
		deviceEndMargin = newDeviceEnd_pixels;
	}

	void pColorBar :: createAndSendEvent(WXTYPE theEventType)
	{
		wxCommandEvent cevent( theEventType, GetId() );
		cevent.SetEventObject( this );
		GetEventHandler()->ProcessEvent( cevent );
	}

	
	/**
		Returns the number of points that the bar color has
	*/
	int pColorBar::getColorPointsSize()
	{
		return _logicalBar->getCount();
	}

	/*
	 Get the RGB values of the color point that is in the
	 index given with respecto to the
	*/
	void pColorBar::getDataAt(int index, double& x,int& red,int& green,int& blue)
	{
		//******************************
		_logicalBar->getDataAt(index, x, red, green, blue);
	}

	/*
	* Sets the guide line color
	* param theNwGuideLineColor The color to set to the guideLineColor
	*/
	void pColorBar :: setGuideLineColour(wxColour theNwGuideLineColor)
	{
		guideLineColor = theNwGuideLineColor;
	}

	/*
	* Gets the guide line color
	* return guideLineColor The color of the guideLine
	*/
	wxColour pColorBar :: getGuideLineColour()
	{
		return guideLineColor;
	}

	/*
	* Sets the guide line color
	* param theNwGuideLineColor The color to set to the guideLineColor
	*/
	void pColorBar :: setBackGroundColour(wxColour theNwBackColor)
	{
		colourParent = theNwBackColor;
	}

	/*
	* Gets the guide line color
	* return guideLineColor The color of the guideLine
	*/
	wxColour pColorBar :: getBackGroundColour()
	{
		return colourParent;
	}

	/*
	* Gets the min value of the color bar
	*/
	double	pColorBar :: getMinValue()
	{
		return (double)minX_represented_Tshow;
	}

	/*
	* Gets the max value of the color bar
	*/
	double pColorBar :: getMaxValue()
	{
		return (double)maxX_represented_Tshow;
	}


	void pColorBar :: updatePointsToDraw()
	{		
		accumTemporalPoints = 0;		
		if (_logicalBar->getCount()>0)
		{
			showedColorPoints.clear();
			
			int startIndex, endIndex;
			_logicalBar -> getPointersToRangeLimits( showedColorPoints, startIndex, endIndex, minX_represented_Tshow, maxX_represented_Tshow );
		
			int internalCount = _logicalBar->getCount();						
			std::deque <pColorPoint*>::iterator iterStart = showedColorPoints.begin( );
			std::deque <pColorPoint*>::iterator iterEnd = showedColorPoints.end( );
			
			int pixelTmpStart = convertToPixelValue (minX_represented_Tshow);
			int pixelTmpEnd = convertToPixelValue (maxX_represented_Tshow);
			int initial_r, final_r, initial_g, final_g, initial_b, final_b, rectangle_width;
			double realINI, realEND;

			bool includeTempStart = false;
			bool includeTempEnd = false;
			//-----------------Adding the first visible point of the list
			if(showedColorPoints.size()>0)
			{
				includeTempStart = (int)(showedColorPoints.front()->getRealX())>minX_represented_Tshow ;									
				includeTempEnd = (int)(showedColorPoints.back()->getRealX())<maxX_represented_Tshow ;									
				if( startIndex >= 0 )
				{
					if( startIndex == 0 )
					{
						//The temporal showing point for the start should have the same color of the initial point in range at startIndex
						_logicalBar -> getDataAt( startIndex, realINI, initial_r, initial_g, initial_b );
						realINI = minX_represented_Tshow;
						_logicalBar -> getDataAt( startIndex, realEND, final_r, final_g, final_b );					
					}
					else if( startIndex > 0 )
					{
						_logicalBar -> getDataAt( startIndex-1, realINI, initial_r, initial_g, initial_b );
						_logicalBar -> getDataAt( startIndex, realEND, final_r, final_g, final_b );
					}
					
					if( includeTempStart )
					{
						int ini_pixelX = convertToPixelValue( realINI );
						int end_pixelX = convertToPixelValue( realEND );

						rectangle_width = end_pixelX - ini_pixelX;						
						
						float m_scope_r = (float)(final_r-initial_r)/rectangle_width;
						float m_scope_g = (float)(final_g-initial_g)/rectangle_width;
						float m_scope_b = (float)(final_b-initial_b)/rectangle_width;		

						
						float bRed = initial_r - m_scope_r*pixelTmpStart;
						float bGreen = initial_g -  m_scope_g*pixelTmpStart;
						float bBlue = initial_b - m_scope_b*pixelTmpStart;

						showedColorPoints.push_front( new pColorPoint ( minX_represented_Tshow, wxColour( (m_scope_r*pixelTmpStart)+bRed, (m_scope_g*pixelTmpStart)+bGreen, (m_scope_b*pixelTmpStart)+bBlue ), true));				
					
						accumTemporalPoints ++;	
					}
				}				
				if( includeTempEnd && minX_represented_Tshow < maxX_represented_Tshow )
				{	
					if( internalCount>0 )
					{
						if( endIndex == internalCount-1 )
						{
							_logicalBar -> getDataAt( endIndex, realINI, initial_r, initial_g, initial_b );
							_logicalBar -> getDataAt( endIndex, realEND, final_r, final_g, final_b );
							realEND = maxX_represented_Tshow;
						}						
						else
						{
							_logicalBar -> getDataAt( endIndex, realINI, initial_r, initial_g, initial_b );
							_logicalBar -> getDataAt( endIndex+1, realEND, final_r, final_g, final_b );							
						}					

						int ini_pixelX = convertToPixelValue( realINI );
						int end_pixelX = convertToPixelValue( realEND );

						rectangle_width = end_pixelX - ini_pixelX;						
						
						float m_scope_r = (float)(final_r-initial_r)/rectangle_width;
						float m_scope_g = (float)(final_g-initial_g)/rectangle_width;
						float m_scope_b = (float)(final_b-initial_b)/rectangle_width;		
						
						float bRed = initial_r - m_scope_r*pixelTmpEnd;
						float bGreen = initial_g -  m_scope_g*pixelTmpEnd;
						float bBlue = initial_b - m_scope_b*pixelTmpEnd;
						showedColorPoints.push_back( new pColorPoint ( maxX_represented_Tshow, wxColour( ( m_scope_r*pixelTmpEnd)+bRed, (m_scope_g*pixelTmpEnd)+bGreen, (m_scope_b*pixelTmpEnd)+bBlue ), true));				
						accumTemporalPoints ++;				
					}					
				}
			}
			else
			{
				if(_logicalBar->getCount()>0)
				{
					if ( minX_represented_Tshow > _logicalBar->getMaxAddedValue() )
					{
						endIndex = _logicalBar->getCount()-1;
					}
					else if ( maxX_represented_Tshow < _logicalBar->getMinAddedValue() )
					{
						endIndex = 0;
					}

					_logicalBar -> getDataAt( endIndex, realINI, initial_r, initial_g, initial_b );
					_logicalBar -> getDataAt( endIndex, realEND, final_r, final_g, final_b );
					
					realINI = minX_represented_Tshow;
					realEND = maxX_represented_Tshow;

					showedColorPoints.push_back( new pColorPoint ( realINI, wxColour( initial_r, initial_g, initial_b ), true));	
					accumTemporalPoints ++;		
					showedColorPoints.push_back( new pColorPoint ( realEND, wxColour( initial_r, initial_g, initial_b ), true));	
					accumTemporalPoints ++;		
				}
			}
		}		
			
		if ( !showedColorPoints.empty() )
		{
			startNode_show = showedColorPoints.front();
			lastNode_show = showedColorPoints.back();
		}
	}

