/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


#include "vtkClipping3DDataViewer.h"
#include "vtkDataSetReader.h"
#include "vtkProperty.h"

vtkClipping3DDataViewer::vtkClipping3DDataViewer()
{
	int i;
	for (i=0; i<VTKMPRDATA_MAXTISSUE; i++)
	{
		_visibleTissue[i]		= false;
		_representationType[i]	= true;
	}
	_visibleVolume				= false;
	_isRayCasting				= true;
    _isMIP						= false;
	_interpolation				= false;
	_shade						= false;

    _volumerendererdata 	= VolumeRendererData::New();

#if (VTK_MAJOR_VERSION <= 7) 
    _volumeMapper 			= vtkVolumeRayCastMapper::New();
    _compositeFunction 		= vtkVolumeRayCastCompositeFunction::New();
	_compositeFunctionMIP 	= vtkVolumeRayCastMIPFunction::New();
#else 
    _volumeMapper 			= vtkFixedPointVolumeRayCastMapper::New();
#endif

    _newvol 				= vtkVolume::New();
    _volumeProperty 		= vtkVolumeProperty::New();

}
//-------------------------------------------------------------------
vtkClipping3DDataViewer::~vtkClipping3DDataViewer()
{
  	int i;
	for (i=0; i< VTKMPRDATA_MAXTISSUE ; i++)
	{
		_mCubes[i]->Delete();
		_tissueStripper[i]->Delete();
		_tissueMapper[i]->Delete();
		_tissuePlanes[i]->Delete();
		_tissueClipper[i]->Delete();
	} // for i

	// Volume
	_tfun->Delete();
	_ctfun->Delete();
#if (VTK_MAJOR_VERSION <= 7) 
	_compositeFunction->Delete();
#else 
	// ..
#endif
	_volumeMapper->Delete();
	_volumeProperty->Delete();
	_newvol->Delete();
	_volumePlanes->Delete();

	for (i=0; i< VTKMPRDATA_MAXTISSUE ; i++){
		_tissue[i]-> Delete();
		_observerS[i]-> Delete();
	}
	_observerV-> Delete();
}

//-------------------------------------------------------------------
vtkClipPolyData* vtkClipping3DDataViewer::GetTissueClipper(int id)
{
	return this->_tissueClipper[id];
}
//-------------------------------------------------------------------
vtkPolyDataMapper* vtkClipping3DDataViewer::GetTissueMapper(int id)
{
	return _tissueMapper[id];
}
//-------------------------------------------------------------------
vtkPlanes* vtkClipping3DDataViewer::GetTissuePlanes(int id)
{
	return _tissuePlanes[id];
}
//-------------------------------------------------------------------
vtkStripper* vtkClipping3DDataViewer::GetTissueStripper(int id)
{
	return _tissueStripper[id];
}
//-------------------------------------------------------------------
void vtkClipping3DDataViewer::Refresh()
{
//	_volumeMapper->SetClippingPlanes(_volumePlanes);

//	this->_volumeMapper->Update();
//	this->_newvol->Update();

//	_newvol->VisibilityOn();


// _volumeMapper = vtkVolumeRayCastMapper::New();
//	_volumeMapper->SetInput( this->GetVtkMPRBaseData()->GetImageData() );
//	_volumeMapper->SetVolumeRayCastFunction(_compositeFunction);
//	_volumeMapper->SetClippingPlanes( _volumePlanes );
//	_newvol->SetMapper(_volumeMapper );


}
//-------------------------------------------------------------------
void vtkClipping3DDataViewer::RefreshSurface()
{
	int i;
	for (i=0; i< VTKMPRDATA_MAXTISSUE ; i++)
	{
		SetIsovalue(i, (int) (GetIsovalue(i)+1) );
//		this->GetMCubes(i)->Update();
	}
}
//-------------------------------------------------------------------
void vtkClipping3DDataViewer::SetVtkMPRBaseData(vtkMPRBaseData *vtkmprbasedata)
{
	_vtkmprbasedata=vtkmprbasedata;
}
//-------------------------------------------------------------------
vtkMPRBaseData* vtkClipping3DDataViewer::GetVtkMPRBaseData()
{
	return _vtkmprbasedata;
}


//-------------------------------------------------------------------
void vtkClipping3DDataViewer::SetVisibleTissue(int idTissue, bool visible)
{
	_visibleTissue[idTissue]=visible;
}
//-------------------------------------------------------------------
bool vtkClipping3DDataViewer::GetVisibleTissue(int idTissue){
	return _visibleTissue[idTissue];
}
//-------------------------------------------------------------------
void vtkClipping3DDataViewer::SetRepresentationType(int idTissue, bool representationType)
{
	_representationType[idTissue]=representationType;
}
//-------------------------------------------------------------------
bool vtkClipping3DDataViewer::GetRepresentationType(int idTissue)
{
	return _representationType[idTissue];
}

//-------------------------------------------------------------------
vtkActor* vtkClipping3DDataViewer::GetTissueActor(int id){
	return _tissue[id];
}

//-------------------------------------------------------------------
boxSurfaceObserver *vtkClipping3DDataViewer::GetObserverS(int idObserverS)
{
	return _observerS[idObserverS];
}
//-------------------------------------------------------------------
boxSurfaceObserver *vtkClipping3DDataViewer::GetObserverV()
{
	return _observerV;
}

//-------------------------------------------------------------------
bool vtkClipping3DDataViewer::GetVisibleVolume()
{
	return _visibleVolume;
}
//-------------------------------------------------------------------
void vtkClipping3DDataViewer::SetVisibleVolume(bool visiblevolume)
{
	_visibleVolume = visiblevolume;
}

bool vtkClipping3DDataViewer::GetVisibleVolumeBox()
{
	return _visibleVolumeBox;
}
//-------------------------------------------------------------------
void vtkClipping3DDataViewer::SetVisibleVolumeBox(bool visibleBox)
{
	_visibleVolumeBox = visibleBox;
}


//-------------------------------------------------------------------
void vtkClipping3DDataViewer::SetImage()
{
	int i;
	vtkImageData *imagedata=_vtkmprbasedata->GetImageData();
	for (i=0; i< VTKMPRDATA_MAXTISSUE ; i++)
	{
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		_mCubes[i]->SetInput( imagedata );
#else
		_mCubes[i]->SetInputData( imagedata );
#endif
	} // for
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	_volumeMapper->SetInput( imagedata );
#else
	_volumeMapper->SetInputData( imagedata );
#endif
	_volumerendererdata->SetImageData( imagedata );
}

//-------------------------------------------------------------------
void vtkClipping3DDataViewer::Configure_Tissue()
{
	int i;
	double range[2];
	vtkImageData *imagedata=_vtkmprbasedata->GetImageData();
	imagedata->GetScalarRange( range );
	for (i=0; i< VTKMPRDATA_MAXTISSUE ; i++)
	{
		// Visualisation - original volume
		_mCubes[i] = vtkMarchingCubes::New( );

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		_mCubes[i]->SetInput( imagedata );
#else
		_mCubes[i]->SetInputData( imagedata );
#endif

		_mCubes[i]->SetValue( 0, range[1]*(4+i) / 8 );
//		_mCubes[i]->SetValue( 0, 1500 );
	    _tissueStripper[i] = vtkStripper::New();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    _tissueStripper[i]->SetInput( _mCubes[i]->GetOutput( ) );
#else
	    _tissueStripper[i]->SetInputData( _mCubes[i]->GetOutput( ) );
#endif

		_tissuePlanes[i]  = vtkPlanes::New();
		int x1,x2,y1,y2,z1,z2;
		imagedata->GetExtent(x1,x2,y1,y2,z1,z2);
		_tissuePlanes[i]->SetBounds(x1,x2,y1,y2,z1,z2);
		_tissueClipper[i] = vtkClipPolyData::New();
		_tissueClipper[i]->SetClipFunction( _tissuePlanes[i] );
		_tissueClipper[i]->InsideOutOn( );

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		_tissueClipper[i]->SetInput( _tissueStripper[i]->GetOutput() );
#else
		_tissueClipper[i]->SetInputData( _tissueStripper[i]->GetOutput() );
#endif

		_tissueMapper[i] = vtkPolyDataMapper::New( );

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		_tissueMapper[i]->SetInput( _tissueClipper[i]->GetOutput() );
#else
		_mCubes[i]->Update();
		_tissueStripper[i]->Update();
		_tissueClipper[i]->Update();
		_tissueMapper[i]->SetInputData( _tissueClipper[i]->GetOutput() );
		_tissueMapper[i]->Update();
#endif


		_tissueMapper[i]->ScalarVisibilityOff( );
//		_tissueMapper[i]->Update();
	} // for i  

	// vtkActor tissue
	for (i=0; i< VTKMPRDATA_MAXTISSUE ; i++){
		// tissue
		_tissue[i] = vtkActor::New();
		_tissue[i]->SetMapper( GetTissueMapper(i) );
		float cr=1,cg=0.5,cb=0.5;
		_tissue[i]->GetProperty()->SetDiffuseColor(cr/255, cg/255   , cb/255   );
		_tissue[i]->GetProperty()->SetSpecular(.3);
		_tissue[i]->GetProperty()->SetSpecularPower(20);
		_tissue[i]->GetProperty()->SetOpacity(0.5);
		if (i==0) { _tissue[i]->GetProperty()->SetColor(0.85, 0.85   , 0.85   ); 	}
		if (i==1) { _tissue[i]->GetProperty()->SetColor(0, 0  ,  1  ); 				}
		if (i==2) { _tissue[i]->GetProperty()->SetColor(0.85, 0.20   , 0.20   );	}
		if (i==3) { _tissue[i]->GetProperty()->SetColor(0, 1   , 0   );				}
	} // for

  	for (i=0; i< VTKMPRDATA_MAXTISSUE ; i++)
	{
		_observerS[i] = boxSurfaceObserver::New();
		_observerS[i]->SetPlanes( GetTissuePlanes(i) );
		_observerS[i]->SetActor( _tissue[i] );
		_observerS[i]->SetFilters( _mCubes[i], _tissueStripper[i] ,_tissueClipper[i] );
	} // for
}

//-----------------------------------------------------------------------------

void vtkClipping3DDataViewer::Configure_Volume()
{
  // Volume

 /** if(_newvol != NULL)
  {
    _newvol->Delete();
    _volumeMapper->Delete();
    _volumeProperty->Delete();
  }*/

  _tfun = vtkPiecewiseFunction::New();
  _ctfun = vtkColorTransferFunction::New();

	double range[2];
	this->_vtkmprbasedata->GetImageData()->GetScalarRange(range);
	double max = range[1];
	
	/*
	adding the poinst of the transference function
	*/
	//X
	greyValuesTransferenceFVector.push_back(max * 0/2);  // JPR ?? 0/2
	greyValuesTransferenceFVector.push_back(max * 1/2);  // JPR ?? division entiere?
	greyValuesTransferenceFVector.push_back(max * 2/2);
	//Y
	intensityValuesTransferenceFVector.push_back(0.0);
	intensityValuesTransferenceFVector.push_back(1.0);
	intensityValuesTransferenceFVector.push_back(1.0);

	_tfun->AddPoint(max * 0/2 , 0.0);
	_tfun->AddPoint(max * 1/2 , 1.0);
	_tfun->AddPoint(max * 2/2 , 1.0);

	/*
		Adding the colors to the vectors
	*/
	//RED
	redColorsOfColorTransferenceFVector.push_back(0.0);
	redColorsOfColorTransferenceFVector.push_back(1.0);
	redColorsOfColorTransferenceFVector.push_back(0.0);
	redColorsOfColorTransferenceFVector.push_back(0.0);
	redColorsOfColorTransferenceFVector.push_back(0.0);

	//GREEN
	greenColorsOfColorTransferenceFVector.push_back(0.0);
	greenColorsOfColorTransferenceFVector.push_back(0.0);
	greenColorsOfColorTransferenceFVector.push_back(0.0);
	greenColorsOfColorTransferenceFVector.push_back(1.0);
	greenColorsOfColorTransferenceFVector.push_back(0.2);

	//BLUE
	blueColorsOfColorTransferenceFVector.push_back(0.0);
	blueColorsOfColorTransferenceFVector.push_back(0.0);
	blueColorsOfColorTransferenceFVector.push_back(1.0);
	blueColorsOfColorTransferenceFVector.push_back(0.0);
	blueColorsOfColorTransferenceFVector.push_back(0.0);
	//GREY VALUE
	greyValueColorsOfColorTransferenceFVector.push_back(max*0/4);
	greyValueColorsOfColorTransferenceFVector.push_back(max*1/4);
	greyValueColorsOfColorTransferenceFVector.push_back(max*2/4);
	greyValueColorsOfColorTransferenceFVector.push_back(max*3/4);
	greyValueColorsOfColorTransferenceFVector.push_back(max*4/4);

	_ctfun->AddRGBPoint( max*0/4 , 0.0, 0.0, 0.0);
	_ctfun->AddRGBPoint( max*1/4 , 1.0, 0.0, 0.0);
	_ctfun->AddRGBPoint( max*2/4 , 0.0, 0.0, 1.0);
	_ctfun->AddRGBPoint( max*3/4 , 0.0, 1.0, 0.0);
	_ctfun->AddRGBPoint( max*4/4 , 0.0, 0.2, 0.0);

  _volumePlanes  = vtkPlanes::New();
//	int x1,x2,y1,y2,z1,z2;
//	vtkImageData *imagedata=_vtkmprbasedata->GetImageData();
//	imagedata->GetExtent(x1,x2,y1,y2,z1,z2);
//	_volumePlanes->SetBounds(x1,x2,y1,y2,z1,z2);

	// EED 13/03/2011

 // _volumeMapper = vtkVolumeRayCastMapper::New();


//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	_volumeMapper->SetInput( this->GetVtkMPRBaseData()->GetImageData() );
#else
	_volumeMapper->SetInputData( this->GetVtkMPRBaseData()->GetImageData() );
#endif


#if (VTK_MAJOR_VERSION <= 7) 
    _volumeMapper->SetVolumeRayCastFunction(_compositeFunction);
	// _volumeMapper->SetVolumeRayCastFunction(_compositeFunctionMIP);
#else 
	// ..
#endif
	
//
	_volumeMapper->SetClippingPlanes( _volumePlanes );
	_volumeMapper->AutoAdjustSampleDistancesOn();

    _volumeProperty->SetInterpolationTypeToNearest();
    //_volumeProperty = vtkVolumeProperty::New();
	_volumeProperty->SetColor(_ctfun);
	_volumeProperty->SetScalarOpacity( _tfun );

// EED 31/03/2008
	_volumeProperty->DisableGradientOpacityOn();


//	_volumeProperty->SetInterpolationTypeToNearest();
//	_volumeProperty->ShadeOff();
//	_volumeProperty->SetAmbient(0.3);
//	_volumeProperty->SetDiffuse(0.1);
//	_volumeProperty->SetSpecular(0.8);
//	_volumeProperty->DisableGradientOpacityOn();

 // _newvol = vtkVolume::New();
	_newvol->SetMapper(_volumeMapper );
	_newvol->SetProperty(_volumeProperty );

    _observerV = boxSurfaceObserver::New();
	_observerV->SetPlanes( _volumePlanes );
	_observerV->SetActor( _newvol );
	
#if (VTK_MAJOR_VERSION <= 7) 
	_observerV->SetvtkVolumeRayCastMapper( _volumeMapper );
#else 
	_observerV->SetvtkFixedPointVolumeRayCastMapper( _volumeMapper );
#endif
}

//-----------------------------------------------------------------------------

void vtkClipping3DDataViewer::Configure_VolumeBox()
{
	_volumerendererdata->SetImageData(_vtkmprbasedata->GetImageData());


    vector<double> *gv = this->GetGreyValuesTransferenceFVector();
    vector<double> *op = this->GetIntensityValuesTransferenceFVector();

    _volumerendererdata->SetLookUpTable((vtkLookupTable*)(_ctfun));
    _volumerendererdata->OpacityChanged(*gv, *op);

    //_wxvtkclipping3Dview->VisibleVolumeActor( false );
    //_volumerendererdata->ShowVolume(event.IsChecked() );
       //_volumerendererdata->BoundingBoxChanged(event.IsChecked() );
}

//-----------------------------------------------------------------------------

void vtkClipping3DDataViewer::SetRenderer(vtkRenderer* renderer)
{
    _volumerendererdata->SetRenderer(renderer);

}

//-----------------------------------------------------------------------------
void vtkClipping3DDataViewer::SetInteractor(vtkRenderWindowInteractor* interactor)
{

    _volumerendererdata->SetInteractor(interactor);
}


//-----------------------------------------------------------------------------
void vtkClipping3DDataViewer::BoxActorChanged(bool changed)
{
    vector<double> *gv = this->GetGreyValuesTransferenceFVector();
    vector<double> *op = this->GetIntensityValuesTransferenceFVector();

   _volumerendererdata->ShowVolume(changed);
    _volumerendererdata->OpacityChanged(*gv, *op);
    _volumerendererdata->BoundingBoxChanged(changed);

}

//-----------------------------------------------------------------------------
void vtkClipping3DDataViewer::UpdateVolumeBox(vector<double> gf, vector<double> vf, vtkColorTransferFunction* ctfun)
{
        _volumerendererdata->SetLookUpTable((vtkLookupTable*)(ctfun));
        _volumerendererdata->OpacityChanged(gf, vf);
}


//-----------------------------------------------------------------------------
void vtkClipping3DDataViewer::updateVolume()
{

#if (VTK_MAJOR_VERSION <= 7) 
    if(_isRayCasting)
    {
        _volumeMapper->SetVolumeRayCastFunction(_compositeFunction);
        _volumerendererdata->changeCompositeMIPFunction(0);
    }    else if(_isMIP) {
        _volumeMapper->SetVolumeRayCastFunction(_compositeFunctionMIP);
        _volumerendererdata->changeCompositeMIPFunction(1);
    }
#else 
	// ...   here Composition of MIP
	printf("EED vtkClipping3DDataViewer::updateVolume ...   here Composition or MIP \n");
#endif


    if(_interpolation)
    {
        _volumeProperty->SetInterpolationTypeToLinear();
    } else {
        _volumeProperty->SetInterpolationTypeToNearest();
    } // if _interpolation

    if(_shade)
    {
        _volumeProperty->ShadeOn();
    } else {
        _volumeProperty->ShadeOff();
    } // if _shade

}

//-------------------------------------------------------------------
void vtkClipping3DDataViewer::Configure()
{
	Configure_Tissue();
	Configure_Volume();
    Configure_VolumeBox();
}


//-------------------------------------------------------------------
void vtkClipping3DDataViewer::SetIsovalue(int idTissue, int isoValue)
{
	_mCubes[idTissue]->SetValue(0, isoValue);
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	// ..
#else
	_mCubes[idTissue]->Update();
	_tissueStripper[idTissue]->Update();
	_tissueClipper[idTissue]->Update();
#endif
}
//-------------------------------------------------------------------
double vtkClipping3DDataViewer::GetIsovalue(int idTissue)
{
	return _mCubes[idTissue]->GetValue(0);
}
//-------------------------------------------------------------------
vtkVolume* vtkClipping3DDataViewer::GetVolumeActor()
{
   return _newvol;
}
//-------------------------------------------------------------------
#if (VTK_MAJOR_VERSION <= 7) 
	vtkVolumeRayCastMapper* vtkClipping3DDataViewer::GetVolumeMapper()
#else 
	vtkFixedPointVolumeRayCastMapper* vtkClipping3DDataViewer::GetVolumeMapper()
#endif
{
	return _volumeMapper;
}
//-------------------------------------------------------------------
vtkPlanes* vtkClipping3DDataViewer::GetVolumePlanes()
{
	return _volumePlanes;
}

//-------------------------------------------------------------------
vtkMarchingCubes *vtkClipping3DDataViewer::GetMCubes(int idTissue)
{
	return _mCubes[idTissue];
}
//--------------------------------------------------------------------
//-------------------
//Getters Vectors
//-------------------

std::vector<double>*	vtkClipping3DDataViewer::GetGreyValuesTransferenceFVector()
{
	return &greyValuesTransferenceFVector;
}
//--------------------------------------------------------------------
std::vector<double>*	vtkClipping3DDataViewer::GetIntensityValuesTransferenceFVector()
{
	return &intensityValuesTransferenceFVector;
}
//--------------------------------------------------------------------
std::vector<double>*	vtkClipping3DDataViewer::GetRedColorsOfColorTransferenceFVector()
{
	return &redColorsOfColorTransferenceFVector;
}
//--------------------------------------------------------------------
std::vector<double>*	vtkClipping3DDataViewer::GetGreenColorsOfColorTransferenceFVector()
{
	return &greenColorsOfColorTransferenceFVector;
}
//--------------------------------------------------------------------
std::vector<double>*	vtkClipping3DDataViewer::GetBlueColorsOfColorTransferenceFVector()
{
	return &blueColorsOfColorTransferenceFVector;
}
//--------------------------------------------------------------------
std::vector<double>*	vtkClipping3DDataViewer::GetGreyValueColorsOfColorTransferenceFVector()
{
	return &greyValueColorsOfColorTransferenceFVector;
}
//--------------------------------------------------------------------
//--------------------------------------
//Getters transference function
//and color of the transference function
//---------------------------------------
vtkPiecewiseFunction* vtkClipping3DDataViewer::GetTransferencefunction()
{
	return this->_tfun;
}
//--------------------------------------------------------------------
vtkColorTransferFunction* vtkClipping3DDataViewer::GetColorTransferenceFunction()
{
	return this->_ctfun;
}

//-------------------------------------------------------------------

void vtkClipping3DDataViewer::ReadVolumeFunctions()
{
/*
	int i=0,xi,yi,r,g,b,gValue;
	vtkImageData *imagedata	= this->_vtkmprbasedata->GetImageData();

	HistogramDialog* hDlg=new HistogramDialog(NULL,_T("Histogram Dialog"),imagedata);
	//
	// put in a method
	//
	int tfSize=this->greyValuesTransferenceFVector.size();
		if(tfSize>0)
		{
			int i=0,y;
			hDlg->erasePointsTransferenceFunction();
			while(i<tfSize)
			{
				double g=greyValuesTransferenceFVector[i];
				double in=intensityValuesTransferenceFVector[i];
				hDlg->addPointToTransferenceFunction(g,in*100);
				i++;
			}

		}

	int ctfSize=this->redColorsOfColorTransferenceFVector.size();
	if(ctfSize>0)
	{
		int i=0,y;
			while(i<ctfSize)
			{
				double gr=greyValueColorsOfColorTransferenceFVector[i];
				double r=redColorsOfColorTransferenceFVector[i];
				double g=greenColorsOfColorTransferenceFVector[i];
				double b=blueColorsOfColorTransferenceFVector[i];
				hDlg->addColorPoint(gr,r*255,g*255,b*255);
				i++;
			}
	}
	//If it is smooth activate next line
	//hDlg->updatePlotter();
	//setting variables if the user wants to do refresh
	hDlg->setCTF(_ctfun);
	hDlg->setTF(_tfun);

	//
	// when the user had changed the transference Function
	//
	if(hDlg->ShowModal()== wxID_OK )
	{
			// -- vtkPiecewiseFunction --
			this->_tfun->RemoveAllPoints();
			greyValuesTransferenceFVector.clear();
			intensityValuesTransferenceFVector.clear();

			int nTFPoints=hDlg->getSizeTransferenceFunction();
			i=0;
			while(i<nTFPoints)
			{
				hDlg->getTransferenceFunctionPoint(i,xi,yi);
				this->_tfun->AddPoint( xi , yi/100.0 );
				greyValuesTransferenceFVector.push_back(xi);
				intensityValuesTransferenceFVector.push_back(yi/100.0);
				i++;
			}
			// -- vtkColorTransferFunction  --
			this->_ctfun->RemoveAllPoints ();
			//clean colors
			redColorsOfColorTransferenceFVector.clear();
			greenColorsOfColorTransferenceFVector.clear();
			blueColorsOfColorTransferenceFVector.clear();
			greyValueColorsOfColorTransferenceFVector.clear();

			int nCTFpoints=hDlg->getSizeBarColor();
			i=0;
			while(i<nCTFpoints)
			{
				hDlg->getDataBarColorPoint(i,xi,r,g,b);
				this->_ctfun->AddRGBPoint(xi,r/255.0,g/255.0,b/255.0 );
				redColorsOfColorTransferenceFVector.push_back(r/255.0);
				greenColorsOfColorTransferenceFVector.push_back(g/255.0);
				blueColorsOfColorTransferenceFVector.push_back(b/255.0);
				greyValueColorsOfColorTransferenceFVector.push_back(xi);
				i++;
			}

			this->_volumeMapper->Update();
			this->_newvol->Update();
	}

	else
	{

		if(hDlg->getRefreshed())
		{
			int i=0,size;
			//--Transference Function----
			this->_tfun->RemoveAllPoints();
			i=0;
			size=greyValuesTransferenceFVector.size();
			for(i=0;i<size;i++)
			{
				double grey1=greyValuesTransferenceFVector[i];
				double  in2=intensityValuesTransferenceFVector[i];
				this->_tfun->AddPoint( grey1 , in2 );
			}

			// -- vtkColorTransferFunction  --
			_ctfun->RemoveAllPoints ();

			i=0;
			size=greyValueColorsOfColorTransferenceFVector.size();
			for(i=0;i<size;i++)
			{
				double grey2=(greyValueColorsOfColorTransferenceFVector)[i];
				double red =(redColorsOfColorTransferenceFVector)[i];
				double green =(greenColorsOfColorTransferenceFVector)[i];
				double blue = (blueColorsOfColorTransferenceFVector)[i];
				_ctfun->AddRGBPoint(grey2,red,green,blue);
			}
			this->_volumeMapper->Update();
			this->_newvol->Update();
		 }

	}
	hDlg->Destroy();
*/
}
/*
void vtkClipping3DDataViewer::ReadVolumeFunctions(char *namefile)
{
	char tmp[256];
	short int	max;
	double		range[2];
	double		x,val,r,g,b;

	FILE *ff = fopen ( namefile ,"r");

	// --  MAX  --
	fscanf(ff,"%s",tmp);
	if (strcmp(tmp,"MAX")==0)
	{
		vtkImageData *vtkimagedata		= this->_vtkmprbasedata->GetImageData();
		vtkimagedata->GetScalarRange(range);
		max = (int) (range[1]);
	} else {
		max = atoi(tmp);
	}

	fscanf(ff,"%s",tmp);   // --

	// -- vtkPiecewiseFunction --
	this->_tfun->RemoveAllPoints();
	fscanf(ff,"%s",tmp);
	while ( strcmp(tmp,"--")!=0 )
	{
		x=atof(tmp);
		fscanf(ff,"%s",tmp);
		val=atof(tmp);
		this->_tfun->AddPoint( x*max , val );
		fscanf(ff,"%s",tmp);
	}

	this->_ctfun->RemoveAllPoints ();
	// -- vtkColorTransferFunction  --
	while ( !feof(ff))
	{
		fscanf(ff,"%s",tmp);
		x=atof(tmp);
		fscanf(ff,"%s",tmp);
		r=atof(tmp);
		fscanf(ff,"%s",tmp);
		g=atof(tmp);
		fscanf(ff,"%s",tmp);
		b=atof(tmp);
		this->_ctfun->AddRGBPoint( x*max , r,g,b );
	}


	this->_volumeMapper->Update();
	this->_newvol->Update();


	fclose(ff);
}
*/

//-------------------------------------------------------------------
// EED 23 Mai 2007
void vtkClipping3DDataViewer::ReadMeshVTK(char *namefile)
{
	vtkDataSetReader *reader = vtkDataSetReader::New();
	reader->SetFileName(namefile);
	reader->Update();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	_tissueStripper[3]->SetInput( reader->GetPolyDataOutput()  );
#else
	_tissueStripper[3]->SetInputData( reader->GetPolyDataOutput()  );
#endif

}

void vtkClipping3DDataViewer::setColorTransferFunction(vtkColorTransferFunction* colortable){

	_volumeProperty->SetColor(colortable);

}


void vtkClipping3DDataViewer::SetRayCasting(bool active)
{
    _isRayCasting   =   active;

}

void vtkClipping3DDataViewer::SetMIPActive(bool active)
{
    _isMIP  = active;
}

void vtkClipping3DDataViewer::SetInterpolation(bool active)
{
    _interpolation = active;
}

void vtkClipping3DDataViewer::SetShade(bool active)
{
    _shade  =   active;
}

