/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

// marMatrix.cpp: implementation of the marMatrix class.
//
//////////////////////////////////////////////////////////////////////

#include "marMatrix.h"
#include <assert.h>

//////////////////////////////////////////////////////////////////////
// Construction
//////////////////////////////////////////////////////////////////////
marMatrix::marMatrix(size_t size1, size_t size2)
	:shallowCopy(false)
{
	_size1=size1;
	_size2=size2;
	_data=new double[_size1*_size2];
}

marMatrix::marMatrix(double *data, size_t size1, size_t size2)
	:shallowCopy(true)
{
	if (size1==0 || size2==0)
	{
		assert(false);
	}
	else
	{
		_size1=size1;
		_size2=size2;
		_data=data;

// PS -> 		size_t iMax=_size1*_size2;
// PS -> 		_data=new double[iMax];
// PS -> 
// PS -> 		for (int i=0;i<iMax;i++)
// PS -> 		{
// PS -> 			_data[i]=data[i];
// PS -> 		}

	}
}


marMatrix::marMatrix(const marMatrix &m)
	:shallowCopy(false)
{
	if (m._size1==0 || m._size2==0)
	{
		assert(false);
	}
	else
	{
		_size1=m._size1;
		_size2=m._size2;

		size_t iMax=_size1*_size2;
		_data=new double[iMax];

		int i;
		for (i=0;i<iMax;i++)
		{
			_data[i]=m._data[i];
		}
	}
}

//////////////////////////////////////////////////////////////////////
// Destruction
//////////////////////////////////////////////////////////////////////
marMatrix::~marMatrix()
{
	if (!shallowCopy) 
	{
		delete _data;
	}
	else
	{
		_data=NULL;
	}
}

//////////////////////////////////////////////////////////////////////
// Op�rateurs
//////////////////////////////////////////////////////////////////////
// ---------------------------------------------------------------------
/*Affichage*/
std::ostream& operator<<(std::ostream& os,const marMatrix& m)
{
	int i,j;
	for (i =0;i<m._size1;i++)
	{
		for (j =0;j<m._size2;j++)
		{
			os << " " << m._data[i*m._size2+j];
		}
		os << "\n";
	}
    return(os);
}

// ---------------------------------------------------------------------
/*Casting*/
marMatrix::operator double*() const 
{ 
	return(_data);
}

// ---------------------------------------------------------------------
/*Indexation*/
double& marMatrix::operator()(size_t i, size_t j) 
{ 
	return(_data[i*_size2+j]);
}

const double& marMatrix::operator()(size_t i, size_t j) const
{
	return(_data[i*_size2+j]);
}

// ---------------------------------------------------------------------
/*Affectation*/
marMatrix& marMatrix::operator=(const marMatrix& o)
{
	_size1=o._size1;
	_size2=o._size2;
	if (!(_data==NULL)&&(!shallowCopy))
	{
		delete _data;
	}
	shallowCopy=false;
	size_t iMax=_size1*_size2;
	_data=new double[iMax];

	int i;
	for (i=0;i<iMax;i++)
	{
		_data[i]=o._data[i];
	}
	return (*this);
}

marMatrix& marMatrix::operator=(double o)
{
	if (_size1!=0 && _size2!=0)
	{
		if (!(_data==NULL)&&(!shallowCopy))
		{
			delete _data;
		}
		shallowCopy=false;
		size_t iMax=_size1*_size2;
		_data=new double[iMax];
		for (int i=0;i<iMax;i++)
		{
			_data[i]=o;
		}
	}
	else
	{
		assert(false);
	}
	return (*this);
}

marMatrix& marMatrix::operator=(double*o)
{
	if (_size1!=0 && _size2!=0)
	{
		if (!(_data==NULL)&&(!shallowCopy))
		{
			delete _data;
		}
		shallowCopy=false;
		size_t iMax=_size1*_size2;
		_data=new double[iMax];
		for (int i=0;i<iMax;i++)
		{
			_data[i]=o[i];
		}
	}
	else
	{
		assert(false);
	}
	return (*this);
}

// ---------------------------------------------------------------------
/*Comparaison*/
bool marMatrix::operator==(const marMatrix& o) const
{
    bool equal=true;
	
    if(_size1!=o._size1 || _size2!=o._size2)
	{
		return(false);
	}
	size_t iMax=_size1*_size2;

	for (int i=0;i<iMax && equal;i++)
	{
		equal=equal && (_data[i]==o._data[i]); 
	}
    return(equal);
}

bool marMatrix::operator!=(const marMatrix& o) const
{
	return(!((*this)==o));
}

// ---------------------------------------------------------------------
/*Addition*/
marMatrix marMatrix::operator+(const marMatrix& o)
{
	marMatrix result(*this);
	
	if ((o._size1!=result._size1) 
		|| (result._size1==0) 
		|| (o._size2!=result._size2) 
		|| (result._size2==0))
	{
		assert(false);
	}
	else 
	{
		size_t iMax=result._size1*result._size2;
		int i;
		for (i=0 ; i < iMax ; i++)
		{
			result._data[i]+=o._data[i];
		}
	}
	return result;
}

marMatrix marMatrix::operator+(double o)
{
	marMatrix result(*this);
	
	if ((result._size1==0) || (result._size2==0))
	{
		assert(false);
	}
	else 
	{
		size_t iMax=result._size1*result._size2;
		int i;
		for (i=0;i < iMax;i++)
		{
			result._data[i]+=o;
		}
	}
	return result;
}

marMatrix marMatrix::operator+(double*o)
{
	marMatrix result(*this);
	
	if ((result._size1==0) || (result._size2==0))
	{
		assert(false);
	}
	else 
	{
		size_t iMax=result._size1*result._size2;
		int i;
		for (i=0;i < iMax;i++)
		{
			result._data[i]+=o[i];
		}
	}
	return result;
}

// ---------------------------------------------------------------------
/*Soustraction*/
marMatrix marMatrix::operator-(const marMatrix& o)
{
	marMatrix result(*this);
	
	if ((o._size1!=result._size1) 
		|| (result._size1==0) 
		|| (o._size2!=result._size2) 
		|| (result._size2==0))
	{
		assert(false);
	}
	else 
	{
		size_t iMax=result._size1*result._size2;
		int i;
		for (i=0;i < iMax;i++)
		{
			result._data[i]-=o._data[i];
		}
	}
	return result;
}

marMatrix marMatrix::operator-(double o)
{
	marMatrix result(*this);
	
	if ((result._size1==0) || (result._size2==0))
	{
		assert(false);
	}
	else 
	{
		size_t iMax=result._size1*result._size2;
		int i;
		for (i=0;i < iMax;i++)
		{
			result._data[i]-=o;
		}
	}
	return result;
}

marMatrix marMatrix::operator-(double*o)
{
	marMatrix result(*this);
	
	if ((result._size1==0) || (result._size2==0))
	{
		assert(false);
	}
	else 
	{
		size_t iMax=result._size1*result._size2;
		int i;
		for (i=0;i < iMax;i++)
		{
			result._data[i]-=o[i];
		}
	}
	return result;
}

// ---------------------------------------------------------------------
/*Multiplication (produit scalaire)*/
marMatrix marMatrix::operator*(double o)
{
	marMatrix result(*this);
	
	if ((result._size1==0) || (result._size2==0))
	{
		assert(false);
	}
	else 
	{
		size_t iMax=result._size1*result._size2;
		int i;
		for (i=0;i < iMax;i++)
		{
			result._data[i]*=o;
		}
	}
	return result;
}

marMatrix marMatrix::operator*(const marMatrix &o)
{
	size_t mA=_size1;
	size_t nA=_size2;
	size_t mB=o._size1;
	size_t nB=o._size2;
	size_t mR=mA;
	size_t nR=nB;

	marMatrix result(mR,nR);
	result=0.0;

	int k,i,j;
	if (nA==mB)
	{
		for (k=0;k<nA;k++)
		{
			for (i=0;i<mR;i++)
			{
				for (j=0;j<nR;j++)
				{
					result(i,j)+=(*this)(i,k)*o(k,j);
				}
			}
		}
	}
	else
	{
		assert(false);
	}
	return result;
}

marVector marMatrix::operator*(const marVector &o)
{
	marVector result(_size1);
	if (o.size()!=_size2)
	{
		assert(false);
	}
	else
	{
		marMatrix resultM(_size1,1);
		marMatrix B((double*)o,o.size(),1);
		resultM=(*this)*B;
		result=(double*)resultM;
	}
	return result;
}

//////////////////////////////////////////////////////////////////////
// M�thodes
//////////////////////////////////////////////////////////////////////
size_t marMatrix::rows() const
{
    return _size1; 
}

size_t marMatrix::columns() const
{
	return _size2;
}
