/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

/*=========================================================================

  Program:   wxMaracas
  Module:    $RCSfile: volume.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/15 14:16:13 $
  Version:   $Revision: 1.10 $

  Copyright: (c) 2002, 2003
  License:
  
     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// PS -> //#include <gsl/gsl_math.h>//PS
#include <memory.h>
#include "volume.hxx"

#ifdef KGFO_USE_VTK

#include <vtkCharArray.h>
#include <vtkDataArray.h>
#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkIntArray.h>
#include <vtkPointData.h>
#include <vtkShortArray.h>
#include <vtkUnsignedCharArray.h>
#include <vtkUnsignedIntArray.h>
#include <vtkUnsignedShortArray.h>

// -------------------------------------------------------------------------
const vtkIdType kVolume::VTKTypes[] = { VTK_CHAR, VTK_FLOAT, VTK_DOUBLE,
					VTK_INT, VTK_SHORT, VTK_UNSIGNED_CHAR,
					VTK_UNSIGNED_INT, VTK_UNSIGNED_SHORT };

#endif // KGFO_USE_VTK

// -------------------------------------------------------------------------
const void* kVolume::BLANK   = ( void* ) 0;
const void* kVolume::NOALLOC = ( void* )-1;
const int kVolume::SIZETypes[] = { sizeof( char ), sizeof( float ), 
				   sizeof( double ), sizeof( int ),
				   sizeof( short ), sizeof( uint8_t ),
				   sizeof( uint32_t ), sizeof( uint16_t ) };

// ---------------------------------------------------------------------------
template< class FROM, class TO >
    static void convertCastT( FROM* src, TO* dest, ulong size )
{
    FROM* end = src + size;
    for( ; src < end; src++, dest++ ) *dest = ( TO )*src;
}


// ---------------------------------------------------------------------------
template< class FROM, class TO >
    static void convertScaleT( FROM *src, TO *dest, ulong size,
			       double smin, double tmin, double slope )
{
    FROM* end = src + size;
    for( ; src < end; src++, dest++ )
	*dest = ( TO )( ( double( *src ) - smin ) * slope + tmin );
}

// ---------------------------------------------------------------------------
template< class TYPE >
    static void getMinMaxT( TYPE *src, ulong size,
			    double& min, double& max )
{
    TYPE* end = src + size;
    TYPE m, M;
    bool st;
	
    m = ( TYPE )0;
    M = ( TYPE )0;
    for( st = true; src < end; src++, st = false ) {
		
	if( *src < m || st ) m = *src;
	if( *src > M || st ) M = *src;
		
    } // rof
	
    min = ( double )m;
    max = ( double )M;
}

// -------------------------------------------------------------------------
kVolume::kVolume( )
    : _type( UCHAR ), 
      _creator( SELF ),
      _raw( NULL ), 
      _columns( NULL ),
      _images( NULL )
#ifdef KGFO_USE_VTK
                  ,
      _vtk( NULL )      
#endif // KGFO_USE_VTK      
{
    _dims[ CX ] = 1; _dims[ CY ] = 1; _dims[ CZ ] = 1;
    _sizes[ CX ] = 1; _sizes[ CY ] = 1; _sizes[ CZ ] = 1;
    allocate( );
    buildIndex( );
}

// -------------------------------------------------------------------------
kVolume::kVolume( Type type,
		  uint32_t xdim, uint32_t ydim, uint32_t zdim,
		  double xsize, double ysize, double zsize,
		  void* data )
    : _type( type ), 
      _creator( SELF ),
      _raw( NULL ), 
      _columns( NULL ),
      _images( NULL )
#ifdef KGFO_USE_VTK
                    ,
      _vtk( NULL )
#endif // KGFO_USE_VTK      
{
    _dims[ CX ] = xdim; _dims[ CY ] = ydim; _dims[ CZ ] = zdim;
    _sizes[ CX ] = xsize; _sizes[ CY ] = ysize; _sizes[ CZ ] = zsize;
    if( data != NOALLOC ) {
	
	if( data != BLANK )
	    _raw = data;
	else
	    allocate( );
	buildIndex( );
			
    } // fi
}

// -------------------------------------------------------------------------
kVolume::kVolume( Type type,
		  const uint32_t *dims,
		  const double *sizes,
		  void* data )
    : _type( type ), 
      _creator( SELF ),
      _raw( NULL ), _columns( NULL ),
      _images( NULL )
#ifdef KGFO_USE_VTK
                   ,
      _vtk( NULL )
#endif // KGFO_USE_VTK      
      
{
    memcpy( _dims, dims, 3 * sizeof( uint32_t ) );
    memcpy( _sizes, sizes, 3 * sizeof( double ) );
    if( data != NOALLOC ) {
	
	if( data != BLANK )
	    _raw = data;
	else
	    allocate( );
	buildIndex( );
			
    } // fi
}

// -------------------------------------------------------------------------
kVolume::kVolume( const kVolume& org )
    : _type( UCHAR ), 
      _creator( SELF ),
      _raw( NULL ), 
      _columns( NULL ),
      _images( NULL )
#ifdef KGFO_USE_VTK
		    ,
      _vtk( NULL )
#endif // KGFO_USE_VTK      
  
{
    copyFrom( org );
}

// -------------------------------------------------------------------------
kVolume& kVolume::operator=( const kVolume& org )
{
    copyFrom( org );
    return( *this );
}

// -------------------------------------------------------------------------
void kVolume::copyFrom( const kVolume& org )
{
    _type = org._type;
    _creator = SELF;
    _raw = 0;
    _columns = 0;
    _images = 0;

    memcpy( _dims, org._dims, 3 * sizeof( uint32_t ) );
    memcpy( _sizes, org._sizes, 3 * sizeof( double ) );
    if( org._raw ) {

	allocate( );
	buildIndex( );
	memcpy( _raw, org._raw, getRawSizeInBytes( ) );

    } // fi
}

// -------------------------------------------------------------------------
bool kVolume::sameDimension( const kVolume& comp )
{
    return( ( _dims[ CX ] == comp._dims[ CX ] &&
	      _dims[ CY ] == comp._dims[ CY ] &&
	      _dims[ CZ ] == comp._dims[ CZ ] ) );
}

// -------------------------------------------------------------------------
double kVolume::getPixel( uint32_t x, uint32_t y, uint32_t z ) const
{
    double p;

    switch( _type ) {
		
    case CHAR:   p = ( double )( ( int8_t*** )_images )[ z ][ y ][ x ];   break;
    case FLOAT:  p = ( double )( ( float*** )_images )[ z ][ y ][ x ];    break;
    case DOUBLE: p = ( double )( ( double*** )_images )[ z ][ y ][ x ];   break;
    case INT:    p = ( double )( ( int32_t*** )_images )[ z ][ y ][ x ];  break;
    case SHORT:  p = ( double )( ( int16_t*** )_images )[ z ][ y ][ x ];  break;
    case UCHAR:  p = ( double )( ( uint8_t*** )_images )[ z ][ y ][ x ];  break;
    case UINT:   p = ( double )( ( uint32_t*** )_images )[ z ][ y ][ x ]; break;
    case USHORT: p = ( double )( ( uint16_t*** )_images )[ z ][ y ][ x ]; break;
    default: p = 0.0; break;

    } // fswitch

    return( p );
}

// -------------------------------------------------------------------------
void kVolume::setPixel( double v, uint32_t x, uint32_t y, uint32_t z )
{

    switch( _type ) {
		
    case CHAR:        ( ( int8_t*** )_images )[ z ][ y ][ x ] = ( int8_t )v;   break;
    case FLOAT:        ( ( float*** )_images )[ z ][ y ][ x ] = ( float )v;    break;
    case DOUBLE:      ( ( double*** )_images )[ z ][ y ][ x ] = ( double )v;   break;
    case INT:        ( ( int32_t*** )_images )[ z ][ y ][ x ] = ( int32_t )v;  break;
    case SHORT:      ( ( int16_t*** )_images )[ z ][ y ][ x ] = ( int16_t )v;  break;
    case UCHAR:      ( ( uint8_t*** )_images )[ z ][ y ][ x ] = ( uint8_t )v;  break;
    case UINT:      ( ( uint32_t*** )_images )[ z ][ y ][ x ] = ( uint32_t )v; break;
    case USHORT:    ( ( uint16_t*** )_images )[ z ][ y ][ x ] = ( uint16_t )v; break;
    default: break;

    } // fswitch
}

// -------------------------------------------------------------------------
void kVolume::convertCast( Type type )
{
    if( _type != type ) {

	void* buffer;
	ulong size = getRawSize( );

	buffer = ( void* )new uint8_t[ size * SIZETypes[ type ] ];

	switch( _type ) {

	case CHAR:
	    switch( type ) {

	    case SHORT:  convertCastT( ( char* )_raw, ( int16_t* )buffer, size );  break;
	    case INT:    convertCastT( ( char* )_raw, ( int32_t* )buffer, size );  break;
	    case USHORT: convertCastT( ( char* )_raw, ( uint16_t* )buffer, size ); break;
	    case UINT:   convertCastT( ( char* )_raw, ( uint32_t* )buffer, size ); break;
	    case FLOAT:  convertCastT( ( char* )_raw, ( float* )buffer, size );    break;
	    case DOUBLE: convertCastT( ( char* )_raw, ( double* )buffer, size );   break;
	    case UCHAR:  convertCastT( ( char* )_raw, ( uint8_t* )buffer, size ); break;
	    default : break;

	    } // fswitch
	    break;

	case SHORT:
	    switch( type ) {

	    case CHAR:   convertCastT( ( int16_t* )_raw, ( uint8_t* )buffer, size );  break;
	    case INT:    convertCastT( ( int16_t* )_raw, ( int32_t* )buffer, size );  break;
	    case USHORT: convertCastT( ( int16_t* )_raw, ( uint16_t* )buffer, size ); break;
	    case UINT:   convertCastT( ( int16_t* )_raw, ( uint32_t* )buffer, size ); break;
	    case FLOAT:  convertCastT( ( int16_t* )_raw, ( float* )buffer, size );    break;
	    case DOUBLE: convertCastT( ( int16_t* )_raw, ( double* )buffer, size );   break;
	    case UCHAR:  convertCastT( ( int16_t* )_raw, ( uint8_t* )buffer, size );  break;
	    default : break;
	    } // fswitch
	    break;

	case INT:
	    switch( type ) {

	    case CHAR:   convertCastT( ( int32_t* )_raw, ( int8_t* )buffer, size ); break;
	    case SHORT:  convertCastT( ( int32_t* )_raw, ( int16_t* )buffer, size ); break;
	    case USHORT: convertCastT( ( int32_t* )_raw, ( uint16_t* )buffer, size ); break;
	    case UINT:   convertCastT( ( int32_t* )_raw, ( uint32_t* )buffer, size ); break;
	    case FLOAT:  convertCastT( ( int32_t* )_raw, ( float* )buffer, size ); break;
	    case DOUBLE: convertCastT( ( int32_t* )_raw, ( double* )buffer, size ); break;
	    case UCHAR:  convertCastT( ( int32_t* )_raw, ( uint8_t* )buffer, size ); break;
	    default : break;
	    } // fswitch
	    break;

	case USHORT:
	    switch( type ) {

	    case CHAR:   convertCastT( ( uint16_t* )_raw, ( int8_t* )buffer, size ); break;
	    case SHORT:  convertCastT( ( uint16_t* )_raw, ( int16_t* )buffer, size ); break;
	    case INT:    convertCastT( ( uint16_t* )_raw, ( int32_t* )buffer, size ); break;
	    case UINT:   convertCastT( ( uint16_t* )_raw, ( uint32_t* )buffer, size ); break;
	    case FLOAT:  convertCastT( ( uint16_t* )_raw, ( float* )buffer, size ); break;
	    case DOUBLE: convertCastT( ( uint16_t* )_raw, ( double* )buffer, size ); break;
	    case UCHAR:  convertCastT( ( uint16_t* )_raw, ( uint8_t* )buffer, size ); break;
	    default : break;
	    } // fswitch
	    break;

	case UINT:
	    switch( type ) {

	    case CHAR:   convertCastT( ( uint32_t* )_raw, ( int8_t* )buffer, size ); break;
	    case SHORT:  convertCastT( ( uint32_t* )_raw, ( int16_t* )buffer, size ); break;
	    case INT:    convertCastT( ( uint32_t* )_raw, ( int32_t* )buffer, size ); break;
	    case USHORT: convertCastT( ( uint32_t* )_raw, ( uint16_t* )buffer, size ); break;
	    case FLOAT:  convertCastT( ( uint32_t* )_raw, ( float* )buffer, size ); break;
	    case DOUBLE: convertCastT( ( uint32_t* )_raw, ( double* )buffer, size ); break;
	    case UCHAR:  convertCastT( ( uint32_t* )_raw, ( uint8_t* )buffer, size ); break;
	    default : break;
	    } // fswitch
	    break;

	case FLOAT:
	    switch( type ) {

	    case CHAR:   convertCastT( ( float* )_raw, ( int8_t* )buffer, size ); break;
	    case SHORT:  convertCastT( ( float* )_raw, ( int16_t* )buffer, size ); break;
	    case INT:    convertCastT( ( float* )_raw, ( int32_t* )buffer, size ); break;
	    case USHORT: convertCastT( ( float* )_raw, ( uint16_t* )buffer, size ); break;
	    case UINT:   convertCastT( ( float* )_raw, ( uint32_t* )buffer, size ); break;
	    case DOUBLE: convertCastT( ( float* )_raw, ( double* )buffer, size ); break;
	    case UCHAR:  convertCastT( ( float* )_raw, ( uint8_t* )buffer, size ); break;
	    default : break;
	    } // fswitch
	    break;

	case DOUBLE:
	    switch( type ) {

	    case CHAR:   convertCastT( ( double* )_raw, ( int8_t* )buffer, size ); break;
	    case SHORT:  convertCastT( ( double* )_raw, ( int16_t* )buffer, size ); break;
	    case INT:    convertCastT( ( double* )_raw, ( int32_t* )buffer, size ); break;
	    case USHORT: convertCastT( ( double* )_raw, ( uint16_t* )buffer, size ); break;
	    case UINT:   convertCastT( ( double* )_raw, ( uint32_t* )buffer, size ); break;
	    case FLOAT:  convertCastT( ( double* )_raw, ( float* )buffer, size ); break;
	    case UCHAR:  convertCastT( ( double* )_raw, ( uint8_t* )buffer, size ); break;
	    default : break;
	    } // fswitch
	    break;

	case UCHAR:
	    switch( type ) {

	    case CHAR:   convertCastT( ( uint8_t* )_raw, ( int8_t* )buffer, size ); break;
	    case SHORT:  convertCastT( ( uint8_t* )_raw, ( int16_t* )buffer, size ); break;
	    case INT:    convertCastT( ( uint8_t* )_raw, ( int32_t* )buffer, size ); break;
	    case USHORT: convertCastT( ( uint8_t* )_raw, ( uint16_t* )buffer, size ); break;
	    case UINT:   convertCastT( ( uint8_t* )_raw, ( uint32_t* )buffer, size ); break;
	    case FLOAT:  convertCastT( ( uint8_t* )_raw, ( float* )buffer, size ); break;
	    case DOUBLE: convertCastT( ( uint8_t* )_raw, ( double* )buffer, size ); break;
	    default : break;
	    } // fswitch
	    break;
     
	} // fswitch

	_type = type;
	deallocate( );
	_creator = SELF;
	_raw = buffer;
	buildIndex( );

    } // fi
}

// -------------------------------------------------------------------------
void kVolume::convertScale( Type type, double min, double max )
{
    double tmin, tmax, smin, smax;

// PS ->     //tmin = GSL_MIN( min, max ); //PS
	tmin= ((min<max)?min:max);
// PS ->     //tmax = GSL_MAX( min, max ); //PS
	tmax= ((min>max)?min:max);

    getMinMax( smin, smax );
	
    // compute scaling slope
    double a = ( tmax - tmin ) / ( smax - smin );

    void* buffer;
    ulong size = getRawSize( );

    buffer = ( void* )new uint8_t[ size * SIZETypes[ type ] ];

    switch( _type ) {

    case CHAR:
	switch( type ) {

	case CHAR:   convertScaleT( ( int8_t* )_raw, ( int8_t* )buffer,   size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( int8_t* )_raw, ( int16_t* )buffer,  size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( int8_t* )_raw, ( int32_t* )buffer,  size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( int8_t* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( int8_t* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( int8_t* )_raw, ( float* )buffer,    size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( int8_t* )_raw, ( double* )buffer,   size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( int8_t* )_raw, ( uint8_t* )buffer,  size, smin, tmin, a ); break;

	} // fswitch
	break;
    case SHORT:
	switch( type ) {

	case CHAR:   convertScaleT( ( int16_t* )_raw, ( int8_t* )buffer,   size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( int16_t* )_raw, ( int16_t* )buffer,  size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( int16_t* )_raw, ( int32_t* )buffer,  size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( int16_t* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( int16_t* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( int16_t* )_raw, ( float* )buffer,    size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( int16_t* )_raw, ( double* )buffer,   size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( int16_t* )_raw, ( uint8_t* )buffer,  size, smin, tmin, a ); break;

	} // fswitch
	break;
    case INT:
	switch( type ) {

	case CHAR:   convertScaleT( ( int32_t* )_raw, ( int8_t* )buffer,   size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( int32_t* )_raw, ( int16_t* )buffer,  size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( int32_t* )_raw, ( int32_t* )buffer,  size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( int32_t* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( int32_t* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( int32_t* )_raw, ( float* )buffer,    size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( int32_t* )_raw, ( double* )buffer,   size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( int32_t* )_raw, ( uint8_t* )buffer,  size, smin, tmin, a ); break;

	} // fswitch
	break;
    case USHORT:
	switch( type ) {

	case CHAR:   convertScaleT( ( uint16_t* )_raw, ( int8_t* )buffer,   size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( uint16_t* )_raw, ( int16_t* )buffer,  size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( uint16_t* )_raw, ( int32_t* )buffer,  size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( uint16_t* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( uint16_t* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( uint16_t* )_raw, ( float* )buffer,    size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( uint16_t* )_raw, ( double* )buffer,   size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( uint16_t* )_raw, ( uint8_t* )buffer,  size, smin, tmin, a ); break;

	} // fswitch
	break;
    case UINT:
	switch( type ) {

	case CHAR:   convertScaleT( ( uint32_t* )_raw, ( int8_t* )buffer,   size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( uint32_t* )_raw, ( int16_t* )buffer,  size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( uint32_t* )_raw, ( int32_t* )buffer,  size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( uint32_t* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( uint32_t* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( uint32_t* )_raw, ( float* )buffer,    size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( uint32_t* )_raw, ( double* )buffer,   size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( uint32_t* )_raw, ( uint8_t* )buffer,  size, smin, tmin, a ); break;

	} // fswitch
	break;
    case UCHAR:
	switch( type ) {

	case CHAR:   convertScaleT( ( uint8_t* )_raw, ( int8_t* )buffer,  size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( uint8_t* )_raw, ( int16_t* )buffer,  size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( uint8_t* )_raw, ( int32_t* )buffer,  size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( uint8_t* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( uint8_t* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( uint8_t* )_raw, ( float* )buffer,    size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( uint8_t* )_raw, ( double* )buffer,   size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( uint8_t* )_raw, ( uint8_t* )buffer,  size, smin, tmin, a ); break;

	} // fswitch
	break;
    case DOUBLE:
	switch( type ) {

	case CHAR:   convertScaleT( ( double* )_raw, ( int8_t* )buffer,   size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( double* )_raw, ( int16_t* )buffer,  size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( double* )_raw, ( int32_t* )buffer,  size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( double* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( double* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( double* )_raw, ( float* )buffer,    size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( double* )_raw, ( double* )buffer,   size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( double* )_raw, ( uint8_t* )buffer,  size, smin, tmin, a ); break;

	} // fswitch
	break;
    case FLOAT:
	switch( type ) {

	case CHAR:   convertScaleT( ( float* )_raw, ( int8_t* )buffer, size, smin, tmin, a ); break;
	case SHORT:  convertScaleT( ( float* )_raw, ( int16_t* )buffer, size, smin, tmin, a ); break;
	case INT:    convertScaleT( ( float* )_raw, ( int32_t* )buffer, size, smin, tmin, a ); break;
	case USHORT: convertScaleT( ( float* )_raw, ( uint16_t* )buffer, size, smin, tmin, a ); break;
	case UINT:   convertScaleT( ( float* )_raw, ( uint32_t* )buffer, size, smin, tmin, a ); break;
	case FLOAT:  convertScaleT( ( float* )_raw, ( float* )buffer, size, smin, tmin, a ); break;
	case DOUBLE: convertScaleT( ( float* )_raw, ( double* )buffer, size, smin, tmin, a ); break;
	case UCHAR:  convertScaleT( ( float* )_raw, ( uint8_t* )buffer, size, smin, tmin, a ); break;

	} // fswitch
	break;

    } // fswitch

    _type = type;
    deallocate( );
    _creator = SELF;
    _raw = buffer;
    buildIndex( );
}

// -------------------------------------------------------------------------
void kVolume::getMinMax( double& min, double& max ) const
{
    ulong size = getRawSize( );

    switch( _type ) {
		
    case CHAR:   getMinMaxT( ( int8_t* )_raw,   size, min, max ); break;
    case FLOAT:  getMinMaxT( ( float* )_raw,    size, min, max ); break;
    case DOUBLE: getMinMaxT( ( double* )_raw,   size, min, max ); break;
    case INT:    getMinMaxT( ( int32_t* )_raw,  size, min, max ); break;
    case SHORT:  getMinMaxT( ( int16_t* )_raw,  size, min, max ); break;
    case UCHAR:  getMinMaxT( ( uint8_t* )_raw,  size, min, max ); break;
    case UINT:   getMinMaxT( ( uint32_t* )_raw, size, min, max ); break;
    case USHORT: getMinMaxT( ( uint16_t* )_raw, size, min, max ); break;

    } // fswitch
}

// -------------------------------------------------------------------------
double kVolume::getMin( ) const
{
    double m, M;

    getMinMax( m, M );
    return( m );
}

// -------------------------------------------------------------------------
double kVolume::getMax( ) const
{
    double m, M;

    getMinMax( m, M );
    return( M );
}

// -------------------------------------------------------------------------
/*double kVolume::GetMaxIntSphere( double* p, double r )
{
    int minX, minY, minZ, maxX, maxY, maxZ;
    gslobj_vector vP( p, 3 ), v( 3 );
    double maxint, tmp;
    bool start;
	
    minX = int( floor( p[ 0 ] - r ) );
    minY = int( floor( p[ 1 ] - r ) );
    minZ = int( floor( p[ 2 ] - r ) );
    maxX = int( ceil( p[ 0 ] + r ) );
    maxY = int( ceil( p[ 1 ] + r ) );
    maxZ = int( ceil( p[ 2 ] + r ) );
	
    minX = GSL_MAX( minX, 0 );
    minY = GSL_MAX( minY, 0 );
    minZ = GSL_MAX( minZ, 0 );
	
    maxX = GSL_MIN( maxX, this->getXdim( ) - 1 );
    maxY = GSL_MIN( maxY, this->getYdim( ) - 1 );
    maxZ = GSL_MIN( maxZ, this->getZdim( ) - 1 );

    maxint = 0.0;
    start = true;
    for( v( 0 ) = minX; v( 0 ) <= maxX; v( 0 )++ )
      for( v( 1 ) = minY; v( 1 ) <= maxY; v( 1 )++ )
        for( v( 2 ) = minZ; v( 2 ) <= maxZ; v( 2 )++ )
          if( ( v - vP ).norm2( ) <= r ) {
            tmp = this->getPixel( ( uint32_t )v(0), ( uint32_t )v(1), ( uint32_t )v(2));
            maxint = ( tmp > maxint || start )? tmp: maxint;
            start = false;
          } // fi

    return( maxint );
}*/
// -------------------------------------------------------------------------
unsigned short kVolume::GetMaxIntSphere2( double* p, double r )
{
   /**
    * unsigned short range : 0 -> 65535
    * unsigned int   range : 0 -> 2147483647 // 2^31 - 1
    */
    int minX, minY, minZ, maxX, maxY, maxZ;
    unsigned int v[3];
    unsigned short maxint = 0, tmp;

    minX = int( floor( p[ 0 ] - r ) );
    minY = int( floor( p[ 1 ] - r ) );
    minZ = int( floor( p[ 2 ] - r ) );
    maxX = int( ceil( p[ 0 ] + r ) );
    maxY = int( ceil( p[ 1 ] + r ) );
    maxZ = int( ceil( p[ 2 ] + r ) );

// PS ->     //minX = GSL_MAX( minX, 0 );//PS
// PS ->     //minY = GSL_MAX( minY, 0 );//PS
// PS ->     //minZ = GSL_MAX( minZ, 0 );//PS
	minX=((minX>0)?minX:0);
	minY=((minY>0)?minY:0);
	minZ=((minZ>0)?minZ:0);
	
// PS ->     //maxX = GSL_MIN( maxX, this->getXdim( ) - 1 );//PS
// PS ->     //maxY = GSL_MIN( maxY, this->getYdim( ) - 1 );//PS
// PS ->     //maxZ = GSL_MIN( maxZ, this->getZdim( ) - 1 );//PS
	int xDim=this->getXdim( ) - 1;
	maxX= (maxX<xDim?maxX:xDim);

	int yDim=this->getYdim( ) - 1;
	maxY= (maxY<yDim?maxY:yDim);

	int zDim=this->getZdim( ) - 1;
	maxZ= (maxZ<zDim?maxZ:zDim);

    double r2 = r*r;  //need to do comparison in double ... don't know why...
    for( v[0] = minX; v[0] <= maxX; v[0]++ )
      for( v[1] = minY; v[1] <= maxY; v[1]++ )
        for( v[2] = minZ; v[2] <= maxZ; v[2]++ )
          if( ((v[0]-p[0])*(v[0]-p[0])+(v[1]-p[1])*(v[1]-p[1])+(v[2]-p[2])*(v[2]-p[2])) <= r2 )
          {
            tmp = (unsigned short)this->getPixel( v[0], v[1], v[2] );
            maxint = ( tmp > maxint ) ? tmp : maxint;
          }

    return( maxint );
}

// -------------------------------------------------------------------------
void kVolume::allocate( )
{
    ulong size = getRawSizeInBytes( );

    if( _creator == SELF ) {

	_raw = ( void* )new uint8_t[ size ];
	memset( _raw, 0, size );

    } // fi
}

// -------------------------------------------------------------------------
void kVolume::buildIndex( )
{
    ulong size;

    size = ( _dims[ CZ ] * sizeof( uint8_t** ) ) +
	( _dims[ CZ ] * _dims[ CY ] * sizeof( void* ) );
		
	_images = ( void*** )new uint8_t[ size ];
		
	_columns = ( void** )( _images + _dims[ CZ ] );
	void** plane = _columns;
	for( uint32_t z = 0; z < _dims[ CZ ]; z++ ) {
			
	    _images[ z ] = plane;
	    plane += _dims[ CY ];
			
	} // rof
	void* line = _raw;
	for( uint32_t y = 0; y < _dims[ CZ ] * _dims[ CY ]; y++ ) {
			
	    _columns[ y ] = line;
	    line = ( void* )( ( uint8_t* ) line +
			      _dims[ CX ] * SIZETypes[ _type ] );
			
	} // rof
	
#ifdef KGFO_USE_VTK
	
    vtkCharArray			*carray;
    vtkDoubleArray			*darray;
    vtkFloatArray			*farray;
    vtkIntArray				*iarray;
    vtkShortArray			*sarray;
    vtkUnsignedCharArray	*ucarray;
    vtkUnsignedIntArray		*uiarray;
    vtkUnsignedShortArray	*usarray;
	
    size = _dims[ CX ] * _dims[ CY ] * _dims[ CZ ];

    if( _creator == SELF || _creator == IDO ) {
	  
	_vtk = vtkImageData::New( );
	_vtk->SetDimensions( _dims[ CX ], _dims[ CY ], _dims[ CZ ] );
	_vtk->SetSpacing( _sizes[ CX ], _sizes[ CY ], _sizes[ CZ ] );

	if (_type==CHAR)
	{
	    carray = vtkCharArray::New( );
	    carray->SetArray( ( char* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    carray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_CHAR );
#else
		vtkInformation* infoC=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoC, VTK_CHAR, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( carray );
	    carray->Delete( );
	}

	if (_type==UCHAR)
	{
	    ucarray = vtkUnsignedCharArray::New( );
	    ucarray->SetArray( ( uint8_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    ucarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_UNSIGNED_CHAR );
#else
		vtkInformation* infoUC=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoUC, VTK_UNSIGNED_CHAR, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( ucarray );
	    ucarray->Delete( );
	}





	if (_type==SHORT)
	{
	    sarray = vtkShortArray::New( );
	    sarray->SetArray( ( int16_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    sarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_SHORT );
#else
		vtkInformation* infoS=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoS, VTK_SHORT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( sarray );
	    sarray->Delete( );
}

	if (_type==INT)
	{
	    iarray = vtkIntArray::New( );
	    iarray->SetArray( ( int32_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    iarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_INT );
#else
		vtkInformation* infoI=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoI, VTK_INT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( iarray );
	    iarray->Delete( );
	}


	if (_type==USHORT)
	{
	    usarray = vtkUnsignedShortArray::New( );
	    usarray->SetArray( ( uint16_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    usarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_UNSIGNED_SHORT );
#else
		vtkInformation* infoUS=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoUS, VTK_UNSIGNED_SHORT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( usarray );
	    usarray->Delete( );
	}


	if (_type==UINT)
	{
	    uiarray = vtkUnsignedIntArray::New( );
	    uiarray->SetArray( ( uint32_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    uiarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_UNSIGNED_INT );
#else
		vtkInformation* infoUI=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoUI, VTK_UNSIGNED_INT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( uiarray );
	    uiarray->Delete( );
	}


	if (_type==FLOAT)
	{
	    farray = vtkFloatArray::New( );
	    farray->SetArray( ( float* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    farray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_FLOAT );
#else
		vtkInformation* infoF=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoF, VTK_FLOAT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( farray );
	    farray->Delete( );
	}

		  
	if (_type==DOUBLE)
	{
	    darray = vtkDoubleArray::New( );
	    darray->SetArray( ( double* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    darray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_DOUBLE );
#else
		vtkInformation* infoD=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoD, VTK_DOUBLE, 1);
#endif


	    _vtk->GetPointData( )->SetScalars( darray );
	    darray->Delete( );
	}






/*
	switch( _type ) {

	case CHAR:
	    carray = vtkCharArray::New( );
	    carray->SetArray( ( char* )( _raw ), size, 1 );

//EED 2017-01-01 Migration VTK7
//#if VTK_MAJOR_VERSION <= 5
	    carray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_CHAR );
#else
		vtkInformation* infoC=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoC, VTK_CHAR, 1);
#endif

	    _vtk->GetPointData( )->SetScalars( carray );
	    carray->Delete( );
	    break;

	case UCHAR:
	    ucarray = vtkUnsignedCharArray::New( );
	    ucarray->SetArray( ( uint8_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    ucarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_UNSIGNED_CHAR );
#else
		vtkInformation* infoUC=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoUC, VTK_UNSIGNED_CHAR, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( ucarray );
	    ucarray->Delete( );
	    break;

	case SHORT:
	    sarray = vtkShortArray::New( );
	    sarray->SetArray( ( int16_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    sarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_SHORT );
#else
		vtkInformation* infoS=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoS, VTK_SHORT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( sarray );
	    sarray->Delete( );
	    break;

	case INT:
	    iarray = vtkIntArray::New( );
	    iarray->SetArray( ( int32_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    iarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_INT );
#else
		vtkInformation* infoI=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoI, VTK_INT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( iarray );
	    iarray->Delete( );
	    break;

	case USHORT:
	    usarray = vtkUnsignedShortArray::New( );
	    usarray->SetArray( ( uint16_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    usarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_UNSIGNED_SHORT );
#else
		vtkInformation* infoUS=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoUS, VTK_UNSIGNED_SHORT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( usarray );
	    usarray->Delete( );
	    break;

	case UINT:
	    uiarray = vtkUnsignedIntArray::New( );
	    uiarray->SetArray( ( uint32_t* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    uiarray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_UNSIGNED_INT );
#else
		vtkInformation* infoUI=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoUI, VTK_UNSIGNED_INT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( uiarray );
	    uiarray->Delete( );
	    break;

	case FLOAT:
	    farray = vtkFloatArray::New( );
	    farray->SetArray( ( float* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    farray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_FLOAT );
#else
		vtkInformation* infoF=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoF, VTK_FLOAT, 1);
#endif
	    _vtk->GetPointData( )->SetScalars( farray );
	    farray->Delete( );
	    break;
		  
	case DOUBLE:
	    darray = vtkDoubleArray::New( );
	    darray->SetArray( ( double* )( _raw ), size, 1 );
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	    darray->SetNumberOfComponents( 1 );
	    _vtk->SetScalarType( VTK_DOUBLE );
#else
		vtkInformation* infoD=_vtk->GetInformation();
		vtkDataObject::SetPointDataActiveScalarInfo(infoD, VTK_DOUBLE, 1);
#endif


	    _vtk->GetPointData( )->SetScalars( darray );
	    darray->Delete( );
	    break;
		  
	} // fswitch
*/  

    } // fi

#endif // KGFO_USE_VTK
}

// -------------------------------------------------------------------------
void kVolume::deallocate( )
{
#ifdef KGFO_USE_VTK
    if( _vtk ) _vtk->Delete();
    _vtk = NULL;
#endif // KGFO_USE_VTK
    delete[] ( uint8_t* )_images;
    if( _raw && _creator == SELF )

//EED purify 12/sept/2006
//	delete[] ( uint8_t* )_raw;

    free ( _raw );

    _creator = SELF;
    _raw     = NULL;
    _columns = NULL;
    _images  = NULL;
}

#ifdef KGFO_USE_VTK

// -------------------------------------------------------------------------
kVolume::kVolume( vtkImageData* org )
    :
      _creator( VTK ),
      _raw( 0 ), 
      _columns( 0 ), 
      _images( 0 ),
      _vtk( 0 )
{
    //int i, j, k, y;
    int itmp[ 3 ];
    double ftmp[ 3 ];
    //double v;

    switch( org->GetScalarType( ) ) {

    case VTK_CHAR:           _type = CHAR;   break;
    case VTK_UNSIGNED_CHAR:  _type = UCHAR;  break;
    case VTK_SHORT:          _type = SHORT;  break;
    case VTK_INT:            _type = INT;    break;
    case VTK_UNSIGNED_SHORT: _type = USHORT; break;
    case VTK_UNSIGNED_INT:   _type = UINT;   break;
    case VTK_FLOAT:          _type = FLOAT;  break;
    case VTK_DOUBLE:         _type = DOUBLE; break;
    default: break;
		
    } // fswitch
	
    org->GetDimensions( itmp );
    _dims[ CX ] = ( uint32_t )itmp[ 0 ];
    _dims[ CY ] = ( uint32_t )itmp[ 1 ];
    _dims[ CZ ] = ( uint32_t )itmp[ 2 ];
    org->GetSpacing( ftmp );
    _sizes[ CX ] = ( double )ftmp[ 0 ];
    _sizes[ CY ] = ( double )ftmp[ 1 ];
    _sizes[ CZ ] = ( double )ftmp[ 2 ];

    _raw = org->GetPointData( )->GetScalars( )->GetVoidPointer( 0 );
    //_vtk = org;
    _vtk = vtkImageData::New();
    _vtk->ShallowCopy( org );
    buildIndex( );
}

// -------------------------------------------------------------------------
kVolume& kVolume::operator=( vtkImageData* org )
{
    copyFrom( org );
    return( *this );
}

// -------------------------------------------------------------------------
void kVolume::copyFrom( vtkImageData* org )
{
    int i, j, k;//, y;
    int itmp[ 3 ];
    int ext[ 6 ];
    double ftmp[ 3 ];
    double v;

    deallocate( );
	
//#ifdef KGFO_USE_IDO

//    _privateIdo = NULL;

//#endif // KGFO_USE_IDO

    _vtk     = NULL;
    _raw     = NULL;
    _columns = NULL;
    _images  = NULL;
    _creator = SELF;

    switch( org->GetScalarType( ) ) {

    case VTK_CHAR:           _type = CHAR;   break;
    case VTK_UNSIGNED_CHAR:  _type = UCHAR;  break;
    case VTK_SHORT:          _type = SHORT;  break;
    case VTK_INT:            _type = INT;    break;
    case VTK_UNSIGNED_SHORT: _type = USHORT; break;
    case VTK_UNSIGNED_INT:   _type = UINT;   break;
    case VTK_FLOAT:          _type = FLOAT;  break;
    case VTK_DOUBLE:         _type = DOUBLE; break;
    default: break;
		
    } // fswitch
	
    org->GetDimensions( itmp );
    _dims[ CX ] = ( uint32_t )itmp[ 0 ];
    _dims[ CY ] = ( uint32_t )itmp[ 1 ];
    _dims[ CZ ] = ( uint32_t )itmp[ 2 ];
    org->GetSpacing( ftmp );
    _sizes[ CX ] = ( double )ftmp[ 0 ];
    _sizes[ CY ] = ( double )ftmp[ 1 ];
    _sizes[ CZ ] = ( double )ftmp[ 2 ];

    allocate( );
    buildIndex( );

    // This avoids vtk extent crap conversion...
    org->GetExtent( ext );
    for( i = ext[ 0 ]; i <= ext[ 1 ]; i++ ) {
	for( j = ext[ 2 ]; j <= ext[ 3 ]; j++ ) {
	    for( k = ext[ 4 ]; k <= ext[ 5 ]; k++ ) {		
		v = org->GetScalarComponentAsDouble( i, j, k, 0 );
		setPixel( v, i - ext[ 0 ], j - ext[ 2 ], k - ext[ 4 ] );		
	    } // rof
	} // rof
    } // rof
}

#endif // KGFO_USE_VTK


// eof - volume.cxx
