/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

/*=========================================================================

  Program:   wxMaracas
  Module:    $RCSfile: volume.hxx,v $
  Language:  C++
  Date:      $Date: 2012/11/15 14:16:12 $
  Version:   $Revision: 1.9 $

  Copyright: (c) 2002, 2003
  License:
  
     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef KGFO__VOLUME__HXX
#define KGFO__VOLUME__HXX

#include "marTypes.h"
#include <string>

#ifdef KGFO_USE_VTK
#include <vtkImageData.h>
#endif // KGFO_USE_VTK

class creaMaracasVisu_EXPORT kVolume
{
    public:

    enum Creator{ VTK, IDO, SELF };
    enum Coord{ CX = 0, CY = 1, CZ = 2 };
    enum Type{ CHAR = 0, FLOAT = 1, DOUBLE = 2,
	       INT = 3, SHORT = 4, UCHAR = 5,
	       UINT = 6, USHORT = 7 };

    static const int SIZETypes[];
    static const void* BLANK;
    static const void* NOALLOC;
#ifdef KGFO_USE_VTK

    static const vtkIdType VTKTypes[];

#endif // KGFO_USE_VTK

    public:

    kVolume( );

    kVolume( Type type,
	     uint32_t xdim, uint32_t ydim, uint32_t zdim,
	     double xsize = 1, double ysize = 1, double zsize = 1,
	     void* data = 0 );
    kVolume( Type type,
	     const uint32_t *dims,
	     const double *sizes,
	     void* data = 0 );

    kVolume( const kVolume& org );
    kVolume& operator=( const kVolume& org );
    void copyFrom( const kVolume& org );

    virtual ~kVolume( ) { deallocate( ); }

    Type getType( ) const { return( _type ); }

    bool sameDimension( const kVolume& comp );

    const uint32_t* getDimensions( ) const { return( _dims ); }
    uint32_t getWidth( ) const { return( _dims[ CX ] ); }
    uint32_t getHeight( ) const { return( _dims[ CY ] ); }
    uint32_t getDepth( ) const { return( _dims[ CZ ] ); }
    uint32_t getNcols( ) const { return( _dims[ CX ] ); }
    uint32_t getNrows( ) const { return( _dims[ CY ] ); }
    uint32_t getNplanes( ) const { return( _dims[ CZ ] ); }
    uint32_t getXdim( ) const { return( _dims[ CX ] ); }
    uint32_t getYdim( ) const { return( _dims[ CY ] ); }
    uint32_t getZdim( ) const { return( _dims[ CZ ] ); }

    void setDimensions( uint32_t* dims )
    {
	memcpy( _dims, dims, 3 * sizeof( uint32_t ) );
    }
    void setWidth( uint32_t w ) { _dims[ CX ] = w; }
    void setHeight( uint32_t h ) { _dims[ CY ] = h; }
    void setDepth( uint32_t d ) { _dims[ CZ ] = d; }
    void setNcols( uint32_t c ) { _dims[ CX ] = c; }
    void setNrows( uint32_t r ) { _dims[ CY ] = r; }
    void setNplanes( uint32_t p ) { _dims[ CZ ] = p; }
    void setXdim( uint32_t x ) { _dims[ CX ] = x; }
    void setYdim( uint32_t y ) { _dims[ CY ] = y; }
    void setZdim( uint32_t z ) { _dims[ CZ ] = z; }

    const double *getSizes( ) const { return( _sizes ); }
    double getXsize( ) const { return( _sizes[ CX ] ); }
    double getYsize( ) const { return( _sizes[ CY ] ); }
    double getZsize( ) const { return( _sizes[ CZ ] ); }

    void setSizes( double* sizes )
    {
	memcpy( _sizes, sizes, 3 * sizeof( double ) );
    }
    void setXsize( double x ) { _sizes[ CX ] = x; }
    void setYsize( double y ) { _sizes[ CY ] = y; }
    void setZsize( double z ) { _sizes[ CZ ] = z; }

    void* getData1D( ) { return( _raw ); }
    const void* getData1D( ) const { return( _raw ); }

    void** getData2D( ) { return( _columns ); }
    void** const getData2D( ) const { return( _columns ); }

    void*** getData3D( ) { return( _images ); }
    void*** const getData3D( ) const { return( _images ); }

    void reset( ) { memset( _raw, 0, getRawSizeInBytes( ) ); }

    ulong getRawSize( ) const
    {
	return( ( ulong )_dims[ CX ] *
		( ulong )_dims[ CY ] *
		( ulong )_dims[ CZ ] );
    }
    ulong getRawSizeInBytes( ) const
    {
	return( SIZETypes[ _type ] * getRawSize( ) );
    }

    double getPixel( uint x, uint y, uint z ) const;
    void setPixel( double v, uint x, uint y, uint z );

    void convertCast( Type type );
    void convertScale( Type type, double min, double max );
    void normalize( double min, double max )
    {
	convertScale( _type, min, max );
    }

    void getMinMax( double& min, double& max ) const;
    double getMin( ) const;
    double getMax( ) const;
    //double GetMaxIntSphere( double* p, const double r ) const;
    unsigned short GetMaxIntSphere2( double* p, const double r );

  private:

    void allocate( );
    void buildIndex( );
    void deallocate( );

  protected:

    Type _type;
    Creator _creator;

    uint32_t _dims[ 3 ];
    double _sizes[ 3 ];
    void* _raw;
    void** _columns;
    void*** _images;

#ifdef KGFO_USE_VTK

    public:

    /**
     * Creators/copiers from a vtkImageData object.
     * C++ creators allows you to create a kVolume cast to you vtkImageData
     * object, but if extent != bounds, you can expect a weird behaviour.
     * If you want to forget all about extents, then crop your volume or
     * use either the operator= or the copyFrom method.
     */
    kVolume( vtkImageData* org );
    kVolume& operator=( vtkImageData* org );
    void copyFrom( vtkImageData* org ); 

    vtkImageData* castVtk( ) const { return( _vtk ); }
    operator vtkImageData*( ) const { return( _vtk ); }

    protected:

    vtkImageData* _vtk;

#endif // KGFO_USE_VTK

};

#endif // KGFO__VOLUME__HXX

// eof - volume.hxx
