/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbstdMagicBox.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:51:32 $
  Version:   $Revision: 1.14 $
=========================================================================*/


#ifndef __bbstdMagicBox_h_INCLUDED_H__
#define __bbstdMagicBox_h_INCLUDED_H__

#include "bbtkAtomicBlackBox.h"
#include "bbstd_EXPORT.h"

namespace bbstd
{

  //==================================================================
  class bbstd_EXPORT MagicBox
    : 
    public bbtk::AtomicBlackBox
  {
    BBTK_BLACK_BOX_INTERFACE(MagicBox,bbtk::AtomicBlackBox);
    BBTK_DECLARE_INPUT(In,bbtk::Data);
    BBTK_DECLARE_INPUT(Active,bool);
    BBTK_DECLARE_OUTPUT(Out,bbtk::Data);
    BBTK_PROCESS(DoProcess);
    void DoProcess();

    virtual void bbComputePostProcessStatus();
 };
  //==================================================================
  

  //==================================================================
  // We have to create a particular SetFunctor for MagicBox because
  // its input is of type bbtk::Data (i.e. any) and :
  // 1) an any cannot store an any (construction with an any invokes the copy constr.)
  // 2) we cannot invoke the Set method with the content of the any because 
  //   it expects an any 
  // hence the Set method of the functor **MUST NOT** extract the 
  // content of the Data prior to invoking the set method of the box
  class MagicBoxSetFunctor : public bbtk::AtomicBlackBoxSetFunctor
  {
  public:
    /// Type of pointer on a UBB::Set method  
    typedef void (MagicBox::*SetMethodPointerType)(bbtk::Data);
    
    /// Construction with the pointer on the Set method
    MagicBoxSetFunctor(SetMethodPointerType s) :
       mSetMethodPointer(s) 
       {
       }
    
    /// Concrete application of the Set method of object o
    void Set(bbtk::AtomicBlackBox* o, const bbtk::Data& d)
    { 
      bbtkDebugMessage("data",9,"MagicBoxSetfunctor::Set("<<
		       bbtk::HumanTypeName(d.type())<<
		       ")"<<std::endl);
      (((MagicBox*)o)->*mSetMethodPointer)(d);
    }
    
    /// 
    bbtk::TypeInfo GetTypeInfo() const { return typeid(bbtk::Data); }
    std::string GetTypeName() const { return bbtk::TypeName<bbtk::Data>(); }
    std::string GetHumanTypeName() const { return bbtk::HumanTypeName<bbtk::Data>(); }
    bool IsPointerType() const { return false; }
    void BruteForceSetPointer(bbtk::AtomicBlackBox*, void*) {}
  private:
    ///  Pointer on the Set method  
    SetMethodPointerType mSetMethodPointer;
  };
  //===========================================================================
  

  //===========================================================================
  BBTK_BEGIN_DESCRIBE_BLACK_BOX(MagicBox,bbtk::AtomicBlackBox);
  BBTK_NAME("MagicBox");
  BBTK_AUTHOR("laurent.guigues@creatis.insa-lyon.fr");
  BBTK_CATEGORY("misc");
  BBTK_DESCRIPTION("Takes *any kind* of data and copies it to its output. Is a magic box as any box output can be plugged into it and its output can be plugged into any other box input (dynamic type checking, see below), hence it can be put between **any** two boxes. Type matching between its output and the input of the box(es) to which it is connected is made at *run-time*. The pipeline will be executed if the data types : i) match exactly ii) can be transformed by an adaptor iii) are related pointers, i.e. if the output pointer can be upcasted (static_cast) or downcasted (dynamic_cast) to an input type pointer (see the bbtk::any output connection rule in the guide for details). Important uses of the MagicBox are : 1) performing run-time pointer cast, either upward or backward an object hierarchy 2) perform data adaptation (adaptor creation) at run-time vs. pipeline creation time.");
 
 AddInputDescriptor
  (new bbtk::AtomicBlackBoxInputDescriptor
   (typeid(MagicBoxDescriptor),
    "In","Input data","",
    new bbtk::AtomicBlackBoxTGetFunctor<MagicBox,bbtk::Data,bbtk::Data>
    (&MagicBox::bbGetInputIn),
    new MagicBoxSetFunctor (&MagicBox::bbSetInputIn) ) );

  BBTK_INPUT(MagicBox, Active, "Active True/False (default True)",bool,"");  


  AddOutputDescriptor
  (new bbtk::AtomicBlackBoxOutputDescriptor
   (typeid(MagicBoxDescriptor),
    "Out","Output data","",
    new bbtk::AtomicBlackBoxTGetFunctor<MagicBox,bbtk::Data,bbtk::Data>
    (&MagicBox::bbGetOutputOut),
    new MagicBoxSetFunctor (&MagicBox::bbSetOutputOut) ) );
  BBTK_END_DESCRIBE_BLACK_BOX(MagicBox);
  //===========================================================================

}
// namespace bbstd

#endif // __bbstdMagicBox_h_INCLUDED_H__
