/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#ifndef __creaImageIOTreeNodeComparators_h_INCLUDED__
#define __creaImageIOTreeNodeComparators_h_INCLUDED__

#include <vector>
#include <iostream>

namespace creaImageIO
{

  namespace tree
  {

    
    class Node;
    
    
	/**
	* \ingroup Tree
	*/
  //=====================================================================
    /// Abstract definition of a comparator of Node
  struct Comparator
  {
    virtual ~Comparator() {}
    virtual bool operator() (Node* const & x, Node* const & y) = 0;
  };
  //=====================================================================



  //=====================================================================
    /// Abstract Comparator whose order can be reversed
  struct ComparatorWithOrder : public Comparator
  {
    ComparatorWithOrder(bool reverse_order = false) 
      : mReverseOrder(reverse_order) {}
    virtual ~ComparatorWithOrder() {}


    virtual bool compare(Node* const &, Node* const &) = 0;

    virtual bool operator() (Node* const & x, Node* const & y)
    {
      if (mReverseOrder) return this->compare(y,x);
      return this->compare(x,y);
    };

    bool mReverseOrder;
  };
  //=====================================================================



  //=====================================================================
    /// A Comparator which stores a vector of Comparators and 
			       /// which performs lexicographical comparison
  class LexicographicalComparator : public Comparator
  {
  public:
    LexicographicalComparator(const std::string& name) 
      : mName(name) {}
    ~LexicographicalComparator() {}

    const std::string& GetName() const { return mName; }
    void SetName(const std::string& s) { mName = s; }
    void Clear() { mComparator.clear(); }
    void DeleteComparators() 
    { 
       std::vector<Comparator*>::iterator i;
       for (i =mComparator.begin();
	    i!=mComparator.end();
	    ++i)
	 {
	   delete *i;
	 }
       mComparator.clear(); 
    }
    void Add(Comparator* c) { mComparator.push_back(c); }
    
    bool operator() (Node* const & x, Node * const & y);

  private:
    std::string mName;
    std::vector<Comparator*> mComparator;
  };
  //=====================================================================




  //=================================================================== 
    /// Comparator which compares the values of a given Attribute of the Nodes which is decoded as an int value
  struct IntComparator : 
    public ComparatorWithOrder
  {    
    IntComparator(const std::string& key,
				     bool reverse_order)
      :
      ComparatorWithOrder(reverse_order),
      mKey(key)
    {}
    virtual bool compare(Node* const & x, Node* const & y);

  private:
    std::string mKey;
  };
  //===================================================================

  //===================================================================
    /// Comparator which compares the values of a given Attribute of the Nodes which is decoded as an float value
  struct FloatComparator : 
    public ComparatorWithOrder
  {    
    FloatComparator(const std::string& key,
				       bool reverse_order )
      : 
      ComparatorWithOrder(reverse_order),
      mKey(key)
    {}

    virtual bool compare(Node* const & x, Node* const & y);

  private:
    std::string mKey;
  };
  //===================================================================

  //===================================================================
    /// Comparator which compares the values of a given Attribute of the Nodes which is decoded as a string value
  struct StringComparator : 
    public ComparatorWithOrder
  {    
    StringComparator(const std::string& key,
				   bool reverse_order )
      : 
      ComparatorWithOrder(reverse_order),
      mKey(key)
    {}
    
    virtual bool compare(Node* const & x, Node* const & y);
    
  private:
    std::string mKey;
  };
  //===================================================================

  //===================================================================
#define INT_FIELD_COMP(NAME,FIELD)	\
  struct Node##NAME##Comparator :	\
    public IntComparator	\
  {						\
    Node##NAME##Comparator(bool o = false) :	\
      IntComparator(FIELD,o)	\
    {}						\
  }
  //==================================================================
  
  //===================================================================
#define FLOAT_FIELD_COMP(NAME,FIELD)		\
  struct Node##NAME##Comparator :	\
    public FloatComparator	\
  {						\
    Node##NAME##Comparator(bool o = false) :	\
      FloatComparator(FIELD,o)	\
    {}						\
  }
  //===================================================================

  //===================================================================
#define STRING_FIELD_COMP(NAME,FIELD)		\
  struct Node##NAME##Comparator :	\
    public StringComparator	\
  {						\
    Node##NAME##Comparator(bool o = false) :	\
      StringComparator(FIELD,o)	\
    {}						\
  }
  //===================================================================



  //===================================================================
  // Patient comparators
  ///Compares the names of the patients
  STRING_FIELD_COMP(PatientName,"A0010_0010");
  ///Compares the sex of the patients
  STRING_FIELD_COMP(PatientSex, "A0010_0040");
  ///Compares the birthdays of the patients
  STRING_FIELD_COMP(PatientBirthday, "A0010_0030");
  //===================================================================

  //===================================================================
  // Study comparators
  ///Compares the dates of the studies
  STRING_FIELD_COMP(StudyDate,"A0008_0020");
  ///Compares the description of the studies
  STRING_FIELD_COMP(StudyDescription,"A0008_1030");
  //===================================================================

  //===================================================================
  // Series comparators
  ///Compares the modality of the series
  STRING_FIELD_COMP(Modality,"A0008_0060");
  ///Compares the description of the series
  STRING_FIELD_COMP(SeriesDescription,"A0008_103E");
  ///Compares the date of the series
  STRING_FIELD_COMP(SeriesDate,"A0008_0021");
  //===================================================================

  //===================================================================
  // Image comparators
  ///Compares the number of the images
  INT_FIELD_COMP(ImageNumber,"A0020_0013");
  ///Compares the location of the images
  FLOAT_FIELD_COMP(SliceLocation,"A0020_1041");
  ///Compares the filename of the images
  STRING_FIELD_COMP(FullFileName,"FullFileName");
  //===================================================================
  } // namespace tree

} // namespace creaImageIO



#endif // #ifndef __creaImageIOComparators_h_INCLUDED__
