/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#ifndef __creaImageIOTreeHandlerImageAdder_h_INCLUDED__
#define __creaImageIOTreeHandlerImageAdder_h_INCLUDED__

#include <creaImageIOTreeHandler.h>
#include <creaImageIOSynchron.h>
#include <creaImageIOImageReader.h>
//#include <creaImageIOImageWriter.h>

// Signal/slot mechanism for progress events
#include <boost/signals2/signal.hpp>
#include <boost/bind.hpp>
#if defined(USE_GDCM2)
#include "creaImageIODicomScanner.h"
#endif

namespace creaImageIO
{
	/**
	* \ingroup Model
	*/

  //=======================================================================
  /// Object which can add images files to a TreeHandler. Is able to parse (recursively) a part of a filesystem to look for known images and load their attributes in order to add the images to a Tree (submission via a TreeHandler::AddBranch)

  class TreeHandlerImageAdder
  {
	  
  public:
    //====================================================================
    /// Ctor
    TreeHandlerImageAdder(TreeHandler* tree);
    /// Dtor
    ~TreeHandlerImageAdder();
    /// Sets the TreeHandler
    void SetTreeHandler(TreeHandler* tree) { mTreeHandler = tree;}
	
	/// Sets the synchronizer
	void SetSynchronizer(Synchronizer* s){mSynchronizer=s;}
	/// Sets the synchronizer
	void SetCurrentDatabase(std::string cur){mCurrentDB=cur;}
    //====================================================================

    //====================================================================
    /// Structure which holds progress information
    /// To stop the image adder use SetStop()
    class Progress
    {
    public:
      Progress() { Reset(); }
      ~Progress() {}
      
      void Reset() 
      {
	mStop = false;
	mNumberScannedFiles = 0;
	mNumberScannedDirs = 0;
	mNumberHandledFiles = 0;
	mNumberAddedFiles = 0;
      }

      int GetNumberScannedFiles() const { return mNumberScannedFiles; }
      int GetNumberScannedDirs() const { return mNumberScannedDirs; }
      int GetNumberHandledFiles() const { return mNumberHandledFiles; }
      int GetNumberAddedFiles() const { return mNumberAddedFiles; }

      void IncNumberScannedFiles() { mNumberScannedFiles++; }
      void IncNumberScannedDirs() { mNumberScannedDirs++; }
      void IncNumberHandledFiles() { mNumberHandledFiles++; }
      void IncNumberAddedFiles() { mNumberAddedFiles++; }

      void SetStop() { mStop = true; }
      bool GetStop() const { return mStop; }

    private:
      bool mStop;
      int mNumberScannedFiles;
      int mNumberScannedDirs;
      int mNumberHandledFiles;
      int mNumberAddedFiles;
    };
    //=============================================

    //=============================================
    const Progress& GetProgress() const { return mProgress; }
    //=============================================

    //=============================================
    typedef boost::signals2::signal<void (Progress&)>  ProgressSignalType;
    typedef ProgressSignalType::slot_function_type ProgressCallbackType;
    //=============================================

   //==================================================================
    /// Adds the function f to the list of functions to call 
    /// when the addition progresses.
    /// f is of type ProgressCallbackType which is:
    /// void (*ProgressCallbackType)(Progress&)
    /// To pass a member function 'f' of an instance 'c' of a class 'C' 
    /// as callback you have to 'bind' it, i.e. call:
    /// ConnectProgressObserver ( boost::bind( &C::f , c, _1 ) );
    void ConnectProgressObserver(ProgressCallbackType callback);
   //==================================================================

    //====================================================================
    /// Returns if the file can be read or not
    bool IsHandledFile( const std::string& filename);
    /// Adds a list of files to the TreeHandler 
    void AddFiles( const std::vector<std::string>& filename );
    /// (Recursively) adds the files of a directory to the TreeHandler 
    void AddDirectory( const std::string& directory, 
		       bool recurse);


#if defined(USE_GDCM2)
	 void AddDirectoryRecursorScanner(const std::string &dirpath, 
						   bool recursive,const std::string &addKey,  DicomImageScanner i_sc, bool b_loaded);
#endif

	/// Removes a file from the databases
	void RemoveFile(tree::Node* node);
	/// Removes files from the databases
	void RemoveFiles(const std::vector<tree::Node*>& nodes);

    void remove( tree::Node* i_node);

	/// Synchronizes the DB and disk by repeating the operations the user has done and returns a report
	std::string Synchronize(bool repair, bool checkAttributes);
	///Recursively checks if the directory is synchronized and optionally the state of the attributes
	void CheckSyncDirectory(const std::string &dirpath, 
							bool recursive, 
							bool repair,
							bool checkAttributes,
							std::vector<std::string> &i_ignorefiles,
							std::vector<std::string> & attsModified,
							std::vector<std::string> & newfiles);
	///Copies the files indicated in the vector and updates all databases
	void CopyFiles(const std::vector<std::string>& filenames, const std::string directory  );
	
	///Saves as the files indicated in the vector in a specific directory
	void SaveAs(const std::vector<std::string>& filenames, std::vector<vtkImageData *> i_images);
	///Finds the node that matches the specified parameters
	void FindNode(tree::Node* parent, int level, 
		const std::string& searchParam, 
		const std::string& searchVal, 
		tree::Node*& node);

	///Finds the nodes that partially match the searchVal
	void FindNodePartial(tree::Node* parent, int level, const std::string& searchParam, const std::string& searchVal, tree::Node*& node);
 
	///Checks the attributes of the database against the ones in disk
	void CheckAttributes(bool repair, std::string& file, std::vector<std::string>& attsModified);
	///Deletes the drive with the given name (use for all databases except maintenance and timestamp)
	void DeleteDriveFromMainDB(const std::string& drive);
	///Deletes the drive with the given name (use for maintenance and timestamp databases)
	void DeleteDriveFromOtherDB(const std::string& drive);
	///Edits the given field and sets the new parameters
	void EditField(tree::Node* node, const std::string& name, const std::string& key, const std::string& val);
	///Returns the demanded attributes for the given file
	void GetAttributes(const std::vector<std::string>& params, 
	  const std::string& filename, 
	  std::vector<std::string>& results);

	/// Test if an attribute is referenced for a given treeHandler
	const std::string isAttributeExist(const std::string i_attr);
  
  
 
    //====================================================================

  private:

    /// Adds a single file to the TreeHandler 
    /// **WITHOUT** testing wether it is handled or not 
    /// hence you have to call IsHandledFile before using AddFile!
    void AddFile( const std::string& filename );

    /// Recursive method which does the main job for AddDirectory
    void AddDirectoryRecursor( const std::string& directory, 
			       bool recurse,
			   const std::string &addKey);

    TreeHandler* mTreeHandler;
	Synchronizer* mSynchronizer;
    ImageReader mReader;
	//ImageWriter mWriter;
	std::string mCurrentDB;
    
    Progress mProgress;
    ProgressSignalType mProgressSignal;


  };
  // EO class TreeHandlerImageAdder
  //=======================================================================


 


} // EO namespace creaImageIO

#include <iostream>
inline std::ostream& operator << ( std::ostream& o, 
			    const creaImageIO::TreeHandlerImageAdder::Progress& p)
{
  o << p.GetNumberScannedFiles() << " files - "
    << p.GetNumberScannedDirs() << " dirs - "
    << p.GetNumberHandledFiles() << " handled -"
    << p.GetNumberAddedFiles() << " added";
  return o;
}

// EOF
#endif  

