/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


#ifndef __pCOLOR_BAR__
#define __pCOLOR_BAR__

// -----------------------------------------------------------------------------------------------------------
// WX headers inclusion.
// For compilers that support precompilation, includes <wx/wx.h>.
// -----------------------------------------------------------------------------------------------------------
#include <wx/wxprec.h>
#ifdef __BORLANDC__
#pragma hdrstop
#endif
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

//------------------------------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------------------------------
#include <iostream>
#include <vector>
#include <deque>

#include "marTypes.h"
#include "pColorPoint.h"
#include "LogicalColorBar.h"
//#include "DegradeRectangle.h"
#include "pFigure.h"
#include "wx/colordlg.h"


#define TRIANGLE 3
#define RECTANGLE 4
//in pixels
/*
* Attribute that determines the width of each color point
*/
#define FIGURE_WIDTH 3

BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_EVENT_TYPE( wxEVT_ADDED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_REMOVED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_MOVED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_CHANGED_POINT, -1 )
	DECLARE_EVENT_TYPE( wxEVT_ON_COLOR_BAR, -1 )
END_DECLARE_EVENT_TYPES()


//-------------------------------------------------------------------------------------------------------------
// Enum declarations
//-------------------------------------------------------------------------------------------------------------
/** Command IDs used by pColorPoint */
enum
{
	cntID_ADD_COLOR_POINT = 6000,
	cntID_DEL_COLOR_POINT,
	cntID_CHANGE_COLOR_POINT,
	cntID_DEGRADE_CONTROL,
	cntID_CHANGE_FIGURE,
	cntID_TRIANGLES_UP,
	cntID_TRIANGLES_DOWN,
	cntID_TRIANGLES_LEFT,
	cntID_TRIANGLES_RIGHT,
	cntID_RECTANGLES,
};

/** file pColorBar.h */
class creaMaracasVisu_EXPORT pColorBar :public  wxScrolledWindow{
public:

	//------------------------------------------------------------------------------------------------------------
	// Constructors & Destructors
	//------------------------------------------------------------------------------------------------------------
	/**
	* Creates a colorBar instance 
	* param *parent Container event listener window
	* param _w Width of the color bar drawing area
	* param _h Height of the color bar drawing area
	* param _bar_orientation VERTICAL (false) or HORIZONTAL (true) direction to set 
	*/
	pColorBar (wxWindow *parent, int _w, int _h, bool _bar_orientation);

	~pColorBar ();

	//------------------------------------------------------------------------------------------------------------
	// Color bar proccessed events methods
	//------------------------------------------------------------------------------------------------------------
	void OnSize( wxSizeEvent &WXUNUSED(event) );

	/**
	* Reacts to the mouse movement inside the color bar for moving the clicked color point in the x-axis.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_MOVED_POINT if a point was moved.
	* param event The mouse actioned event
	*/
	void onMouseMove ( wxMouseEvent& event );
	/*
	* Reacts to the event of drawing the bar and draws it
	* param &WXUNUSED(event) The correspondig wxPaintEvent actioned event
	*/
	void onPaint ( wxPaintEvent &WXUNUSED(event) );

	/**
	* Reacts to the cntID_ADD_COLOR_POINT wxCommandEvent and adds a color degrade point to the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_ADDED_POINT if the point was inserted succesfully.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onAddColorPoint ( wxCommandEvent& anEvent );

	/**
	* Reacts to the cntID_DEL_POINT wxCommandEvent and deletes a color degrade point of the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_REMOVED_POINT if the point was succesfully removed.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onDeleteColorPoint ( wxCommandEvent& anEvent );

	/**
	* Reacts to the cntID_CHANGE_COLOR_POINT wxCommandEvent and changes the assigned color to the selected color degrade point of the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_CHANGED_POINT if the point was succesfully removed.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onChangeColourSelectedPoint ( wxCommandEvent& anEvent );

	/**
	* Reacts to the cntID_TRIANGLES_UP wxCommandEvent and changes the assigned figure to -triangles up- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onTrianglesUp_Figure ( wxCommandEvent& anEvent  );
	/**
	* Reacts to the cntID_TRIANGLES_DOWN wxCommandEvent and changes the assigned figure to -triangles down- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onTrianglesDown_Figure ( wxCommandEvent& anEvent  );
	/**
	* Reacts to the cntID_TRIANGLES_LEFT wxCommandEvent and changes the assigned figure to -triangles left- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onTrianglesLeft_Figure ( wxCommandEvent& anEvent  );
	/**
	* Reacts to the cntID_TRIANGLES_RIGHT wxCommandEvent and changes the assigned figure to -triangles right- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onTrianglesRight_Figure ( wxCommandEvent& anEvent  );
	/**
	* Reacts to the cntID_RECTANGLES wxCommandEvent and changes the assigned figure to -rectangles- for the color points of the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onRectangles_Figure ( wxCommandEvent& anEvent  );

	/**
	* Reacts to the cntID_DEGRADE_CONTROL wxCommandEvent and turns on/off the degrade in the color bar.
	* param & anEvent The wxCommandEvent actioned event 
	*/
	void onDegradeControl ( wxCommandEvent& anEvent );

	/**
	* Reacts to the wxEVT_RIGHT_DCLICK wxMouseEvent and adds a color degrade point to the color bar.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_ADDED_POINT if the point was succesfully inserted.
	* param & event The wxMouseEvent actioned event 
	*/
	void onLeftButtonDClick (wxMouseEvent& event);

	/**
	* Shows the popup menu 
	*/
	void onShowPopupMenu (wxMouseEvent& event);

	
	//------------------------------------------------------------------------------------------------------------
	// Other methods
	//------------------------------------------------------------------------------------------------------------

	/**
	* Set active state 
	* param activeNow The new state
	*/
	void setActiveStateTo (bool activeNow);
	
	/**
	* Gets the active state of the bar
	*  return activeState The actual state
	*/
	bool isActive();

	
	/**
	* Gets the real-x value to draw a vertical line
	* return realX_vertical_line The real x value for the vertical line
	*/
	int	getRealX_vertical_line();

	/**
	* Sets the real-x value to draw a vertical line
	* param realX_vertical_line The new real x value for the vertical line
	*/
	void setRealX_vertical_line(int newReal_x);


	/**
	* Method that reinitiates attributes of the color bar to the given points
	* param minRealValue The minimum represented value (real value)
	* param maxRealValue The maximum represented value (real value)
	*/
	void reinitiateColorBar(double minRealValue, double maxRealValue);

	/**
	* Method that reinitiates  attributes of the color bar and set the points given by parameter
	* param newPointsList Pointer to the list of the new points
	*/
	void reinitiateColorBarTo (std::vector<pColorPoint *> pointsVector);

	/**
	*Method that sets a copy of the points of the color bar to the given pointer parameter
	* param pointsList The list for getting the points
	*/
	void getPointsListWithTemps (std::vector<pColorPoint*> &pointsVector);

	/**
	* Method that sets a copy of the points of the color bar to the given pointer parameter, and sets the default values to the needeed attributes, it doesn't
	* includes the teporal points created
	* param pointsVector The list for getting the points
	*/
	void getAddedColorsPointsList (std::vector<pColorPoint*>& pointsVector);

	/**
	* Adds a color degrade point to the color bar.
	* param xRealValue The real xValue of the point
	* param theColour The assigned color for the point
	* param temporalStart Indicates if the inserted point is the temporal startShowing point
	* param temporalEnd Indicates if the inserted point is the temporal startShowing point
	* return Returns true if the point was succesfully inserted.
	*/
	bool addColorPoint (double xRealValue, wxColour theColour/*, bool temporalStart=false, bool temporalEnd=false*/);

	/**
	* Delets a color degrade point to the color bar.
	* param xRealValue The real xValue of the point to delete	
	* return Returns true if the point was succesfully inserted.
	*/
	bool deleteColorPoint (double xRealValue);

	/**
	* Changes the color degrade point color value.
	* Informs the result of the handled event like a wxCommandEvent wxEVT_CHANGED_COLOR_POINT if the point changed its colour.
	* param clickedValue The x-coord pixel value of the point to which the color change interests	
	* return Returns true if the color point was succesfully updated.
	*/
	bool changeColor ( int clickedValue );

	/**
	* Repaints the color bar in direction mode of the orientation assigned 
	* param createTempPoints True for creating temporal points for start and end when a visible range changes
	*/	
	void repaintView (  );
			
	/**
	* Forces the refresh view of the color bar
	*/
	void RefreshForce ();

	void Refresh(bool eraseBackground = true, const wxRect* rect = NULL);
	
	/**
	* Draws the color bar with its degrades
	* param recentlyChangedOrientation indicates if the orientation has been changed before calling this method
	*/
	void drawColorBar ( bool recentlyChangedOrientation);
		
	/**
	* Changes the figure of the color points according to the parameter
	* param theFigure Is the corresponding number for identifying the figure 
	* RECTANGLE = 4,TRIANGLE LEFT = -4, TRIANGLE RIGHT = -6, TRIANGLE UP = -8, TRIANGLE DOWN = -2
	*/
	void changeFigure(int theFigure);
	/**
	* Changes the figure number of edges used for the degreade color points to the indicated one by parameter
	* param figEdges Is the constant that represents the figure number of edges (TRIANGLE | RECTANGLE)
	*/
	void changePointsFigure_Edges (int figEdges);
	
	/**
	* Gets the constant that represents the figure number of edges used for the degreade color points
	* return figureEdges Is the constant that represents the figure number of edges (TRIANGLE | RECTANGLE)
	*/
	int getPointsFigure_Edges ();

	/**
	* Sets the height of the drawing bar area
	* param _h The height to set
	*/
	void setHeight (int _h);

	/**
	* Sets the height of the drawing bar area
	* return _h The height to get
	*/
	int getHeight();

	/**
	* Sets the width of the drawing bar area
	* param _w The width to set
	*/
	void setWidth (int _w);

	/**
	* Gets the width of the drawing bar area
	* param width The width to get
	*/
	int getWidth ();

	/**
	* Sets the height of the containing rectangle bar
	* param _h The height to set
	*/
	void setBarHeight (int _h);

	/**
	* Gets the height  of the containing rectangle bar
	* return bar_height The height to get
	*/
	int getBarHeight();

	/**
	* Sets the width of the containing rectangle bar
	* param _w The width to set
	*/
	void setBarWidth (int _w);

	/**
	* Gets the width of the containing rectangle bar
	* return bar_width The width to get
	*/
	int getBarWidth ();

	/**
	* Sets the orientation of the color bar
	* param _orientation The orientation to set VERTICAL = false, HORIZONTAL = true
	*/
	void setOrientation (bool _orientation);

	/**
	* Gets the orientation of the color bar
	* return bar_orientation The bar orientation assigned
	*/
	bool getOrientation ();

	/**
	* Sets the collection of color points
	* param _points The new points to set, each one of data type pColorPoint ( xValue, wxColour_assigned)
	*/
	void setPoints (std::vector<pColorPoint *> _points);

	/**
	* Gets the last clickedX pixel coord inside the bar.
	* return clickedX The x-coord pixel value
	*/
	int getClickedX();

	/**
	* Sets the last clickedX pixel coord inside the bar.
	* param xCoordPixel The x-coord value to set
	*/
	void setClickedX(int xCoordPixel);

	/**
	* Gets the last clickedY pixel coord inside the bar.
	* return clickedY The y-coord pixel value
	*/
	int getClickedY();

	/**
	* Sets the last clickedY pixel coord inside the bar.
	* param yCoordPixel The y-coord pixel value to set
	*/
	void setClickedY(int yCoordPixel);

	/**
	* Gets the real x value for a givex x-pixel value using the rule real_x = (x_pixel * (maxX_represented_Tshow - minXRepresented) ) / bar_width
	* param x_Pixel The pixel value to convert into real value with respect to the x scale
	* return realX The real-x value corresponding to the xPixel
	*/
	double convertToRealValue ( int x_Pixel );

	/**
	* Gets the x-pixel value for a given x_real value using the rule x_pixel = (x_real * bar_width)/(maxX_represented_Tshow - minX_represented_Tshow)
	* param x_Pixel The pixel value to convert into real value with respect to the x scale
	* return realX The real-x value corresponding to the xPixel
	*/
	int convertToPixelValue ( double x_real );

	/**
	* Gets the selected color and updates the state of the okSelectedColor
	* return selectedColor Is the selected rbg color
	*/
	wxColour getSelectedColour();	

	/**
	* Sets the represented minimum and maximunm values
	* param minRealValue The minimum represented value (real value)
	* param maxRealValue The maximum represented value (real value)
	*/
	void setRepresentedValues ( double minRealValue, double maxRealValue);

	/**
	* Gets the data of the last point moving
	* return pointData Is a pColorPoint with the data of las moved color
	*/
	pColorPoint * getLastMovedColorPoint();

	/**
	* Sets the gap values for the color bar bitmap
	* param gap_x Gap in x
	* param gap_y Gap in y
	*/
	void setGapValues (int gap_x, int gap_y);

	/**
	* Sets the degrade state of the color bar
	* param newState The degrade stare to set 
	*/
	void setDegradeState(bool newState);

	/*
	* Sets the visible range of the bar and repaints the bar according to it, the min value must be less than the max value.
	* param minToShow Is the minimum value to show in the colorbar
	* param maxToShow Is the maximum value to show in the colorbar
	*/
	void setVisibleRange(int minToShow, int maxToShow);

	/**
	* Gets the degrade state of the color bar
	* return doingDegrade is the actual degrade state of the bar
	*/
	bool getDegradeState();

	/**
	* Sets the state of static or not for the limit color points
	* pamar areStatic Is the state to set for the limits
	*/
	void setStaticLimitsTo(bool areStatic);

	/**
	* Gets the state of static or not for the limit color points
	* return staticLimits Is the state for limits
	*/
	bool getStaticLimits();

	/**
	* Sets the device start drawing left-superior (pixel) start point and draws automatically the color bar
	* param deviceStart_x Pixel start for x-coord
	* param deviceStart_y Pixel start for y-coord
	*/
	void setDeviceBlitStart ( wxCoord deviceStart_x, wxCoord deviceStart_y );

	/**
	* Clears the temporal color points of the list
	*/
	void clearTemporalColors();

	/**
	* Gets the device value form the end of this panel to the end of the drawing area in the device in pixels
	* return deviceEndMargin The value asigned to the right margin
	*/
	int	getDeviceEndX();
	

	/**
	* Sets the new device (deviceEndMargin) value form the end of this panel to the end of the drawing area in the device
	* param newDeviceEnd_pixels The new pixel value to asign to the right margin in pixels
	*/
	void setDeviceEndX(int newDeviceEnd_pixels);


	/**
	* Sets the new device (deviceEndMargin) value form the end of this panel to the end of the drawing area in the device
	* param newDeviceEnd_pixels The new pixel value to asign to the right(horizontal view), underneath(vertical view) margin in pixels
	*/
	void setDeviceEndMargin(int newDeviceEnd_pixels);
	
	/**
	* Method that creates and send the given id-event to the event handler
	* param theEventType Is the event id type identification for generating the event
	*/
	void createAndSendEvent(WXTYPE theEventType);


	/**
	* Returns the number of points that the bar color has
	* return
	*/
	int getColorPointsSize();

	/**
	 Get the RGB values of the color point that is in the
	 index given
	*/
	void getDataAt(int index, double& x,int& red,int& green,int& blue);
	

	/**
	* Sets the guide line color
	* param theNwGuideLineColor The color to set to the guideLineColor
	*/
	void setGuideLineColour(wxColour theNwGuideLineColor);

	/**
	* Gets the guide line color
	* return guideLineColor The color of the guideLine
	*/
	wxColour getGuideLineColour();

	/**
	* Sets the background color
	* param theNwBackColor The color to set to the colourParent
	*/
	void setBackGroundColour(wxColour theNwBackColor);

	/**
	* Gets the background color
	* return colourParent The color of the background
	*/
	wxColour getBackGroundColour();

	/**
	* Gets the min value of the color bar
	*/
	double	getMinValue();

	/**
	* Gets the max value of the color bar
	*/
	double	getMaxValue();

	/**
	* Updates the colors of the first nonTemporal point in the list and the last nonTemporal point 
	* That logically corresponds always to the the first and last nodes of the color bar
	*/
	void updateExtremeColors();
		
	void onLeftClicDown(wxMouseEvent& event );

	void onLeftClickUp(wxMouseEvent& event );

	void updatePointsToDraw();

	//------------------------------------------------------------------------------------------------------------
	// Attributes declaration
	//------------------------------------------------------------------------------------------------------------
	
private:

	/*
	*
	*/
	bool acceptedClick;

	/*
	* Represents the logical bar instance
	*/
	LogicalColorBar * _logicalBar;
	
	/*
	* Represents the margin gap on x
	*/
	int gapX;
	
	/*
	* Represents the margin gap on x
	*/
	int gapY;
	/*
	* Represents the height of the color bar drawing area
	*/
	int			height;
	/*
	* Represents the width of the color bar drawing area
	*/
	int			width;

	/*
	* Represents the height  of the bar
	*/
	int			bar_height;

	/*
	* Represents the width of the bar
	*/
	int			bar_width;

	/*
	* Represents the collection of degrade definition points 
	*/
	std::deque<pColorPoint *> showedColorPoints;
	
	/* 
	* The orientation of the rectangule is true when is horizontal, false for vertical
	*/
	//bool		bar_orientation;

	/*
	* The points figure orientation UP, DOWN, LEFT, or RIGTH
	*/ 
	int			points_orientation;

	/*
	* The minimum represented real value used for calculating the scaling for point by the rule real_x = (x_pixel * (maxX_represented_Tshow_Tshow - minX_represented_Tshow) ) / bar_width
	*/
	int			minX_represented_Tshow;

	/**
	* The maximum represented real value used for calculating the scaling for point by the rule real_x = (x_pixel * (maxX_represented_Tshow_Tshow - minXRepresented_Tshow) ) / bar_width
	*/
	int			maxX_represented_Tshow;

	/**
	* The minimum represented real value used for calculating the scaling for point by the rule real_x = (x_pixel * (maxX_represented_Tshow_Tshow - minX_represented_Tshow) ) / bar_width
	*/
	int			temporalMinXToShow;

	/*
	* The maximum represented real value used for calculating the scaling for point by the rule real_x = (x_pixel * (maxX_represented_Tshow_Tshow - minXRepresented_Tshow) ) / bar_width
	*/
	int			temporalMaxXToShow;
	
	/**
	* Indicates the first node to show 
	*/
	pColorPoint *	startNode_show;

	/**
	* Indicates the last node to show
	*/
	pColorPoint *	lastNode_show;
	
	/**
	*  Indicates the amount of temporal initial color points
	*/
    int accumTemporalPoints;

	/**
	* Represents the state of viewing a range
	*/
	bool viewingRange;

	/**
	* Is the constant that represents the figure number of edges (TRIANGLE | RECTANGLE)
	*/
	pFigure*	figure;
	/**
	* Represents the last clickedX coord inside the bar.
	*/
	int			clickedX;

	/*
	* Represents the last clickedY coord inside the bar.
	*/
	int			clickedY;

	/**
	* Represents the movind index of the last moving point in the colorPoints list
	*/
	int			movingPointIndex;

	/**
	* Indicates if in the last colour selection the ok button was clicked
	*/
	bool		okSelectedColor;

	/**
	* Represents the statc or movable state of limit start and end color points, default value true for static
	*/
	bool		staticLimits;

	/**
	* Indicates the last node moving in the coloPoints list
	*/
	pColorPoint *	movingNodePoint;

//	pDegradeRectangle *			degrade;
	
	/**
	* Is the bitmap for the bar color
	*/
	wxBitmap *					colorBar_Bitmap;
	/**
	* Is the bitmap of the information associated to the bar color
	*/
	wxBitmap *					information_Bitmap;
	/**
	* Represents the state of doing or not the degrade in the bar (default value true for doing degrade)
	*/
	bool						doingDegrade;

	/**
	* Represents that the minimum value for the visible range has been found in the original color points
	*/
	bool						minFound;

	/**
	* Represents that the maximum value for the visible range has been found in the original color points
	*/
	bool						maxFound;

	/**
	* X-Coord for left-superior device visible drawing
	*/
	wxCoord						deviceStart_x;	

	/**
	* Y-Coord for left-superior device visible drawing
	*/
	wxCoord						deviceStart_y;

	/*
	* X-Coord for left-superior device margin, important for horizontal view
	*/
	wxCoord						deviceEndMargin;	

	/**
	* Y-Coord for left-superior device margin, important for vertical view
	*/
	wxCoord						deviceEnd_y;
	/**
	* Represents the active state of the bar
	*/
	bool						activeState;

	int						avaliablePointToMove;

	/**
	* Represents the real-x value to draw a vertical line
	*/
	int							realX_vertical_line;

    /**
	* Represents the color of the dot-line use as guides. The default color is RED. 
	*/
	wxColour	guideLineColor;

	 /*
	* Represents the color of the backGround. Default color is the parent color. 
	*/
	wxColour	colourParent;

	/*
	* The color bar pop menu
	*/
	wxMenu c_popmenu; 
	/*
	* The figure changes submenu
	*/
	wxMenu * changesMenu;
	
	
	DECLARE_CLASS(pColorBar)
	// any class wishing to process wxWindows events must use this macro
	DECLARE_EVENT_TABLE()
};


#endif


