/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkBlackBoxDescriptor.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.13 $
=========================================================================*/


/**
 *  \file 
 *  \brief Class bbtk::BlackBoxDescriptor : (abstract) describes of a BlackBox (name, description, author) and is able to create an instance of it.
 */
/**
 * \class bbtk::BlackBoxDescriptor
 * \brief (Abstract) Contains information on a BlackBox type (name, description, author, inputs, outputs) and is able to create an instance of it.
 */

#ifndef __bbtkBlackBoxDescriptor_h__
#define __bbtkBlackBoxDescriptor_h__

#include "bbtkObject.h"
#include "bbtkBlackBoxInputDescriptor.h"
#include "bbtkBlackBoxOutputDescriptor.h"
#include <map>
#include <fstream>
#include "bbtkUtilities.h"
namespace bbtk
{

  class BlackBox;
  class Package;
  BBTK_FORWARD_DECLARE_POINTER(BlackBox);
  BBTK_FORWARD_DECLARE_POINTER(Package);

  //==========================================================================
  class BBTK_EXPORT BlackBoxDescriptor : public Object
  {
    BBTK_ABSTRACT_OBJECT_INTERFACE(BlackBoxDescriptor);
  public:
    /// The type of descriptor of inputs 
    typedef BlackBoxInputDescriptor InputDescriptor;
    /// The type of descriptor of outputs 
    typedef BlackBoxOutputDescriptor OutputDescriptor;
    /// The type of dictionnary of inputs = map of input descriptors pointers 
    typedef std::map<std::string, InputDescriptor*> InputDescriptorMapType;
    /// The type of dictionnary of outputs= map of output descriptors pointers 
    typedef std::map<std::string, OutputDescriptor*> OutputDescriptorMapType;
    /// The kinds of black box
    typedef enum
    {
      STANDARD=0,
      ADAPTOR=1,
      DEFAULT_ADAPTOR=2,
      GUI=3,
      DEFAULT_GUI=4
    }
    Kind;

	typedef enum
	{
		TS_BINARY=0,
		TS_SCRIPT_COMPLEXBOX=1,
		TS_SCRIPT_APPLICATION=2
	} TypeOfScript;
	  
    /// Releases a descriptor
    //    static void Release(BlackBoxDescriptor::WeakPointer);

     /// Creates an instance with name boxname of the BlackBox
    /// of which this is the descriptor 
    virtual BlackBoxPointer NewBlackBox(const std::string& boxname) = 0;

    /// Returns the name of the **TYPE** of the black box
    const std::string& GetTypeName() const { return mTypeName; }
    
    /// Returns the full name of the **TYPE** of the black box (+package name)
    std::string GetFullTypeName() const;
    
    /// Returns the description of the BlackBox 
    const std::string& GetDescription() const { return mDescription; }
    
    /// Returns the author(s) of the BlackBox
    const std::string& GetAuthor() const { return mAuthor; }
    
    /// Returns the category(s) of the BlackBox
    const std::string& GetCategory() const { return mCategory; }
    
    /// Returns the Package to which the box belongs (const pointer)
    PackagePointer GetPackage() const { return mPackage; }
    
    /// Returns the kind of box 
    Kind GetKind() const { return mKind; }

    /// Prints help on the black box
    virtual void GetHelp(bool full=true) const;

    /// Returns a const ref on the map of input descriptors
    const InputDescriptorMapType& GetInputDescriptorMap() const 
      { return mInput; }
      
    /// Returns a const ref on the map of output descriptors
    const OutputDescriptorMapType& GetOutputDescriptorMap() const 
      { return mOutput; }
      
    /// Returns a const pointer on the descriptor of the input of name <name> 
    const InputDescriptor* GetInputDescriptor(const std::string &name) const;

    /// Returns a const pointer on the descriptor of the output of name <name> 
    const OutputDescriptor* GetOutputDescriptor(const std::string &name) const;

    /// Sets the name of the **TYPE** of BlackBox
    void SetTypeName( const std::string& name ) { mTypeName=name; }

    /// Adds the string to the BlackBox description
    void AddToDescription( const std::string&, bool clear = false );
    
    /// Adds the string to the BlackBox author list
    void AddToAuthor( const std::string&, bool clear = false );
    
    /// Adds the string to the BlackBox category list
    void AddToCategory( const std::string&, bool clear = false );
    
    /// Sets the Package to which the box belongs
    void SetPackage(PackagePointer package) { mPackage = package; }
    
    /// Sets the kind of box 
    void SetKind(Kind kind) { mKind = kind; }

    /* 
   virtual void InsertHTMLGraph( std::ofstream& s, int detail, int level, 
				  const std::string& output_dir = "" ) { }
    */
    /// Writes html formatted help into the output file stream. 
    /// detail and level are used for graphical representation of 
    /// complex black boxes (see ComplexBlackBox::InsertHTMLGraph)
    /// output_dir is the directory in which to write auxilliary files 
    /// (.dot/.png/.cmap)
    virtual void InsertHtmlHelp ( std::ofstream& s, 
				  int detail, int level,
				  const std::string& output_dir = "",
				  bool relative_link = false );
    ///
    virtual void Check(bool recursive=true) const;

	  virtual void GetBoxesInside (NodeTreeC& tree, int cont);
	  
	  /// Sets the name of the script file from which it is defined
	  void SetScriptFileName(const std::string& n) { mScriptFileName = n; }
	  /// Gets the name of the script file from which it is defined
	  std::string GetScriptFileName() const { return mScriptFileName; }
	  
	  
	  
	  
	  /// Gets the type of script  BINARY, SCRIPT_COMPLEXBOX, SCRIPT_APPLICATION 
	  TypeOfScript GetTypeOfScript()			{	return mTypeOfScript;	};
	  void SetTypeOfScript(TypeOfScript type)	{	mTypeOfScript=type;		};	  
	  bool IsTypeOfScript_Application();
	  bool IsTypeOfScript_ComplexBox();
	  void SetTypeOfScript_Application();
	  
	  
	  
	  
  private:
	  
	  TypeOfScript mTypeOfScript;
	  
	  /// The name of the script file from which it was created
	  std::string mScriptFileName;
	  
	  
	  
  protected:
    /// Adds an input descriptor
    void AddInputDescriptor( BlackBoxInputDescriptor *d ) 
    { mInput[d->GetName()] = d; }
    /// Adds an output descriptor
    void AddOutputDescriptor( BlackBoxOutputDescriptor *d ) 
    { mOutput[d->GetName()] = d; }

    //  private:
    /// The name of the type of black box
    std::string mTypeName;
    /// The description of the black box
    std::string mDescription;
    /// The author of the black box
    std::string mAuthor;
    /// The category of the black box
    std::string mCategory;    
    /// The kind of box (standard / adaptor)
    Kind mKind;
    /// The Package to which the box belongs
    PackagePointer mPackage;
    /// The inputs 
    InputDescriptorMapType mInput;
    /// The outputs 
    OutputDescriptorMapType mOutput;

  };
  // class BlackBoxDescriptor
  //==========================================================================


 
}
// namespace bbtk
#endif
