/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkFactory.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.19 $
=========================================================================*/

                                                                      

/**
 *\file
 *\brief Class bbtk::Factory : can load and unload dynamic libraries containing black boxes packages and create instances of the black boxes registered in the packages loaded.
 */
/**
 *\class bbtk::Factory
 *\brief Can load and unload dynamic libraries containing black boxes packages and create instances of the black boxes registered in the packages loaded.
 *
 */


#ifndef __bbtkFactory_h__
#define __bbtkFactory_h__

#include "bbtkPackage.h"

namespace bbtk
{

  class Executer;
  BBTK_FORWARD_DECLARE_POINTER(Executer);

  class BBTK_EXPORT Factory : public Object
  {
    BBTK_OBJECT_INTERFACE(Factory);
    typedef Object Superclass;
  public:
    static Pointer New();

    // @name Packages management
    //@{
    /// Pushes back the names of the Package s which are in the Factory
    void GetPackagesList(std::vector<std::string>&);
    /// Loads a dynamic library which contains a Package 
    void LoadPackage( const std::string& name );
    /// Unloads a Package which was loaded from a dynamic library
    void UnLoadPackage( const std::string& name );
    /// Inserts a Package in the Factory
    void InsertPackage( Package::Pointer );
    /// Removess a Package from the Factory
    void RemovePackage( Package::Pointer );
    /// Returns the pointer on a Package provided by name
    Package::Pointer GetPackage(const std::string& name) const;
    /// The type of map of packages
    typedef std::map< std::string, Package::Pointer > PackageMapType;
    /// Returns the map of Packages
    const PackageMapType& GetPackageMap() const { return mPackageMap; }
    //@}

    // @name Methods which print help
    //@{
    /// Prints the list of Package
    void PrintHelpListPackages(bool details = true, 
			       bool adaptors = false) const;
    /// Prints help on a particular Package
    void PrintHelpPackage(const std::string& name, 
			  bool adaptors = false) const;
    /// Prints help on a BlackBoxDescriptor
    void PrintHelpDescriptor(const std::string& name, 
			     std::string& package,
			     bool full=true ) const;
    
    void ShowGraphTypes(const std::string& name) const;
    // @}


    /// @name Object creation methods
    //@{
    BlackBox::Pointer NewBlackBox(const std::string& type, 
                          const std::string& name) const;
    
    BlackBox::Pointer NewAdaptor(const DataInfo& typein,
			 const DataInfo& typeout,
                         const std::string& name) const;

    BlackBox::Pointer NewWidgetAdaptor(const DataInfo& typein,
			       const DataInfo& typeout,
			       const std::string& name) const;

    Connection::Pointer NewConnection(BlackBox::Pointer from,
				      const std::string& output,
				      BlackBox::Pointer to,
				      const std::string& input) const;
   //@}

    /// @name Package inspection methods
    /// Allow to test if an Adaptor is present in the packages WITHOUT instanciating it
    //@{
    bool FindAdaptor(const DataInfo& typein,
		     const DataInfo& typeout,
		     std::string& adaptor) const;

    bool FindWidgetAdaptor(const DataInfo& typein,
			   const DataInfo& typeout,
			   std::string& adaptor) const;
    
    bool FindWidgetAdaptor2(const DataInfo& typein,
			    const DataInfo& typeout,
			    std::string& widget,
			    std::string& adaptor) const;
    //@}
  
    void WriteDotFilePackagesList(FILE *ff);

    /// Releases all the packages of the Factory
    void Reset();
    
    void Check() const;

    typedef enum
      {
	Packages,
	Categories,
	Initials,
	Adaptors
      }
      IndexEntryType;
    void CreateHtmlIndex(IndexEntryType type, const std::string& filename);

    /// Sets the executer who created the factory (if any)
    void SetExecuter(ExecuterPointer e) { mExecuter = e; }
    /// Gets the executer who created the factory (if any)
    ExecuterPointer GetExecuter() { return mExecuter.lock(); }
    /// Gets the executer who created the factory (if any) - const
    //    const Executer::Pointer GetExecuter() const { return mExecuter.lock(); }

	std::string GetPackageNameOfaBlackBox(std::string boxType);
	std::string GetTypeNameOfaBlackBox(std::string boxType);


  private:



    /// The map of packages
    PackageMapType mPackageMap;

    /// The executer which created the factory (if any)
    ExecuterWeakPointer mExecuter;

    bool DoLoadPackage(std::string libname,
		       std::string pkgname,
		       std::string path);
    void CloseAllPackages();
    void ClosePackage(PackageMapType::iterator& i);



  };
  // class Factory



}// namespace bbtk



#endif

