/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ 
*/ 

/*! \file
  \brief Class creaMessageManager and Macros for outputing messages in crea
  There are 4 kinds of messages :
  - Messages (normal messages)
  - Debug messages (not compiled in release)
  - Warnings 
  - Errors
  There are also "types" of messages which are strings which identify the nature of the message 
  (for example : "Kernel" messages are generated by the core classes of the library, there can be a type of 
  message for each type of Node, and so on...)
  A type of message must be declared by registering it into the MessageManager. This is done by a line like :
  crea::MessageManager::RegisterMessageType("Kernel","Messages generated by the core classes of the library",5);
  where : 
  -The first string is the type of the message (the category which will be used to generate a message of this type)
  -The second string is help string
  -The integer is the initial level for the messages of this type (see below).
  To generate a message of a known type then use one of the macros :
  creaMessage, creaDebugMessage, creaWarning, creaError or their variants.
  example :
  creaMessage("Kernel",4,"problem with "<<GetName()<<creaendl);
  will push the 3rd argument in std::cout if the message level of "Kernel" messages is greater or equal to 4.
  which means that it generates a message of level 4 (0 : very important/always displayed ... 9 : deep debug message).
  At run time, one is able to change the level of the messages displayed by using a command like :
  crea::MessageManager::SetMessageLevel("Kernel",5); 
  which tells the manager to display all Kernel messages of level up to 5.
  Variants :
  crea*Cont : continues a previous creaMessage on the same line (without rewriting the type and level)
  crea*Inc / Dec : displays the message and then increments/decrement the messages tabulation 
*/

  //===========================================================
  /**
     \class crea::MessageManager
     \brief Manages the messages displayed by crea
  */

#ifndef __creaMessageManager_h__
#define __creaMessageManager_h__

// The do { } while(0) statement in macros is made to "swallow the semicolon" 
// see http://gcc.gnu.org/onlinedocs/cpp/Swallowing-the-Semicolon.html#Swallowing-the-Semicolon
#include "creaSystem.h"
#include "creaRTTI.h" // for CREA_GET_CURRENT_OBJECT_NAME
// Signal/slot mechanism for message events
#include <boost/signals2/signal.hpp>
#include <boost/bind.hpp>
#include <string>
#include <map>
#include <iostream>
#include <sstream>
// Comment out these symbols to prevent compilation 
//#define CREA_COMPILE_MESSAGES
//#define CREA_COMPILE_DEBUG_MESSAGES
//#define CREA_COMPILE_WARNING_MESSAGES
//#define CREA_COMPILE_ERROR_MESSAGES
#define creaOnMessageLevel(key,value)			\
  int __creaOnMessageLevelVariable =			\
    crea::MessageManager::GetMessageLevel(key);		\
  if ( __creaOnMessageLevelVariable<0)			\
    {							\
      creaWarning("message type '"<<key<<"' unknown");	\
    }							\
  else if (value<= __creaOnMessageLevelVariable) 

#ifdef CREA_PREPEND_MESSAGE_WITH_CODE
#define creaMessageCode				\
  key[0] << key[1] << key[2] << value << " "
#else 
#define creaMessageCode ""
#endif 

#ifdef CREA_PREPEND_MESSAGE_WITH_TAB
#define creaMessageTab				\
  crea::MessageManager::GetTab()
#else 
#define creaMessageTab ""
#endif

//#define CREA_PREPEND_MESSAGE_WITH_SPACE
#ifdef CREA_PREPEND_MESSAGE_WITH_SPACE
#define creaMessageSpace(value)			\
  crea::MessageManager::GetSpace(value)
#else 
#define creaMessageSpace(value) ""
#endif

 



//===========================================================

#ifdef CREA_COMPILE_MESSAGES

// Macro for messages
#define creaMessage(key,value,MESSAGE)			\
  do {							\
    creaOnMessageLevel(key,value)			\
      {							\
	std::ostringstream s;				\
	s << creaMessageCode				\
	  << creaMessageTab				\
	  << creaMessageSpace(value)			\
	  << MESSAGE;					\
	crea::MessageManager::SendMessage(key,s.str());	\
      }							\
  }							\
  while (0)



// Macro for continuing a message (when one wants to split the macro
// call into multiple lines)
#define creaMessageCont(key,value,MESSAGE)		\
  do							\
    {							\
      creaOnMessageLevel(key,value)			\
	{						\
	std::ostringstream s;				\
	s << MESSAGE;					\
	crea::MessageManager::SendMessage(key,s.str());	\
	}						\
    }							\
  while (0)


#define creaMessageInc(key,value,MESSAGE)		\
  do							\
    {							\
	std::ostringstream s;				\
	s << creaMessageCode				\
	  << creaMessageTab				\
	  << creaMessageSpace(value)			\
	  << MESSAGE;					\
	crea::MessageManager::SendMessage(key,s.str());	\
	crea::MessageManager::IncTab();			\
	}						\
    }							\
  while (0)


#define creaMessageDec(key,value,MESSAGE)			\
  do								\
    {								\
      creaOnMessageLevel(key,value)				\
	{							\
	  crea::MessageManager::DecTab();			\
	  std::ostringstream s;					\
	  s << creaMessageCode					\
	    << creaMessageTab					\
	    << creaMessageSpace(value)				\
	    << MESSAGE;						\
	  crea::MessageManager::SendMessage(key,s.str());	\
	}							\
    }								\
  while (0)


#define creaDecTab(key,value)			\
  do						\
    {						\
      creaOnMessageLevel(key,value)		\
	{					\
	  crea::MessageManager::DecTab();	\
	}					\
    }						\
  while (0)


#define creaIncTab(key,value)			\
  do						\
    {						\
      creaOnMessageLevel(key,value)		\
	{					\
	  crea::MessageManager::IncTab();	\
	}					\
    }						\
  while (0)


#define creaResetTab()				\
  do						\
    {						\
      crea::MessageManager::ResetTab();		\
    }						\
  while (0)

#else
#define creaMessage(key,value,MESSAGE)
#define creaMessageInc(key,value,MESSAGE)
#define creaMessageDec(key,value,MESSAGE)
#define creaMessageCont(key,value,MESSAGE)
#define creaDecTab(key,value)
#define creaIncTab(key,value)
#define creaResetTab()
#endif

//===========================================================







//===========================================================

// Macros for debug messages

#ifdef CREA_COMPILE_DEBUG_MESSAGES

#define creaDebugMessage(key,value,MESSAGE) creaMessage(key,value,MESSAGE)

#define creaDebugMessageCont(key,value,MESSAGE)	creaMessageCont(key,value,MESSAGE)

#define creaDebugMessageInc(key,value,MESSAGE) creaMessageInc(key,value,MESSAGE)  

#define creaDebugMessageDec(key,value,MESSAGE) creaMessageDec(key,value,MESSAGE)

#define creaDebugDecTab(key,value) creaDecTab(key,value)

#define creaDebugIncTab(key,value) creaIncTab(key,value)

#define creaDebugResetTab() creaResetTab()

#else

#define creaDebugMessage(key,value,MESSAGE) 

#define creaDebugMessageCont(key,value,MESSAGE) 

#define creaDebugMessageInc(key,value,MESSAGE)

#define creaDebugMessageDec(key,value,MESSAGE) 

#define creaDebugDecTab(key,value)

#define creaDebugIncTab(key,value)

#endif

//===========================================================



//===========================================================

#ifdef CREA_COMPILE_WARNING_MESSAGES
#define creaWarning(MESSAGE)						\
  do									\
    {									\
      int lev = crea::MessageManager::GetMessageLevel("warning");	\
      if (lev >0)							\
	{								\
	  std::cerr << "!! WARNING !! " << MESSAGE << std::endl;	\
	  if (lev >1)							\
	    {								\
	      std::cerr << "!! WARNING !! In file '"<<__FILE__		\
			<<"' ; Line "<<__LINE__<<std::endl;		\
	    }								\
	}								\
    }									\
  while (0) 

#else
#define creaWarning(MESSAGE) 
#endif

//===========================================================





//===========================================================

#ifdef CREA_COMPILE_ERROR_MESSAGES
//#include "creaWx.h"
#define creaError(MESSAGE)				\
  do							\
    {							\
      std::ostringstream s;				\
      s << MESSAGE;					\
      std::ostringstream f;				\
      f << __FILE__ << " (l."<<__LINE__<<")";		\
      crea::Exception e( CREA_GET_CURRENT_OBJECT_NAME,	\
			f.str(),			\
			s.str());			\
      throw e;						\
    }							\
  while (0) 



#define creaGlobalError(MESSAGE)				\
  do							\
    {							\
      std::ostringstream s;				\
      s << MESSAGE;					\
      std::ostringstream f;				\
      f << __FILE__ << " (l."<<__LINE__<<")";		\
      crea::Exception e( "global scope",		\
			f.str(),			\
			s.str());			\
      throw e;						\
    }							\
  while (0) 



#define CREA_INTERNAL_ERROR_MESSAGE \
  "\n\n***********************************************\n**** THIS IS AN INTERNAL ERROR TO crea     ****\n**** Please send a full bug report to :    ****\n****  creatools@creatis.insa-lyon.fr ****\n***********************************************\n\n"

#define creaInternalError(MESSAGE)			\
  do							\
    {							\
      std::ostringstream s;				\
      s << MESSAGE << CREA_INTERNAL_ERROR_MESSAGE;	\
      std::ostringstream f;				\
      f << __FILE__ << " (l."<<__LINE__<<")";		\
      crea::Exception e( CREA_GET_CURRENT_OBJECT_NAME,	\
			 f.str(),			\
			 s.str());			\
      throw e;						\
    }							\
  while (0) 



#else
#define creaError(MESSAGE)
#define creaGlobalError(MESSAGE)
#define creaInternalError(MESSAGE)
#endif

//===========================================================



//===========================================================

#define creaendl std::endl

//===========================================================





namespace crea 

{

  

  class CREA_EXPORT MessageManager

  {

  public:

    //=============================================

    typedef boost::signals2::signal<void (const std::string&)>  MessageSignalType;

    typedef MessageSignalType::slot_function_type MessageCallbackType;

    //=============================================

    ///

    MessageManager();

    ///

    ~MessageManager();

    ///

    static MessageManager* GetInstance();

    ///

    static void RegisterMessageType(const std::string& key, 

                                    const std::string& help,

                                    unsigned char default_level = 9);

    ///

    static void SetMessageLevel(const std::string& key, unsigned char level);

    ///

    static int GetMessageLevel(const std::string& key);

    ///  

    static void SendMessage(const std::string& key, const std::string& mess);

    ///

    static void AddMessageObserver(const std::string& key, MessageCallbackType callback );

    ///

    static void SendMessagesToCout(bool v = true);

    ///

    static std::string& GetTab() { static std::string s; return s; }

    ///

    static std::string GetSpace(int n) { 

      std::string s; s.insert(0,"                ",n); return s; }

    ///

    static void IncTab() { GetTab() += std::string(" "); }

    ///

    static void DecTab() { GetTab() = GetTab().substr(0,GetTab().length()-1); }

    ///

    static void ResetTab() { GetTab() = std::string(""); }

    ///

    static void PrintInfo();

    /// 

    

  private:

    struct MessageType

    {

      MessageType(int l, const std::string& h) : Level(l), Help(h) {}

      int Level;

      std::string Help;

      MessageSignalType Signal;

    };

    typedef std::map<std::string,MessageType*> MessageMapType;

    MessageMapType mMessageMap;

    unsigned int mMaxMessageLength;

    bool mSendToCout;

  };

  //===========================================================

  

}



#include "creaException.h"



#endif

