/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#ifndef __COMMANDS_HANDLER__
#define __COMMANDS_HANDLER__

//------------------------------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------------------------------

#include "CommandObject.h"
#include "CommandsRegisterStructure.h"
#include "ContourWorkspace.h"
#include "ICommandsUser.h"

class ContourWorkspace;

class CommandsHandler{

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------
public:


	/*
	* Constructs the CommandsHandler
	*/
	CommandsHandler();

	/*
	* Destructs the CommandsHandler
	*/
	~CommandsHandler();
//------------------------------------------------------------------------------------------------------------
// Methods
//------------------------------------------------------------------------------------------------------------

	/*
	* Registers in the vectors of doneActions and unDoActions the given commands that all ready corresponds each other to the inverse of the otherone. 
	* If is the first registered action notifies the posibleUNDO avaliability.
	* @param doneAction Is the action to register in the redo_actions vector.
	* @param unDoAction Is the action to register in the unDo_actions vector.
	*/
	void registerCommand(CommandObject* doneAction, CommandObject* unDoAction);

	/*
	* Undo a command. Managing the correspondig vectors and the execution of the inverse action to the - ACTUAL DONE- action
	* @return Returns true if the inverse command ( the actual to execute in UNDO actions ) is executed. If it is false the state of the vector must not change.
	*/
	bool undo();

	/*
	* Redo a command. Managing the correspondig vectors and the execution of the inverse action to the - ACTUAL DONE- action
	* @return Returns true if the actual command to execute ( the actual to execute in REDO actions )has been executed. If it is false the state of the vector must not change.
	*/
	bool redo();

	/*
	* Notitify if posibleREDO is posible or not.
	* @return Returns the state of posibleUNDO
	*/
	bool isPosibleUNDO();

	/*
	* Indicates if posibleUNDO is posible or not.
	* @return Returns the state of posibleREDO
	*/
	bool isPosibleREDO();

	/*
	* Sets  posibleREDO state.
	* @param UNDOstate The state of posibleUNDO to set
	*/
	void setPosibleUNDO(bool UNDOstate);

	/*
	* Sets posibleUNDO state.
	* @param REDOstate The state of posibleREDO to set
	*/
	void setPosibleREDO(bool REDOstate);
   	
	/*
	* Clear the registered actions in the DO and UNDO vectors.
	*/
	void clearActions();

	/*
	* Returns hoy mane paired commands had been registered, if the done and unDo vectors dont match returns -1 as error code.
	* @return Returns how many paired-commands had been registered
	*/
	int getTotalCommands();

	/*
	* Gets the actual command in the UNDO-list
	* @return  Returns a pointer to the actual undo action
	*/
	CommandObject * getActual_UNDO();

	/*
	* Gets the actual command in the REDO-list
	* @return  Returns a pointer to the actual do action
	*/
	CommandObject * getActual_REDO();

	/*
	* Gets the command at the given position in the DO (REDO) vector
	* @return The pointer to the referenced object by the position
	*/
	CommandObject * get_DO_CommandAt(int position);

	/*
	* Gets the command at the given position in the UNDO vector
	* @return The pointer to the referenced object by the position
	*/
	CommandObject * get_UNDO_CommandAt(int position);

	
	/*
	* Validates if it is posible of not to do UNDO and/or REDO and sets the corresponding values
	*/
	void validateOperationsAvaliability();

	/*
	* Sets the model parent of the action handler
	* @param theModelParent The boss reference
	*/
	void setModelBoss(ICommandsUser * theModelParent);


//------------------------------------------------------------------------------------------------------------
// Constants
//------------------------------------------------------------------------------------------------------------

private: 	
//------------------------------------------------------------------------------------------------------------
// Attributes
//------------------------------------------------------------------------------------------------------------
	CommandsRegisterStructure * redo_actions;

	CommandsRegisterStructure * unDo_actions;

	bool posibleUNDO;

	bool posibleREDO;

	ICommandsUser * theWorksSpaceBoss;

	bool isLastREDO_executed;

	bool isLastUNDO_executed;

	std::deque<CommandObject *> executionQueue;	

};
#endif
