/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#ifndef __CONTOUR_WORKSPACE__
#define __CONTOUR_WORKSPACE__


//------------------------------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------------------------------

#include <iostream>
#include <vector>
#include  "ICommandsUser.h"
#include  "OutlineModelManager.h"
#include  "OutlineGroup.h"
#include  "CommandObject.h"
#include  "CommandsHandler.h"
#include  "ICommandsUser.h"

class OutlineModelManager;
class CommandsHandler; 

class ContourWorkspace : public ICommandsUser{

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------
public:

	/*
	* Constructs the workspace with the corresponding given parent
	* @param parent Is the parent relation with
	* @return Returns a pointer to the created ContourWorkspace
	*/
	ContourWorkspace (OutlineModelManager * parent);

	/*
	* Destroys the ContourWorkspace
	*/
	~ ContourWorkspace();

//------------------------------------------------------------------------------------------------------------
// Implamented methods from ICommandsUser.h
//------------------------------------------------------------------------------------------------------------

	/*
	* Executes a command by resending the received parameters to its parent to do the appropiate interpretation and execution.
	* @param theCommand Is the command to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/
	virtual bool executeCommand(CommandObject * theCommand, bool fromRegistration=false);
	
	/*
	* Executes a command queue by resending the received parameters to its parent to do the appropiate interpretation and execution.
	* @param executionQueue Is the command queue to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/
	virtual bool executeCommandsQueue(std::deque<CommandObject *> & executionQueue, bool fromRegistration);

//------------------------------------------------------------------------------------------------------------
// Methods
//------------------------------------------------------------------------------------------------------------

	/**
	* Method that propagates an outline over the deep concept of the axe
	*/
	void spreadInDepth();

	/*
	* Sets the working group id and pointer
	* @param theWorkingGroupKey Is the correspondig key (id) of the working group 
	* @param theGroup Is the correspondig group to work with
	*/
	void setWorkingGroup( int theWorkingGroupKey , OutlineGroup * theGroup);

	/*
	* Gets the working group key
	* @return theGroupKey Is the correspondig key (id) of the working group 
	*/
	int getWorkingGroupKey( );

	/*
	* Calls the execution for UNION of the given outlines
	* @param outlinesNamesToCall Is the correspondig vector with the names of the outlines to operate
	*/
	void callOutline_Union_Of( std::vector<std::string> outlinesNamesToCall );

	/*
	* Calls the execution for INTERSECTION of the given outlines
	* @param outlinesNamesToCall Is the correspondig vector with the names of the outlines to operate
	*/
	void callOutline_Intersection_Of( std::vector<std::string> outlinesNamesToCall );
	
	/*
	* Calls the execution for COMBINATION of the given outlines
	* @param outlinesNamesToCall Is the correspondig vector with the names of the outlines to operate
	*/
	void callOutline_Combination_Of( std::vector<std::string> outlinesNamesToCall );

	/*
	* Calls the execution for FRAGMENTATION of the given outlines
	* @param outlinesNamesToCall Is the correspondig vector with the names of the outlines to operate
	*/
	void callOutline_Fragmentation_Of( std::vector<std::string> outlinesNamesToCall );
	
	/*
	* Calls the execution for AGRUPATION of the given outlines
	* @param outlinesNamesToCall Is the correspondig vector with the names of the outlines to operate
	*/
	void callOutline_Agrupation_Of( std::vector<std::string> outlinesNamesToCall );

	/*
	* Calls to redo a the actual REDO command. 
	* @return Returns true if the actual command to execute has been executed. 
	*/
	bool callREDO();

	/*
	* Calls to undo the actual UNDO command. 
	* @return Returns true if the inverse command is executed. 
	*/
	bool callUNDO();
	
	/*
	* Calls the registration of the given commands (do, undo) in the commands handler
	* If is the first registered command notifies the posibleUNDO avaliability.
	* @param doCommand Is the command to register in the redo_commands vector.
	* @param unDoCommand Is the command to register in the unDo_commands vector.
	*/
	void callRegisterCommand(CommandObject * doCommand, CommandObject * unDoCommand);

	/*
	* Gets the command handler
	*/
	CommandsHandler * getCommandHandler();

	void setCommandHandler(CommandsHandler * aCommHandler);

//------------------------------------------------------------------------------------------------------------
// Constants
//------------------------------------------------------------------------------------------------------------

private: 	
//------------------------------------------------------------------------------------------------------------
// Attributes
//------------------------------------------------------------------------------------------------------------

	std::vector<std::string> selectedOutlineskNames;

	std::string actualOutlinekName;

	int workingGroupKey;

	OutlineModelManager * theModelBoss;
	
	CommandsHandler * commHandler;

	OutlineGroup * actualGroup;

};
#endif
