/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#ifndef __OTULINE_GROUP__
#define __OTULINE_GROUP__


//------------------------------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------------------------------
#include <iostream>
#include <vector>

class OutlineGroup{

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------
public:

	/*
	* Constructs an outline group with the given name
	* @param theName Is the name for the group
	* @param theGroupType Is the type for the group corresponding to one of the constants of this class
	*/ 
	OutlineGroup(std::string theName, int theGroupType);
	
	/*
	* Destroyes the outline and its dependencies
	*/
	~ OutlineGroup();

//------------------------------------------------------------------------------------------------------------
// Methods definition
//------------------------------------------------------------------------------------------------------------

	/*
	* Indicates if a given name of an outline is member of the group or not
	* @param aKeyName Is the name of an outline to search for
	*/
	bool isMemberOfGroup(std::string aKeyName);

	/*
	* Removes an outline with the given name from the group 
	* @param theNameToRemove Is the name to remove from member name list
	* @param allOcurrencies Indicates to errase all ocurrences
	*/
	void removeOutline(std::string theNameToRemove,  bool allOcurrencies = true);
	
	/*
	* Adds an outline with the given name to the group members list
	* @param theNameNw Is the name to add to the group
	*/
	void addOutline(std::string theNameNw);

	/*
	* Gets the name of the group
	* @return name Is the name of the group
	*/ 
	std::string getName();

	/*
	* Sets the name of the group as the given one 
	* @param name Is the new name of the group
	*/ 
	void setName(std::string theNwName);

	/*
	* Gets the state of visiblility (true:visible) or not of the group
	* @return visibleGroup Is the corresponding state
	*/ 
	bool getIfVisibleGroup();

	/*
	* Sets state of visible (true) or not of the with the given one 
	* @param theNwVisiblity Is the corresponding state
	*/ 
	void setIfVisibleGroup(bool theNwVisiblity);

	/*
	* Gets the state of static (true:static) or not of the group
	* @return staticGroup Is the corresponding state
	*/ 
	bool getIfStaticGroup();

	/*
	* Sets state of static (true) or not of the with the given one 
	* @param theNwStatic Is the corresponding state
	*/ 
	void setIfStaticGroup(bool theNwStatic);

	/*
	* Gets the state of selection (true:selected) or not of the group
	* @return selecetedGroup Is the corresponding state
	*/ 
	bool getIfSelectedGroup();

	/*
	* Sets state of visible (true) or not of the with the given one 
	* @param theNwSelected Is the corresponding state
	*/ 
	void setIfSelectedGroup(bool theNwSelected);

	/*
	* Gets the state of edition (true:editing) or not of the group
	* @return editingGroup Is the corresponding state
	*/ 
	bool getIfEditingGroup();

	/*
	* Sets state of editing (true) or not of the with the given one 
	* @param theNwEditing Is the corresponding state
	*/ 
	void setIfEditingGroup(bool theNwEditing);

	/*
	* Gets the total count of outlines in the group
	* @return totalCount Is the corresponding number of elements
	*/ 
	int getGroupType();

	/*
	* Sets the group type 
	* @param theType Is the corresponding new type to assign 
	*/ 
	void setGroupType(int theType);

	/*
	* Gets the group type 
	* @return type Is the corresponding number of elements
	*/ 
	int getOutlinesCount();

	/*
	* Sets the total count of outlines in the group
	* @param theNwVisiblity Is the corresponding state
	*/ 
	void setIfEditingGroup(int theTotal);

	/*
	* Adds an outline to the group as propagation type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void addOutline_PropagationType(std::string theOutlineKeyName);

	/*
	* Adds an outline to the group as plane section type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void addOutline_PlaneSectionType(std::string theOutlineKeyName);

	/*
	* Adds an outline to the group as overlaped type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void addOutline_OverlapedType(std::string theOutlineKeyName);
	
	/*
	* Adds an outline to the group as strip type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void addOutline_StripType(std::string theOutlineKeyName);

	/*
	* Adds an outline to the group as manual type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void addOutline_ManualType(std::string theOutlineKeyName);


	/*
	* Gets the outlines of the group
	* @return Returns the names of the outlines that belong to the group
	*/
	 std::vector< std::string >  getGroupOutlinesNames ( );

//------------------------------------------------------------------------------------------------------------
// Attributes
//------------------------------------------------------------------------------------------------------------

private: 	

	bool acceptsRepetedOutlines;

	/*
	* Represents the name of the group
	*/
	std::string name;

	/*
	* Represents the state of visible for the outlines in the group
	*/
	bool visibleGroup;

	/*
	* Represents the state of selection for the outlines in the group
	*/
	bool selectedGroup;

	/*
	* Represents the state of edition for the outlines in the group
	*/
	bool editingGroup;

	/*
	* Represents the state of static for the outlines in the group
	*/
	bool staticGroup;

	/*
	* Represents the type of the group
	*/
	int groupType;

	/*
	* Represents the total elements count of the group
	*/
	int totalCount;

	/*
	* Represents the outlines of the group
	*/
	std::vector< std::string > outlines_keyNames;

	//------------------------------------------------------------------------------------------------------------
	// Constants
	//------------------------------------------------------------------------------------------------------------
	
public :
	enum
		{
			PROPAGATION = 0,
			PLANE_SECTION = 1,
			OVERLAPED = 2,
			STRIP = 3,		
			MANUAL_GROUP = 4,		
		};
		

};
#endif
