/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#ifndef __PREFIX_MAX__KEYGENERATOR_
#define __PREFIX_MAX__KEYGENERATOR_

//------------------------------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------------------------------
#include <iostream>
#include <string>
#include <sstream>
#include <stdio.h>
#include <stdlib.h>
#include <map>

#include "KeyThing.h"

/*
* Class that manages the creation of std::string keys of multiple objects independently, identified each by a unique name.
* Format of the key is: <prefixOfTheKeyThing> <maxOfTheKeyThing>
*/
class PrefixMaxKeyGenerator{
//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------
public:

	/*
	* Creates the prefix+max key generator	
	*/
	PrefixMaxKeyGenerator();

	/*
	* Destroys the outline manager
	*/
	~PrefixMaxKeyGenerator();

//------------------------------------------------------------------------------------------------------------
// Public Methods
//------------------------------------------------------------------------------------------------------------

	/*
	* Adds a key thing to the keyThings building the respective KeyThing (new keyThing). 
	* @param theName Is the name of the new keyThing to include. If the name is not unique, returns false.
	* @param thePrefix Is the prefix of the new keyThing for the key generation correponding to the new keyThing
	* @param theMax Is the maximum value for the key generation correponding to the new keyThing
	* @return Returns true if the keyThing could be added of not.
	*/
	bool addKeyThing( std::string theName, std::string thePrefix, int theMax=0 );

	/*
	* Remove a key thing  
	* @param theName Is the name of the keyThing to remove. 
	*/
	void removeKeyThing( std::string theName );

	/*
	* Indicates if a key thing existis in the generator
	* @param theName Is the name of the keyThing to search. 
	* @return Returns true if the keyThing exists in the keyThings.
	*/
	bool existsKeyThing( std::string theName );

	/*
	* Updates the maximum value of a key thing if necesary (posibleMax>theMaxOfKeyThing). If the key thing doesn't exist nothing is done.
	* @param theName Is the name of the keyThing to update. 	
	* @param posibleMax Is the number that corresponds to a posible max value of the keyThing to update. 
	*/
	void updateMaxTo( std::string theName, int posibleMax );

	/*
	* Generates a (std::string) key for a given keyThing. If the key thing doesn't exist nothing is done and returns false.
	* @param theName Is the name of the keyThing to search. 
	* @param theInputString Is string to load the generated key formed like <prefixOfTheKeyThing> <maxOfTheKeyThing>
	* @return theKey Returns true if the key was generated successfully. (If theName is an existent keyThing)
	*/
	bool generateKeyOf( std::string theName , std::string &theInputString );

	/*
	* Generates a (std::string) key for a given keyThing and updates the max value of it if necesary. If the key thing doesn't exist nothing is done. 
	* @param theName Is the name of the keyThing to search. 
	* @param posibleMax Is the number that corresponds to a posible max value of the keyThing to update. 
	* @param theInputString Is string to load the generated key formed like <prefixOfTheKeyThing> <maxOfTheKeyThing>
	* @return Returns true if the key was generated successfully. (If theName is an existent keyThing)
	*/
	bool generateKeyOf( std::string theName,  int posibleMax, std::string &theInputString );

	/*
	* Gets the prefix of a specific keyThing identified with the name in the parameter, if the key thing doesn't exists return false.
	* @param theName Is the name of the keyThing to search the prefix. 
	* @param theInputString Is string to load the prefix of the searched keyThing.
	* @return isStringOutputReal Returns if the loaded string in theInputString is real (i.e if the -theName- keyThing exists)
	*/
	bool getPrefixOf(std::string theName, std::string &theInputString);

	/*
	* Gets the max value of a specific keyThing identified with the name in the parameter. If the key thing doesn't exists returns -1.
	* @param theName Is the name of the keyThing to search the maximum. 	
	* @return theMax Returns the maxumum value for key generation at the keyThing. Returns -1 if not finded keyThing.
	*/
	int getCurrentMaxOf(std::string theName);

	/*
	* Gets the total of keyThings managed
	* @return Retuns the total of keyThing managed
	*/
	int getTotalThingsNumber();

	/*
	* Clears the generator deleating the existring keyThings
	*/
	void clear();
	
//------------------------------------------------------------------------------------------------------------
// Constants
//------------------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------------------
// Attributes
//------------------------------------------------------------------------------------------------------------
	
private: 	

	/*
	* Represents the number of things that are currently managing
	*/
	int numberOfKeyThings;

	/*
	* Represents the map or table in which the key-things are saved
	*/
	std::map<std::string,KeyThing> keyThings;
};
#endif

