/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#ifndef __creaImageIOGimmickView_h_INCLUDED__
#define __creaImageIOGimmickView_h_INCLUDED__

#include <creaImageIOGimmick.h>
#include <creaImageIOTreeView.h>
#include <creaImageIOSystem.h>
#include <creaImageIOImagePointerHolder.h>

//#include <map>
#include <vtkImageData.h>
#if defined (USE_WXWIDGETS)
#include <creaImageIOMultiThreadImageReader.h>
#endif

#if defined(USE_QT)
#include "Qt/creaImageIOQMultiThreadImageReader.h"
#endif


// Signal/slot mechanism for progress events
#include <boost/signals2/signal.hpp>
#include <boost/bind.hpp>

#define GIMMICK_NO_IMAGE_SELECTION 0
#define GIMMICK_2D_IMAGE_SELECTION 2
#define GIMMICK_3D_IMAGE_SELECTION 3
#define GIMMICK_4D_IMAGE_SELECTION 4

#define NATIVE 0
#define _2D    2
#define _3D    3

namespace creaImageIO
{
        typedef std::map<std::string, std::string> mapInfoDicom;

	typedef struct 
	{ 
		vtkImageData *img;
		//std::map<std::string, std::string> infos;
		mapInfoDicom infos; 
	} OutStrGimmick;


	/**
	* \ingroup View
	*/
	 
	class ImageExtent;
    //=====================================================================
    
    //=====================================================================
    ///Abstract class that handles views, attributes and previews (GUI) for Gimmick.
	class GimmickView: public MultiThreadImageReaderUser
    {
    public:
      /// Ctor
      GimmickView(boost::shared_ptr<Gimmick>, int number_of_threads = 0 );
      /// Virtual destructor
      virtual ~GimmickView();
      /// Initializes the view : 
      /// Creates the TreeViews for all the TreeHandler of the Controller
      /// 
      virtual void Initialize();

      /// Type of map from View name to TreeView* 
      /// (This map is equivalent for Views of the TreeHandlerMap of Gimmick)
      typedef std::map<std::string, TreeView*> TreeViewMapType;
      
      /// Returns the TreeViewMap (ref)
      TreeViewMapType& GetTreeViewMap() { return mTreeViewMap; }
      /// Returns the TreeViewMap (const ref)
      const TreeViewMapType& GetTreeViewMap() const
      { return mTreeViewMap; }

      /// Finalize 
      virtual void Finalize();

	  ///Returns the maximal priority
      int GetMaximalPriority(){return mReader.GetMaximalPriority();}

      ///Adds the selected Images to the given vector
      virtual void GetSelectedImagesInVector(std::vector<vtkImageData*>& s, int dim)
	{ GimmickError("INTERNAL ERROR : GetSelectedImagesInVector not implemented"); }

      ///Adds the selected Images to the given vector and validates to see if they comply with the given parameter (4D)
      virtual void GetSelectedImages(std::vector<vtkImageData*>& s, int dim) 
	  { GimmickError("INTERNAL ERROR : GetSelectedImages not implemented"); }

      virtual void GetSelectedFiles(std::vector<std::string>& s)
      { GimmickError("INTERNAL ERROR : GetSelectedFiles not implemented"); }

      virtual void GetImages(int dim, std::vector<std::string> files, std::vector<vtkImageData*>& s) 
      { GimmickError("INTERNAL ERROR : GetImages not implemented"); }

      virtual void OnSelectionChange(const std::vector<tree::Node*>& s, bool isSelection, int selection, bool mProcess)
      { GimmickError("INTERNAL ERROR : OnSelectionChange not implemented"); }

	   virtual void ClearSelection()
      { GimmickError("INTERNAL ERROR : ClearSelection not implemented"); }

	  ///Adds a file to ignore
	  virtual void AddIgnoreFile(tree::Node* toRemove)
	  { GimmickError("INTERNAL ERROR : AddIgnoreFile not implemented"); }
      
	  ///Copies selected files
	  virtual void CopyFiles(const std::vector<std::string>& filenames)
	  { GimmickError("INTERNAL ERROR : CopyFiles not implemented"); }

	  ///Edits the fields of a given node
	  virtual void CreateEditFieldsDialog(tree::Node* node, std::vector<std::string> names, std::vector<std::string> keys)
	  { GimmickError("INTERNAL ERROR : EditFields not implemented"); }
      
	  /// Anonymize or de-anonymize data
	  void Anonymize(std::vector<std::string> i_filenames, int type);

	  /// Display all Dicom Tags
	  virtual void DumpTags(const std::string filename)
	  {GimmickError("INTERNAL ERROR : DumpTags not implemented"); }
	  
	    ///Edits the fields of a given node
	  virtual void ExportToStorage(const std::vector<std::string> keys)
	  { GimmickError("INTERNAL ERROR : ExportToStorage not implemented"); }
      
	   ///Copies selected files
	  virtual void SaveAs(const std::vector<std::string>& filenames)
	  { GimmickError("INTERNAL ERROR : SaveAs not implemented"); }

	  /// No selected image
	  bool NoValidateSelected();

      ///Validates the dimension compliance of the images with the maximum and minimum given, and between their sizes
      bool ValidateSelected (tree::Node* sel, int min_dim, int max_dim);

      ///Reads the vector of images, builds it in the dimension required and returns them in the supplied vector.
	  void ReadImagesNotThreaded(std::vector<vtkImageData*>& s,std::vector<std::string> files, int dim);

      ///Reads the vector of images, builds it in the dimension required and returns them in the supplied vector.
	  void ReadImagesNotThreadedInVector(std::vector<vtkImageData*>& s,std::vector<std::string> files, int dim);

///Requests the reading of an image with priority and index in the 
      /// current selection (-1 if not in selection)
      //void RequestReading(tree::Node* n, int prio, int selection_index , ImagePointerHolder *p);
	  void RequestReading(tree::Node* n, int prio, int selection_index , boost::shared_ptr<ImagePointerHolder> p);

      ///Obtains the message of the state
      std::string GetMessage(){return mMess;}

     ///Obtains the message of the state
      void SetMessage(std::string mess){mMess=mess;}

	  ///Resets the data of the extent and begins a new selection
	  void ResetExtent();

      /// Create the tree views 
      void CreateTreeViews();

	  /// Create a tree view with a given name
	  void CreateSingleTreeView(std::string &i_name);

      /// Create the tree view for TreeHandler provided
      virtual void CreateTreeView( TreeHandler* ) 
      { GimmickError("INTERNAL ERROR : CreateTreeView not implemented"); }

      /// Updates the TreeView of given name from level l to bottom
      /// (calls the virtual method TreeView::UpdateLevel(l))
      virtual void UpdateTreeViewLevel(const std::string&, int l);

	  // Multi-thread image reader callback
	  void OnMultiThreadImageReaderEvent(const std::string& filename,
				       MultiThreadImageReaderUser::EventType t,
				       vtkImageData* image);

      vtkImageData* GetDefaultImage() { return mReader.GetImage(""); }

      //=============================================
      typedef boost::signals2::signal<void (bool)>  ValidationSignalType;
      typedef ValidationSignalType::slot_function_type ValidationCallbackType;
      //=============================================

     //==================================================================
      /// Adds the function f to the list of functions to call 
      /// when the addition progresses.
      /// f is of type ProgressCallbackType which is:
      /// void (*ProgressCallbackType)(Progress&)
      /// To pass a member function 'f' of an instance 'c' of a class 'C' 
      /// as callback you have to 'bind' it, i.e. call:
      /// ConnectValidationObserver ( boost::bind( &C::f , c, _1 ) );
      void ConnectValidationObserver(ValidationCallbackType callback);
     //==================================================================

	void modifyValidationSignal(bool ivalid);

	
	void readImages(std::vector<OutStrGimmick>& o_output, std::vector<std::string> im,
			 OutputAttr i_attr, int i_dim, double i_zspc);
			bool isSingle(const std::string i_file);

	void stopReader();
    private:


	typedef void (*readXD)(std::vector<OutStrGimmick>&,  std::vector<std::string>, OutputAttr, double);

	void getAttributes(const std::string i_file, std::map<std::string, std::string> &o_infos, OutputAttr i_attr);

	virtual void readImages2(std::vector<OutStrGimmick>& o_output, std::vector<std::string> im,
			 OutputAttr i_attr, double i_zspc);

	virtual void readImages3(std::vector<OutStrGimmick>& o_output, std::vector<std::string> im,
			 OutputAttr i_attr, double i_zspc);

	virtual void readImages4(std::vector<OutStrGimmick>& o_output, std::vector<std::string> im,
			 OutputAttr i_attr);

	virtual void readImages1(std::vector<OutStrGimmick>& o_output, std::vector<std::string> im,
			 OutputAttr i_attr);


      /// Controller which manages the interaction with the model
      boost::shared_ptr< Gimmick> mGimmick;

      /// The views 
      TreeViewMapType mTreeViewMap;

      /// The message that results from the validation
      std::string mMess;

      /// Multi-thread image reader
      MultiThreadImageReader mReader;

      /// Internal type of image reading event
      /// If the image pointer is non null then the image is available (loaded)
      /// else it has been unloaded
      struct ImageEventType
      {
	ImageEventType( tree::Node* no = 0,   
			int sel_index = -1)
//			ImagePointerHolder* ph= 0)
          : node(no), index(sel_index){}//,  pointerHolder(ph){}
        ImageEventType(vtkImageData* im )
	  : image(im) {}
        tree::Node* node;
        vtkImageData* image;
		int index;
		boost::shared_ptr<ImagePointerHolder> pointerHolder;
      };
      typedef std::map<std::string,ImageEventType> ImageEventTypeMap;
      /// Map of images' names to ImageEventType
      /// Used to associated a filename to a the data of a request
      ImageEventTypeMap mImageEventMap;

      // queue of image event 
      typedef std::deque<ImageEventType> ImageEventQueueType;

      //ImageEventQueueType mImageEventQueue;

	  ///The current image extent
	  boost::shared_ptr<ImageExtent> mImageExtent;

	  ///The validation signal
	  ValidationSignalType mValidationSignal;

	  ///Boolean that determines if the selection is valid
	  bool valid;

	  ///Boolean that determines if the reader has been started
	  bool mReaderStarted;

	};
  // EO class GimmickView
    //=====================================================================

 
} // EO namespace creaImageIO

// EOF
#endif
