/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#ifndef __creaImageIOSQLiteTreeHandler_h_INCLUDED__
#define __creaImageIOSQLiteTreeHandler_h_INCLUDED__

#include <creaImageIOTreeHandler.h>
#include <creaImageIOTree.h>

class CppSQLite3DB;

namespace creaImageIO
{

  
  /**
   * \ingroup Model
   */
  //=======================================================================
  /// Concrete TreeHandler which manages a Tree stored in a sqlite database
  class SQLiteTreeHandler : virtual public TreeHandler
  {
  public:
    //====================================================================
    /// Ctor with database file name 
    SQLiteTreeHandler(const std::string& filename);
    /// Dtor
    virtual ~SQLiteTreeHandler();
    //====================================================================

    //====================================================================
    /// Returns the sqlite db file name 
    const std::string& GetFileName() const { return mFileName; }
    //====================================================================
 
    //====================================================================
    // QUERY METHODS
    /// Is the 'source' readable ?
    virtual bool IsReadable() { return true; }
    /// Is the 'source' writable ?
    virtual bool IsWritable() { return true; }
    //====================================================================


    //====================================================================
    // INITIALIZATION / FINALIZATION
    //====================================================================

    //====================================================================
    /// Opens an existing 'source' 
    // Default mode is read only 
    // If IsWritable and writable==true then opens in read/write mode
    virtual bool Open(bool writable = false);
    /// Closes the 'source'
    virtual bool Close();
    /// Creates a new 'source' 
    // Default mode is read only 
    // If IsWritable and writable==true then opens in read/write mode
    virtual bool Create(bool writable = false);
    /// Destroys the 'source'
    virtual bool Destroy();
	/// Begins a transaction
	virtual void BeginTransaction();
	///Commits results and ends transaction
	virtual void EndTransaction();
    //====================================================================


    //====================================================================
    // READ METHODS
    //====================================================================


    //====================================================================
    /// Returns the number of children of the Node *WITHOUT LOADING THEM*
    // REM : The Tree itself is a Node and asking for its number of 
    //       children returns the number of children of level 1.
    virtual unsigned int GetNumberOfChildren(tree::Node* n);
    //====================================================================

	//====================================================================
    /// Returns the attribute requested. Useful for synchronization.
	virtual void GetAttribute(std::string levelDescriptor,
									   std::string searchParam, 
									   std::string searchVal, 
									   std::string key, 
									   std::string& result);
    //====================================================================


    //====================================================================
    /// Recursively loads the children of node 'parent' until maxlevel 
    // is reached.
    // If parent == NULL or parent == tree then starts with the 'children' of 
    // the tree itself.
    // Returns the total number of children loaded.
    virtual int LoadChildren(tree::Node* parent, int maxlevel);
    //====================================================================

    //====================================================================
    /// Unloads the Node and its descendants
    // WITHOUT altering the source, e.g. the database
    virtual void UnLoad(tree::Node* n);
    ///====================================================================
	
	//====================================================================
    /// Returns the top level node id for the given search param and search value
    virtual void GetTopLevelNodeId(const std::string& searchParam, 
											const std::string& searchValue, 
											std::string& parent_id);
    ///====================================================================

    //====================================================================
    // WRITE METHODS : WORK ONLY IN WRITE MODE
    //====================================================================
    /// Adds a branch in the tree with the attributes provided
    // returns the Level in the tree where the branch was connected 
    // (-1 for error, 0 for top level, etc. ) 
    // Of course the branch is loaded on exit
    virtual int AddBranch( const AttributeMapType& attr );
    // Removes the node and its descendants 
    virtual bool Remove(tree::Node*);
    // Sets an attribute of a Node
    virtual bool SetAttribute(tree::Node*, 
			      const std::string& key,
			      const std::string& value);
	// Sets an attribute
    virtual void SetAttribute(const std::string& levelDescriptor, 
			      const std::string& key,
			      const std::string& value,
				  const std::string& searchParam, 
				  const std::string& searchVal);
	//Deletes the tuple that matches the parameters given
	virtual void DeleteTuple(std::string levelDescriptor, std::string key, std::string value);
	//Deletes the entries that match the parameters given
	virtual void RemoveEntries(const std::string i_table, 
		const std::string i_attribute, 
		const std::string i_operand, 
		const std::string i_val);
 
    //====================================================================
    /// get all attributes from database for a given file
  	void getAllAttributes(std::string i_filename, std::map<std::string, std::string> &i_results);
	
	/// get selected attributes from database for a given file
  	void GetAttributes(std::string name, std::string i_id, std::string i_value, tree::LevelDescriptor::AttributeDescriptorListType i_attr, std::vector<std::string> &i_results);
	
	/// get Id for the up level node
	void GetUpLevelNodeId(int level, const std::string& searchParam, const std::string& searchValue, std::string& parent_id);

  protected:
    //======================================================================
    /// Open the database
    bool DBOpen();
    /// Import the Tree::Description from database (verifies the structure)
    bool DBImportTreeDescription();
    //======================================================================
    //======================================================================
    // Creation
    /// Creates a new database on disk and the tables
    bool DBCreate();
    /// Appends to string s the SQL command to create the attributes of a given level
    void SQLAppendAttributesDefinition(int level, std::string& s);
    //======================================================================

    //======================================================================

    /// Returns the parent to which the branch defined by the attributes 
    // provided must be grafted 
    tree::Node* DBGetParent( const AttributeMapType& attr);
    //======================================================================

    //======================================================================

    /// Loads the children of Node parent
    // Can recurse to numberoflevels levels
    // \return The total number of Node loaded (may be at different levels)
    int DBLoadChildren( tree::Node* parent, int numberoflevels = 1);
    //======================================================================

    //======================================================================

    /// Appends to string s the SQL command to set the attributes values 
    // of node n
    void SQLAppendAttributesValues(tree::Node* n, std::string& s);
    //======================================================================

    //======================================================================

    /// Graft the branch defined by the attributes to the parent
    void DBGraftToParent( tree::Node* parent, const AttributeMapType& attr);
    //======================================================================
    //======================================================================

    /// Sets an attribute of a Node and updates the database
    bool DBSetAttribute(tree::Node*, 
			const std::string& key,
			const std::string& value);
    //======================================================================
    //======================================================================
	/// Sets an attribute and updates the database
	void DBSetAttribute(const std::string& levelDescriptor, 
			      const std::string& key,
			      const std::string& value,
				  const std::string& searchParam, 
				  const std::string& searchVal);
    //======================================================================
    //======================================================================
	
    /// Inserts the Node in the database
    void DBInsert(tree::Node* n);
    //======================================================================

	
	//======================================================================

    /// Deletes the tuple that matches the value specified in the given key and that belongs to the given level
	void DBDelete(std::string levelDescriptor, std::string key, std::string value);
    //======================================================================

	//======================================================================

	/// Recursively Removes the nodes whose parent is given as a parameter
    void DBRecursiveRemoveNode(tree::Node* node);
	/// Recursively Removes the nodes found in the given level with the given parent id
	void DBRecursiveRemoveNode(int level, std::string parentId);
 
	//======================================================================

  
  private:
    /// The DB
    CppSQLite3DB* mDB;
   /// The physical location associated to the DicomDatabase (directory, db file...)
    std::string mFileName;
    /// Is the DB writable ?
    bool mWritable;
    void SetWritable(bool w) { mWritable = w; }
    bool GetWritable() const { return mWritable; }
	bool mIsAdding;
	const std::string convert(const std::string &i_word);

   
  };
  // EO class SQLiteTreeHandler
  //=======================================================================


} // EO namespace creaImageIO

// EOF
#endif  

