/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkAtomicBlackBoxGetSetFunctor.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.6 $
=========================================================================*/


/**
 *  \file 
 *  \brief Class bbtk::AtomicBlackBoxGetFunctor / Class bbtk::AtomicBlackBoxSetFunctor : abstract functors of the Get and Set accessors of the inputs and outputs of a AtomicBlackBox ; Concrete derivatives.
 */

/**
 *  \class bbtk::AtomicBlackBoxGetFunctor
 *  \brief Abstract functor of the Get accessors of the inputs and outputs of a AtomicBlackBox
 *  \class bbtk::AtomicBlackBoxSetFunctor
 *  \brief Abstract functor of the Set accessors of the inputs and outputs of a AtomicBlackBox
 * \class bbtk::AtomicBlackBoxTGetFunctor
 * \brief Template for concrete functors of the Get accessors of the inputs and outputs of a AtomicBlackBox (inherits from bbtk::AtomicBlackBoxGetFunctor).
 * \class bbtk::AtomicBlackBoxTSetFunctor
 * \brief Template for concrete functors of the Set accessors of the inputs and outputs of a AtomicBlackBox (inherits from bbtk::AtomicBlackBoxSetFunctor).
 */
 
#ifndef __bbtkAtomicBlackBoxGetSetFunctor_h__
#define __bbtkAtomicBlackBoxGetSetFunctor_h__

#include "bbtkData.h"
#include "bbtkMessageManager.h"

namespace bbtk
{

   // Forward declaration
  class AtomicBlackBox;


  //===========================================================================
  class BBTK_EXPORT AtomicBlackBoxGetFunctor
  {
  public:
    /// Default constructor
    AtomicBlackBoxGetFunctor() {}
    /// Dtor
    virtual ~AtomicBlackBoxGetFunctor() {}
    /// Abstract method which applies the "Get" function of AtomicBlackBox o
    virtual Data Get(AtomicBlackBox* o) = 0;
    /// 
    virtual TypeInfo GetTypeInfo() const = 0;
    /// 
    virtual std::string GetTypeName() const = 0;
    /// 
    virtual std::string GetHumanTypeName() const = 0;
    /// 
    virtual bool IsPointerType() const = 0;

  };
  //===========================================================================


  //===========================================================================
  class AtomicBlackBoxSetFunctor
  {
  public:
    /// Default constructor
    AtomicBlackBoxSetFunctor() {}
    /// Dtor
    virtual ~AtomicBlackBoxSetFunctor() {}
    /// Abstract method which applies the "Set" function of AtomicBlackBox o
    virtual void Set(AtomicBlackBox* o, const Data&) = 0;
    /// 
    virtual TypeInfo GetTypeInfo() const = 0;
    /// 
    virtual std::string GetTypeName() const = 0;
    /// 
    virtual std::string GetHumanTypeName() const = 0;
    /// 
    virtual bool IsPointerType() const = 0;
    /// Abstract method which applies the "Set" function of AtomicBlackBox o
    /// using brute force cast to the typed pointer required by the "Set" fun.
    /// Only works if the param type of the "Set" function is a pointer 
    /// (see template specialization below).
    /// !!! Use with care !!!
    virtual void BruteForceSetPointer(AtomicBlackBox* o, void* p) = 0;

  };
  //===========================================================================



  //===========================================================================
  template <class UBB, class T, class TRETURN> 
  class AtomicBlackBoxTGetFunctor : public bbtk::AtomicBlackBoxGetFunctor
  {
  public:
    /// Type of pointer on a UBB::Get method  
    typedef TRETURN (UBB::*GetMethodPointerType)(void); //const

    /// Construction with the pointer on the Get method
    AtomicBlackBoxTGetFunctor(GetMethodPointerType g) :
      mGetMethodPointer(g)
      {
	bbtkDebugMessage("data",9,"AtomicBlackBoxTGetFunctor<"<<
			 TypeName<UBB>()<<","<<
			 TypeName<T>()<<","<<
			 TypeName<TRETURN>()<<
			 ">::AtomicBlackBoxTGetFunctor()"<<std::endl);
      }
      
    /// Concrete application of the Get method of object o
    Data Get(AtomicBlackBox* o) 
    {
      bbtkDebugMessage("data",9,"AtomicBlackBoxTGetFunctor<"<<
		       TypeName<UBB>()<<","<<
		       TypeName<T>()<<","<<
		       TypeName<TRETURN>()<<
		       ">::Get()"<<std::endl);
      return (((UBB*)o)->*mGetMethodPointer)();
    }
    /// 
    TypeInfo GetTypeInfo() const { return typeid(T); }
    std::string GetTypeName() const { return TypeName<T>(); }
    std::string GetHumanTypeName() const { return HumanTypeName<T>(); }
    /// 
    virtual bool IsPointerType() const 
    {
      return boost::is_pointer<T>::value;
    }

  private:
    ///  Pointer on the Get method  
    GetMethodPointerType mGetMethodPointer;
  };
  //===========================================================================



  //===========================================================================
  template <class UBB, class T, class TACCESS> 
  class AtomicBlackBoxTSetFunctor : public AtomicBlackBoxSetFunctor
  {
  public:
    /// Type of pointer on a UBB::Set method  
    typedef void (UBB::*SetMethodPointerType)(TACCESS);

    /// Construction with the pointer on the Set method
    AtomicBlackBoxTSetFunctor(SetMethodPointerType s) :
      mSetMethodPointer(s) 
      {
	bbtkDebugMessage("data",9,"AtomicBlackBoxTSetFunctor<"<<
			TypeName<UBB>()<<","<<
			TypeName<T>()<<","<<
			TypeName<TACCESS>()<<
			 ">::AtomicBlackBoxTSetFunctor()"<<std::endl);
      }
      
    /// Concrete application of the Set method of object o
    void Set(AtomicBlackBox* o, const Data& d)
    { 
      bbtkDebugMessage("data",9,"AtomicBlackBoxTSetfunctor<"<<
		      	TypeName<UBB>()<<","<<
		      	TypeName<T>()<<","<<
		      	TypeName<TACCESS>()<<
		       ">::Set()"<<std::endl);
      //      (((UBB*)o)->*mSetMethodPointer)(*(T*)d);
      //      bbtkAssert( bbtkEqualTypes( d.type(), typeid(T) ) );
      T t = d.unsafe_get<T>();
      (((UBB*)o)->*mSetMethodPointer)(t);
      //      bbtkDebugMessage("Kernel",9,"SetOK"<<std::endl);
    }

    /// 
    TypeInfo GetTypeInfo() const { return typeid(T); }
    std::string GetTypeName() const { return TypeName<T>(); }
    std::string GetHumanTypeName() const { return HumanTypeName<T>(); }
    virtual bool IsPointerType() const { return false; }
    virtual void BruteForceSetPointer(AtomicBlackBox* b, void* p) 
    {
      bbtkInternalError("AtomicBlackBoxTSetFunctor<"
			<<TypeName<UBB>()<<","
			<<TypeName<T>()<<","
			<<TypeName<TACCESS>()
			<<">::BruteForceSetPointer("
			<<b<<","<<p<<")"
			<<" called whereas type '"
			<<TypeName<T>()
			<<"' is not a pointer type"); 
    }
  private:
    ///  Pointer on the Set method  
    SetMethodPointerType mSetMethodPointer;
  };
  //===========================================================================



  //===========================================================================
  /// Template specialization of AtomicBlackBoxTSetFunctor for pointer types
  template <class UBB, class T, class TACCESS> 
  class AtomicBlackBoxTSetFunctor<UBB,T*,TACCESS*> 
    : public AtomicBlackBoxSetFunctor
  {
  public:
    /// Type of pointer on a UBB::Set method  
    typedef void (UBB::*SetMethodPointerType)(TACCESS*);

    /// Construction with the pointer on the Set method
    AtomicBlackBoxTSetFunctor(SetMethodPointerType s) :
      mSetMethodPointer(s) 
    {
      bbtkDebugMessage("data",9,"AtomicBlackBoxTSetFunctor<"<<
		       TypeName<UBB>()<<","<<
		       TypeName<T*>()<<","<<
		       TypeName<TACCESS*>()<<
		       ">::AtomicBlackBoxTSetFunctor()"<<std::endl);
    }
    
    /// Concrete application of the Set method of object o
    void Set(AtomicBlackBox* o, const Data& d)
    { 
      bbtkDebugMessage("data",9,"AtomicBlackBoxTSetfunctor<"<<
		       TypeName<UBB>()<<","<<
		       TypeName<T*>()<<","<<
		       TypeName<TACCESS*>()<<
		       ">::Set()"<<std::endl);
      
      (((UBB*)o)->*mSetMethodPointer)(d.unsafe_get<T*>());

    }

    /// 
    TypeInfo GetTypeInfo() const { return typeid(T*); }
    std::string GetTypeName() const { return TypeName<T*>(); }
    std::string GetHumanTypeName() const { return HumanTypeName<T*>(); }
    virtual bool IsPointerType() const { return true; }

 
    virtual void BruteForceSetPointer(AtomicBlackBox* o, void* p) 
    {  
      bbtkDebugMessage("data",9,"AtomicBlackBoxTSetFunctor<"
		       <<TypeName<UBB>()<<","
		       <<TypeName<T*>()<<","
		       <<TypeName<TACCESS*>()
		       <<">::BruteForceSetPointer() (pointer specialization)");

      (((UBB*)o)->*mSetMethodPointer)((T*)p);

    }
  private:
    ///  Pointer on the Set method  
    SetMethodPointerType mSetMethodPointer;
  };
  //===========================================================================

}
// namespace bbtk
#endif
