/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkBlackBoxInputOutputDescriptor.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.7 $
=========================================================================*/


/**
 *  \file 
 *  \brief Class bbtk::BlackBoxInputOutputDescriptor : abstract descriptor of an input or an output of a black box (has a name, description, type and nature)
 */
/**
 * \class bbtk::BlackBoxInputOutputDescriptor
 * \brief Abstract descriptor of an input or an output of a black box (has a name, description, type and nature)
 */

#ifndef __bbtkBlackBoxInputOutputDescriptor_h__
#define __bbtkBlackBoxInputOutputDescriptor_h__

#include "bbtkSystem.h"
#include "bbtkData.h"
#include "bbtkRTTI.h"


namespace bbtk
{
  class BBTK_EXPORT BlackBoxInputOutputDescriptor
  {
  public:
    /// Ctor with name, description and naure
    BlackBoxInputOutputDescriptor(TypeInfo creator_type_info,
				  const std::string& name,
				  const std::string& description,
				  const std::string& nature,
				  bool copy_construct = true
				  )
      : mCreatorTypeInfo(creator_type_info),
	mName(name), 
	mDescription(description),
	mNature(nature),
	mCopyConstruct(copy_construct)
    {}
    /// Dtor
    virtual ~BlackBoxInputOutputDescriptor() {}

    /// Returns the TypeInfo of the BlackBoxDescriptor which created this descriptor
    TypeInfo GetCreatorTypeInfo() const { return mCreatorTypeInfo; }
    /// Returns the name of the in/output
    const std::string& GetName() const { return mName; }
    /// Returns the description of the in/output
    const std::string& GetDescription() const { return mDescription; }
    /// Returns the nature of the in/output
    const std::string& GetNature() const { return mNature; }
    /// Returns the type of the in/output
    virtual TypeInfo GetTypeInfo() const = 0;
    /// Returns the name of the type of the in/output
    virtual std::string GetTypeName() const = 0;
    /// Returns the DataInfo of the input (Type+Nature)
    virtual DataInfo GetDataInfo() const 
    { return DataInfo(this->GetTypeInfo(),mNature); }
    /// Returns the name of the type of the in/output
    virtual std::string GetHumanTypeName() const = 0;
    /// Returns true iff the type is a pointer to class 
    virtual bool IsPointerType() const = 0;
    /// Returns true iff the in/output must be copied 
    /// by the copy constructor of the box
    bool GetCopyConstruct() const { return mCopyConstruct; }

  protected:
    /// Default ctor is reserved to children
    BlackBoxInputOutputDescriptor() : mCreatorTypeInfo(typeid(void)),
				      mName(""), mDescription(""),
				      mNature(""),
				      mCopyConstruct(true)
				{}
  private:

    /// The TypeInfo of the BlackBoxDescriptor which created this descriptor
    TypeInfo mCreatorTypeInfo;
    /// The name of the input/output
    std::string mName;
    /// The description of the input/output
    std::string mDescription;
    /// The nature of the input/output
    std::string mNature;
    /// Does the in/output has to be copied by copy constructor 
    /// default = true. Must be set to false if it is 
    /// initialized by the user in bbUserCopyConstructor 
    /// (typically if it is a pointer)
    bool mCopyConstruct;
  };
  
}
// namespace bbtk
#endif
