/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkComplexBlackBox.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.9 $
=========================================================================*/

                                                                   



/**
 *  \file 
 *  \brief class bbtk::ComplexBlackBox : user defined complex black boxes
 */

/**
 *  \class bbtk::ComplexBlackBox
 *  \brief Abstract user defined black boxes
 */
 
#ifndef __bbtkComplexBlackBox_h__
#define __bbtkComplexBlackBox_h__

#include "bbtkBlackBox.h"
#include "bbtkComplexBlackBoxDescriptor.h"
//#include "bbtkComplexBlackBoxInputDescriptor.h"
//#include "bbtkComplexBlackBoxOutputDescriptor.h"
#include "creaSystem.h"
#include <list>

namespace bbtk
{


 
  //==================================================================
  class BBTK_EXPORT ComplexBlackBox : public bbtk::BlackBox
  {
    BBTK_OBJECT_INTERFACE(ComplexBlackBox);
    friend class ComplexBlackBoxDescriptor;
    typedef BlackBox Superclass;
  public: 
    //==================================================================
    // PUBLIC PART : ACCESSIBLE TO THE END-USER
    /// Returns the pointer on the box' descriptor
    BlackBoxDescriptor::Pointer bbGetDescriptor() const { return mDescriptor.lock(); }
    /// Returns a pointer on a new instance with name <name>
    //BlackBox* bbNew(const std::string& name);
    /// Returns a pointer on a clone of the box with name <name>
    BlackBox::Pointer bbClone(const std::string& name);

    /// Clear
    void Clear();
    /// Set as prototype
    void SetAsPrototype() { mLockedDescriptor.reset(); }
    /// Returns true if it is a prototype 
    bool IsAPrototype() const { return (!mLockedDescriptor); }

    std::string bbGetNameWithParent() const;

    void bbExecute(bool force = false);
  
    Data bbGetOutput( const std::string &label );
    Data bbGetInput ( const std::string &label );
    void bbSetOutput( const std::string &name, Data data);
    void bbSetInput ( const std::string &name, Data data,
		      bool setModified = true);
    void bbBruteForceSetInputPointer( const std::string &name, 
				      void* data, 
				      bool setModified = true);

     /// Connects the input <name> to the connection c
    virtual void bbConnectInput( const std::string& name, Connection* c);
    /// Connects the output <name> to the connection c
    virtual void bbConnectOutput( const std::string& name, Connection* c);

    BlackBox::Pointer bbGetBlackBox( const std::string& name );

    /// The type of map of black boxes
    typedef std::map<std::string, BlackBox::Pointer> BlackBoxMapType;

    const BlackBoxMapType& bbGetBlackBoxMap() { return mBlackBoxMap; }

    void bbPrintBlackBoxes();

    /// Generates a png image representing the pipeline graph of the 
    /// complex box and writes html code to insert it 
    /// into the output file stream. 
    /// detail : 
    /// level : depth of nested complex boxes graph development 
    /// (0:only this box level 
    /// output_dir is the directory in which to write the files 
    void bbInsertHTMLGraph(  std::ofstream& s, 
			     int detail, 
			     int level,
			     bool instanceOrtype,
			     const std::string& output_dir,
			     bool relative_link );

    /// Writes Graphviz-dot description in file.
    /// Generates own description and recursively calls itself 
    /// on internal boxes with level-1 
    virtual void bbWriteDotFileBlackBox(FILE *ff,
					BlackBox::Pointer parentblackbox, 
					int detail, int level,
					bool instanceOrtype,
					bool relative_link );
    
    virtual void bbWriteDotInputOutputName(FILE *ff,
					   bool inputoutput, 
					   int detail, int level);
    
    virtual BlackBox::Pointer bbFindBlackBox(const std::string &blackboxname);
    

    void Check(bool recursive=true);

       

  protected:
    //==================================================================
    // PROTECTED PART : ACCESSIBLE TO ComplexBlackBoxDescriptor
    static ComplexBlackBox::Pointer 
    New(const std::string &name,
	ComplexBlackBoxDescriptor::Pointer desc);
				 
  private:			 
    ///  Constructor that takes the ComplexBlackBox name
    ComplexBlackBox(const std::string &name,
		    ComplexBlackBoxDescriptor::Pointer desc);
    /// Constructor from an existing box (copy) with a new name 
    ComplexBlackBox(ComplexBlackBox& from, const std::string &name);
    
  protected:
  


    void bbAddBlackBox( BlackBox::Pointer );
    void bbAddToExecutionList( const std::string& name );
    void bbUnsafeAddBlackBox( BlackBox::Pointer );
    void bbRemoveBlackBox( const std::string& name, 
			   bool remove_connections = true);
    void bbAddConnection( Connection::Pointer );
    
    //    void RemoveConnection( );

    BlackBox::Pointer bbUnsafeGetBlackBox( const std::string& name );

    /// 
    void bbAllocateConnectors();
    void bbDesallocateConnectors();

  private:
    //==================================================================
    // PRIVATE PART : MEMBERS AND USEFULL INTERNAL METHODS

    /// Default constructor : derived classes must use the constructor with the ComplexBlackBox's name
    //ComplexBlackBox() : BlackBox("") {}
    
    // true if the box is a prototype of a ComplexBlackBoxDescriptor
    //    bool mIsAPrototype;

    /// The descriptor pointer : is 0 if the box is a prototype 
    ComplexBlackBoxDescriptor::Pointer mLockedDescriptor;
    /// The descriptor pointer
    ComplexBlackBoxDescriptor::WeakPointer mDescriptor;
    
    /// The map of black boxes
    BlackBoxMapType mBlackBoxMap;
    
    
    /// The type of list of connections
    typedef std::list<Connection::Pointer> ConnectionListType;
    /// The list of connections
    ConnectionListType mConnectionList;

/// The execution list
    std::vector<std::string> mExecutionList;
    
  };
  // Class ComplexBlackBox
  //===========================================================================



}
// namespace bbtk


//#include "bbtkComplexBlackBoxMacros.h"

#endif

