/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkConnection.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.14 $
=========================================================================*/

                                                                       

/**
 *\file
 *\brief Class bbtk::Connection
 */
/**
 *\class bbtk::Connection
 *\brief DDD
 *
 */

#ifndef __bbtkConnection_h__
#define __bbtkConnection_h__

#include "bbtkSystem.h"
#include "bbtkObject.h"

#include <string>

namespace bbtk
{

  class Factory;
  BBTK_FORWARD_DECLARE_POINTER(Factory);
  class BlackBox;
  BBTK_FORWARD_DECLARE_POINTER(BlackBox);
  class BlackBoxInputConnector;
  class BlackBoxOutputConnector;

  /// The type of input / output status 
  typedef unsigned char IOStatus;
  /// Up-to-date status value
  const IOStatus UPTODATE  = 0;
  /// Modified status value
  const IOStatus MODIFIED  = 1;
  /// Out-of-date status value
  const IOStatus OUTOFDATE = 2;

  BBTK_EXPORT const std::string& GetIOStatusString( IOStatus );


  class BBTK_EXPORT Connection : public Object
  {
     BBTK_OBJECT_INTERFACE(Connection);
    typedef Object Superclass;
  public:

    static Pointer New(BlackBoxPointer from, const std::string& output,
		       BlackBoxPointer to, const std::string& input,
		       const FactoryPointer f);
    static Pointer New(BlackBoxPointer from, const std::string& output,
		       BlackBoxPointer to, const std::string& input);
    /// Dtor
    //    ~Connection();

    // void Delete();
    
    /// Pipeline processing method
    /// 1) call bbRecursiveExecute(this) on the from box
    /// 2) copies the from box output to the to box input 
    ///    adapting it if needed
    /// 3) sets the new IOStatus of the to box input to the 
    ///    status of the from box output
    void RecursiveExecute();
    
    /// Change callback
    void OnOutputChange(BlackBoxPointer, const std::string&, 
			IOStatus);

    std::string GetFullName() const; 

    /// Returns the original initial black box of the connection
    BlackBoxPointer GetOriginalBlackBoxFrom() const { return mOriginalFrom.lock(); }
    /// Returns the origianl final black box of the connection
    BlackBoxPointer GetOriginalBlackBoxTo() const { return mOriginalTo.lock(); }
    /// Returns the original output of the initial black box of the connection
    const std::string& GetOriginalBlackBoxFromOutput() const { return mOriginalOutput; }
    /// Returns the original input of the final black box of the connection
    const std::string& GetOriginalBlackBoxToInput() const { return mOriginalInput; }
    
    /// Returns the initial black box of the connection
    BlackBoxPointer GetBlackBoxFrom() const { return mFrom; }
    /// Returns the final black box of the connection
    BlackBoxPointer GetBlackBoxTo() const { return mTo; }
    /// Returns the output of the initial black box of the connection
    const std::string& GetBlackBoxFromOutput() const { return mOutput; }
    /// Returns the input of the final black box of the connection
    const std::string& GetBlackBoxToInput() const { return mInput; }

    /// Sets the initial black box of the connection
    void SetBlackBoxFrom(BlackBoxPointer b) { mFrom = b; }
    /// Sets the final black box of the connection
    void SetBlackBoxTo(BlackBoxPointer b) { mTo = b; }
    /// Sets the output of the initial black box of the connection
    void SetBlackBoxFromOutput(const std::string& o) { mOutput = o; }
    /// Sets the input of the final black box of the connection
    void SetBlackBoxToInput(const std::string& o) { mInput = o; }

    /// Checks that the connection is ok (throws error if not)
    void Check() const;
    
  protected:
    /// Black box origin of the connection
    BlackBoxPointer mFrom;
    BlackBoxWeakPointer mOriginalFrom;
    /// Output of mFrom which is connected
    std::string mOutput;
    std::string mOriginalOutput;
    /// Output connector of mFrom which is connected
    //  BlackBoxOutputConnector* mOutputConnector;
    /// Black box destination of the connection
    BlackBoxPointer mTo;
    BlackBoxWeakPointer mOriginalTo;
    /// Input of mTo which is connected
    std::string mInput;
    std::string mOriginalInput;
    /// Input connector of mTo which is connected
    //  BlackBoxInputConnector* mInputConnector;
    /// Adaptor black box if needed
    BlackBoxPointer mAdaptor;

    /// The factory used to create adaptors
    const FactoryWeakPointer mFactory;

    /// Is the connection input type is any<thing> ?
    bool mFromAny;

    /// Is the connection output type is any<thing> ?
    bool mToAny;


    /// Have to do dynamic_cast ?
    bool mDoDynamicCast;

    /// Ctor with the black box from and to and their input and output
    /// and a dummy int to differentiate from the public constructor.
    /// Sets the members but does not test compatibility (used by bbtk::AdaptiveConnection)
    //  Connection(BlackBox* from, const std::string& output,
    //	       BlackBox* to, const std::string& input, int   );
  public:   
    void TransferData(); 

  private:
    /// Ctor 
    Connection(BlackBoxPointer from, const std::string& output,
	       BlackBoxPointer to, const std::string& input,
	       const FactoryPointer f);
    /// Ctor 
    Connection(BlackBoxPointer from, const std::string& output,
	       BlackBoxPointer to, const std::string& input);

  };

}// namespace bbtk

#endif

