/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkInterpreter.h,v $
  Language:  C++
  Date:      $Date: 2010/09/14 07:18:47 $
  Version:   $Revision: 1.41 $
=========================================================================*/



/**
 *  \file 
 *  \brief class Interpreter : The bbtk language interpreter
 */
/**
 *  \class bbtk::Interpreter 
 *  \brief The bbtk language interpreter
 */

#ifndef __bbtkInterpreterVirtual_h__
#define __bbtkInterpreterVirtual_h__

//#include "bbtkSystem.h"

#include "bbtkMessageManager.h"
#include "bbtkException.h"
#include "bbtkObject.h"


#include <fstream>
#include <deque>

// Signal/slot mechanism for 'break' commands
//#include <boost/signals2/signal.hpp>
#include <boost/bind.hpp>

namespace bbtk
{

  
  //=======================================================================
  class BBTK_EXPORT InterpreterUser
  {
  public: 
    InterpreterUser() {}
    virtual ~InterpreterUser() {}

    virtual bool InterpreterUserHasOwnHtmlPageViewer() { return false; }
    virtual void InterpreterUserViewHtmlPage(const std::string&) {}


  };
  //=======================================================================



  //=======================================================================
  class BBTK_EXPORT InterpreterException : public Exception
  {
  public:
    InterpreterException( const std::string& message,
		      bool in_script_file,
		      const std::string& script_file,
		      int script_line 
		      );
    InterpreterException( const Exception& excep,
		      bool in_script_file,
		      const std::string& script_file,
		      int script_line 
		      );
    ~InterpreterException() throw() {}

    bool IsInScriptFile() const { return mInScriptFile; }
    const std::string& GetScriptFile() const { return mScriptFile; }
    int GetScriptLine() const { return mScriptLine; }
  private:
    bool mInScriptFile;
    std::string mScriptFile;
    int mScriptLine;
  };
  //=======================================================================

  /*
  //=======================================================================
  // The "Quit" exception
  class BBTK_EXPORT QuitException : public InterpreterError
  {
  public:
    QuitException( bool in_script_file,
		   const std::string& script_file,
		   int script_line 
		   )
      : InterpreterError("QUIT",in_script_file,script_file,script_line) 
    {}
    ~QuitException() throw() {}
  };
  //=======================================================================
  const std::string BREAK("BREAK");
  //=======================================================================
  // The "Break" exception
  class BBTK_EXPORT BreakException : public InterpreterError
  {
  public:
    BreakException( bool in_script_file,
		    std::string script_file,
		    int script_line 
		    )
      : InterpreterError(BREAK,in_script_file,script_file,script_line) 
    { std::cout << "BUILDING BREAK"<<std::endl; }
    ~BreakException() throw() {}
  };
  //=======================================================================
  */

  //=======================================================================
  class BBTK_EXPORT InterpreterVirtual : public Object
  {
    BBTK_OBJECT_INTERFACE_NO_CONDES(InterpreterVirtual);
    typedef Object Superclass;
  public:
	  static Pointer New();

    typedef enum 
      {
	Interpreter_OK,
	Interpreter_ERROR,
	Interpreter_QUIT,
	Interpreter_BREAK,
      }
      ExitStatus;

    /// Runs the interpretation of a file
    ExitStatus InterpretFile( const std::string& filename, bool source = false);

    /// Runs the interpretation of a buffer and deletes it !
    ExitStatus InterpretBuffer( std::stringstream* buffer );

    /// Runs the interpretation of a command
    ExitStatus InterpretLine( const std::string& line );

    /// Runs the interpretation of the currently open streams
    ExitStatus InterpretCurrentStreams();

    /// Launches a command line interpreter (with a prompt)
    void CommandLineInterpreter();



    /// Sets the bool that indicates wether we are in command line context
    void SetCommandLine(bool v = true) { mCommandLine = v; }

    void SetThrow(bool b) { mThrow = b; }

#ifdef USE_WXWIDGETS
    /// Sets the user of the interpreter (if any)
    void SetUser(InterpreterUser* c) { mUser = c; }
    /// Gets the InterpreterUser of this 
    InterpreterUser* GetUser() { return mUser; }
    /// Gets the InterpreterUser of this (const)
    const InterpreterUser* GetUser() const { return mUser; }

#endif


	  
	  
    /*
    // For 'break' commands observation
    typedef boost::signals::trackable BreakObserverType;
    typedef boost::signal<void ()>  BreakSignalType;
    typedef BreakSignalType::slot_function_type BreakCallbackType;
  
    // Adds a callback when 'break' command issued
    void AddBreakObserver( BreakCallbackType );
    */

  protected:

    /// The enumeration of command codes == Command name
    typedef enum
    {
      cBreak, // LG 12/12/08 : Stops the current script execution (if not exec frozen) - used in tutorial + debugging 
      cClear, // LG 12/12/08 : Clears the current complex black box (e.g. workspace) - used in tours
      cNew,
      cDelete,
      cConnect,
      cExec,
      cPackage,
      cEndPackage,
      cDefine,
      cEndDefine,
      cInput,
      cOutput,
      cSet,
      cConfig,  // JPR
      cReset,   // EED
      cAuthor, 
      cCategory, // JPR
      cDescription,
      cHelp,
      cMessage,
      cInclude,
      cQuit,
      cLoad,
      cUnload,
      cGraph,
      cPrint,
      cIndex,
      cKind, // LG
      cNewGUI, // LG
      cWorkspace, // LG
      cDebug // LG
    } CommandCodeType;

    /// The structure storing the informations on a command 
    typedef struct 
    {
      std::string keyword;
      int argmin, argmax;
      CommandCodeType code;
      std::string syntax;
      std::string help;
    } CommandInfoType;

    /// The type of dictionnary of commands 
    typedef std::map<std::string,CommandInfoType> CommandDictType;


    /// Interprets a line 
    void DoInterpretLine( const std::string& line ); //, bool &insideComment );

    /// Reads a line from prompt
    void GetLineFromPrompt( std::string& line );

    /// Splits a line into words
    void SplitLine ( const std::string& line,
                     std::vector<std::string>& words );

    /// Executes the right action depending on the command name
    void InterpretCommand( const std::vector<std::string>& words,
                           CommandInfoType& info );

    /// Switch to the interpretation of a file
    void SwitchToFile( const std::string& filename, bool source = false );

   /// Switch to the interpretation of a stringstream
    void SwitchToStream( std::stringstream* stream );

    /// Closes the currently open file
    void CloseCurrentFile();

    /// Closes all open files
    void CloseAllFiles();

    /// Resets all
    virtual void commandReset();

    /// Displays help (entry point of any help)
    virtual void commandHelp(const std::string &words);
    virtual void commandHelp(const std::vector<std::string>& words);

    ///
    virtual void commandGraph(const std::vector<std::string>& words);

    ///
    virtual void commandConfig() const; 
    /// 
    //    void Print(const std::string&);

    void commandIndex(const std::string& filename,const std::string& type = "Initials");
    ///
    void FindCommandsWithPrefix( char* buf,
				 int n,
				 std::vector<std::string>& commands );

    /// Creates and connects the piece of pipeline which defines a GUI 
    /// for the box box. 
    /// Define it as a complex box type with name instanceName+"Type"
    /// The instance is called instanceName
    /// and connected to the existing pipeline
      virtual void commandNewGUI(const std::string& box,const std::string& instanceName);
      virtual void commandDebug(const std::string& arg);
	  virtual void commandNew(const std::string &boxType, const std::string &boxName);
	  virtual void commandDelete(const std::string &boxName);
	  virtual void commandConnection(const std::string &nodeFrom,const std::string &outputLabel,const std::string &nodeTo,const std::string &inputLabel);
	  virtual void commandPackage(const std::string &packageName);
	  virtual void commandEndPackage();
	  virtual void commandDefine(const std::string &name,const std::string &pack,const std::string &scriptfilename);
	  virtual void commandEndDefine();
	  virtual void commandKind(const std::string &kind);
	  virtual void commandPrint(const std::string &value);
	  virtual void commandExec(const std::string &word);
	  virtual void commandInput(const std::string &name,const std::string &box,const std::string &input,const std::string  &help);
	  virtual void commandOutput(const std::string &name,const std::string &box,const std::string &output,const std::string  &help);
	  virtual void commandSet(const std::string &box,const std::string &input,const std::string &value);
	  virtual void commandAuthor(const std::string &author);
	  virtual void commandCategory(const std::string &categorytype);
	  virtual void commandDescription(const std::string &description);
	  virtual void commandClear();
	  virtual void commandInclude(const std::string &word, bool ok);
	  virtual void commandLoad(const std::string &packageName);
	  virtual void commandUnload(const std::string &packageName);
	  virtual void commandBreak();
	  virtual void commandQuit();
	  virtual void commandMessage();
	  virtual void commandMessage(const std::string &kind,const std::string &levelstr);
	  virtual void SetCurrentFileName(const std::string &fullPathScriptName);
	  virtual void SetTypeOfScript_Application();
	  
	  /// Constructor
	  InterpreterVirtual();
	  

	  void Init();

	  
	  /// Vector of names of open files with full path (as open)
	  std::vector<std::string> mFileName;

	  
	  /// Stores the current line number in each open file
	  std::vector<int> mLine;
	  
	  bool mThrow;

	  /// Are we in a command line context ?
	  bool mCommandLine;

	  
	  /// Vector of names of files which have been open 
	  /// before (and may closed then which are no more in mFileName)
	  /// with full path (as open)
	  std::vector<std::string> mFileNameHistory;

	  /// The user of  the interpreter (0 if none)
	  bbtk::InterpreterUser* mUser;
  
	  /// The dictionnary of commands
	  CommandDictType mCommandDict;
	  
	  
  private:



    /// Opens the file fullPathScriptName 
    /// includeScriptName is the name as given to the include command 
    void LoadScript( std::string fullPathScriptName,std::string includeScriptName);

  
    //==================================================================
    // ATTRIBUTES



    /// Vector of open files / buffers (can be stringstream)
    std::vector<std::istream*> mFile;


    /// Vector of names of open files as given to the include command
    std::vector<std::string> mIncludeFileName;


    /// Are we inside a commented-out zone ?
    bool mInsideComment;
    
    /// The current interpreter status
    ExitStatus mStatus;

    /// The history of commands
    std::deque< char* > mHistory;

    int bufferNb;
    

    /// The break signal
    //    BreakSignalType mBreakSignal;

  protected:
	  ~InterpreterVirtual();

	  /// 
	  virtual void CatchInterpreterException( const InterpreterException& e );
	  virtual void CatchBbtkException( const bbtk::Exception& e );
	  virtual void CatchStdException( const std::exception& e );
	  virtual void CatchUnknownException();
	  
	  
  };
  // EO class Interpreter



}
#endif
