/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkMessageManager.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.10 $
=========================================================================*/





/*! \file
 
  \brief Class bbtkMessageManager and Macros for outputing messages in bbtk

  There are 4 kinds of messages :
  - Messages (normal messages)
  - Debug messages (not compiled in release)
  - Warnings 
  - Errors
  There are also "types" of messages which are strings which identify the nature of the message 
  (for example : "Kernel" messages are generated by the core classes of the library, there can be a type of 
  message for each type of Node, and so on...)
  A type of message must be declared by registering it into the MessageManager. This is done by a line like :
  bbtk::MessageManager::RegisterMessageType("Kernel","Messages generated by the core classes of the library",5);
  where : 
  -The first string is the type of the message (the category which will be used to generate a message of this type)
  -The second string is help string
  -The integer is the initial level for the messages of this type (see below).
  
  To generate a message of a known type then use one of the macros :
  bbtkMessage, bbtkDebugMessage, bbtkWarning, bbtkError or their variants.

  example :

  bbtkMessage("Kernel",4,"problem with "<<GetName()<<bbtkendl);

  will push the 3rd argument in std::cout if the message level of "Kernel" messages is greater or equal to 4.
  which means that it generates a message of level 4 (0 : very important/always displayed ... 9 : deep debug message).

  At run time, one is able to change the level of the messages displayed by using a command like :
  
  bbtk::MessageManager::SetMessageLevel("Kernel",5); 
  
  which tells the manager to display all Kernel messages of level up to 5.

  Variants :

  bbtk*Cont : continues a previous bbtkMessage on the same line (without rewriting the type and level)
  bbtk*Inc / Dec : displays the message and then increments/decrement the messages tabulation 

*/
  //===========================================================
  /**
     \class bbtk::MessageManager
     \brief Manages the messages displayed by bbtk
  */


#ifndef __bbtkMessageManager_h__
#define __bbtkMessageManager_h__

// The do { } while(0) statement in macros is made to "swallow the semicolon" 
// see http://gcc.gnu.org/onlinedocs/cpp/Swallowing-the-Semicolon.html#Swallowing-the-Semicolon

#include "bbtkSystem.h"
#include "bbtkRTTI.h"

#include <string>
#include <map>
#include <iostream>
#include <sstream>

// Comment out these symbols to prevent compilation 
//#define BBTK_COMPILE_MESSAGES
//#define BBTK_COMPILE_DEBUG_MESSAGES
//#define BBTK_COMPILE_WARNING_MESSAGES
//#define BBTK_COMPILE_ERROR_MESSAGES


#define bbtkOnMessageLevel(key,value)			\
  int __bbtkOnMessageLevelVariable =			\
    bbtk::MessageManager::GetMessageLevel(key);		\
  if ( __bbtkOnMessageLevelVariable<0)			\
    {							\
      bbtkWarning("message type '"<<key<<"' unknown");	\
    }							\
  else if (value<= __bbtkOnMessageLevelVariable) 

#define BBTK_PREPEND_MESSAGE_WITH_CODE
#ifdef BBTK_PREPEND_MESSAGE_WITH_CODE
#define bbtkMessageCode(key,value)			\
  bbtk::MessageManager::FormatKey(key,value) 
#else 
#define bbtkMessageCode(key,value) ""
#endif 

#ifdef BBTK_PREPEND_MESSAGE_WITH_TAB
#define bbtkMessageTab				\
  bbtk::MessageManager::GetTab()
#else 
#define bbtkMessageTab ""
#endif

//#define BBTK_PREPEND_MESSAGE_WITH_SPACE
#ifdef BBTK_PREPEND_MESSAGE_WITH_SPACE
#define bbtkMessageSpace(value)			\
  bbtk::MessageManager::GetSpace(value)
#else 
#define bbtkMessageSpace(value) ""
#endif
 

//===========================================================
#ifdef BBTK_COMPILE_MESSAGES

// Macro for messages
#define bbtkMessage(key,value,MESSAGE)			\
  do {							\
    bbtkOnMessageLevel(key,value)			\
      {							\
	std::cout << bbtkMessageCode(key,value)		\
		  << bbtkMessageTab			\
		  << bbtkMessageSpace(value)		\
		  << MESSAGE;				\
      }							\
  }							\
  while (0)

// Macro for continuing a message (when one wants to split the macro
// call into multiple lines)
#define bbtkMessageCont(key,value,MESSAGE)	\
  do						\
    {						\
      bbtkOnMessageLevel(key,value)		\
	{					\
	  std::cout << MESSAGE;			\
	}					\
    }						\
  while (0)

#define bbtkMessageInc(key,value,MESSAGE)		\
  do							\
    {							\
      bbtkOnMessageLevel(key,value)			\
	{						\
	  std::cout << bbtkMessageCode(key,value)		\
		    << bbtkMessageTab			\
		    << bbtkMessageSpace(value)		\
		    << MESSAGE;				\
	  bbtk::MessageManager::IncTab();		\
	}						\
    }							\
  while (0)

#define bbtkMessageDec(key,value,MESSAGE)               \
  do							\
    {							\
      bbtkOnMessageLevel(key,value)			\
	{						\
	  bbtk::MessageManager::DecTab();		\
	  std::cout << bbtkMessageCode(key,value)		\
		    << bbtkMessageTab			\
		    << bbtkMessageSpace(value)		\
		    << MESSAGE;				\
	}						\
    }							\
  while (0)

#define bbtkDecTab(key,value)			\
  do						\
    {						\
      bbtkOnMessageLevel(key,value)		\
	{					\
	  bbtk::MessageManager::DecTab();	\
	}					\
    }						\
  while (0)

#define bbtkIncTab(key,value)			\
  do						\
    {						\
      bbtkOnMessageLevel(key,value)		\
	{					\
	  bbtk::MessageManager::IncTab();	\
	}					\
    }						\
  while (0)

#define bbtkResetTab()				\
  do						\
    {						\
      bbtk::MessageManager::ResetTab();		\
    }						\
  while (0)

#else
#define bbtkMessage(key,value,MESSAGE)
#define bbtkMessageInc(key,value,MESSAGE)
#define bbtkMessageDec(key,value,MESSAGE)
#define bbtkMessageCont(key,value,MESSAGE)
#define bbtkDecTab(key,value)
#define bbtkIncTab(key,value)
#define bbtkResetTab()
#endif
//===========================================================



//===========================================================
#ifdef BBTK_COMPILE_DEBUG_MESSAGES

// Macro for debug messages
#define bbtkDebugMessage(key,value,MESSAGE)             \
  do							\
    {							\
      bbtkOnMessageLevel(key,value)			\
	{						\
	  std::cout << bbtkMessageCode(key,value)		\
		    << bbtkMessageTab			\
		    << bbtkMessageSpace(value)		\
		    << MESSAGE;				\
	}						\
    }							\
  while (0)

// Macro for continuing a debug message (when one wants to split the
// macro call into multiple lines)
#define bbtkDebugMessageCont(key,value,MESSAGE)	\
  do						\
    {						\
      bbtkOnMessageLevel(key,value)		\
	{					\
	  std::cout << MESSAGE;			\
	}					\
    }						\
  while (0)

#define bbtkDebugMessageInc(key,value,MESSAGE)		\
  do							\
    {							\
      bbtkOnMessageLevel(key,value)			\
	{						\
	  std::cout << bbtkMessageCode(key,value)		\
		    << bbtkMessageTab			\
		    << bbtkMessageSpace(value)		\
		    << MESSAGE;				\
	  bbtk::MessageManager::IncTab();		\
	}						\
    }							\
  while (0)

#define bbtkDebugMessageDec(key,value,MESSAGE)		\
  do							\
    {							\
      bbtkOnMessageLevel(key,value)			\
	{						\
	  bbtk::MessageManager::DecTab();		\
	  std::cout << bbtkMessageCode(key,value)	\
		    << bbtkMessageTab			\
		    << bbtkMessageSpace(value)		\
		    << MESSAGE;				\
	}						\
    }							\
  while (0)

#define bbtkDebugDecTab(key,value)		\
  do						\
    {						\
      bbtkOnMessageLevel(key,value)		\
	{					\
	  bbtk::MessageManager::DecTab();	\
	}					\
    }						\
  while (0)

#define bbtkDebugIncTab(key,value)		\
    do						\
      {						\
	bbtkOnMessageLevel(key,value)		\
	  {					\
	    bbtk::MessageManager::IncTab();	\
	  }					\
      }						\
    while (0)
    
#define bbtkDebugResetTab()			\
    do						\
      {						\
	bbtk::MessageManager::ResetTab();	\
      }						\
    while (0)

#else
#define bbtkDebugMessage(key,value,MESSAGE) 
#define bbtkDebugMessageCont(key,value,MESSAGE) 
#define bbtkDebugMessageInc(key,value,MESSAGE)
#define bbtkDebugMessageDec(key,value,MESSAGE) 
#define bbtkDebugDecTab(key,value)
#define bbtkDebugIncTab(key,value)
#endif
//===========================================================

//===========================================================
#ifdef BBTK_COMPILE_WARNING_MESSAGES
#define bbtkWarning(MESSAGE)						\
  do									\
    {									\
      int lev = bbtk::MessageManager::GetMessageLevel("Warning");	\
      if (lev >0)							\
	{								\
	  std::cerr << "!! WARNING !! " << MESSAGE << std::endl;	\
	  if (lev >1)							\
	    {								\
	      std::cerr << "!! WARNING !! In file '"<<__FILE__		\
			<<"' ; Line "<<__LINE__<<std::endl;		\
	    }								\
	}								\
    }									\
  while (0) 

#else
#define bbtkWarning(MESSAGE) 
#endif
//===========================================================


//===========================================================
#ifdef BBTK_COMPILE_ERROR_MESSAGES
//#include "bbtkWx.h"
#define bbtkError(MESSAGE)				\
  do							\
    {							\
      std::ostringstream s;				\
      s << MESSAGE;					\
      std::ostringstream f;				\
      f << __FILE__ << " (l."<<__LINE__<<")";		\
      bbtk::Exception e( BBTK_GET_CURRENT_OBJECT_NAME,	\
			f.str(),			\
			s.str());			\
      throw e;						\
    }							\
  while (0) 

#define bbtkGlobalError(MESSAGE)				\
  do							\
    {							\
      std::ostringstream s;				\
      s << MESSAGE;					\
      std::ostringstream f;				\
      f << __FILE__ << " (l."<<__LINE__<<")";		\
      bbtk::Exception e( "global scope",		\
			f.str(),			\
			s.str());			\
      throw e;						\
    }							\
  while (0); 

#define BBTK_INTERNAL_ERROR_MESSAGE \
  "\n\n***********************************************\n**** THIS IS AN INTERNAL ERROR TO BBTK     ****\n**** Please send a full bug report to :    ****\n****  bbtk-developers@creatis.insa-lyon.fr ****\n***********************************************\n\n"

#define bbtkInternalError(MESSAGE)			\
  do							\
    {							\
      std::ostringstream s;				\
      s << MESSAGE << BBTK_INTERNAL_ERROR_MESSAGE;	\
      std::ostringstream f;				\
      f << __FILE__ << " (l."<<__LINE__<<")";		\
      bbtk::Exception e( BBTK_GET_CURRENT_OBJECT_NAME,	\
			 f.str(),			\
			 s.str());			\
      throw e;						\
    }							\
  while (0) 

#else
#define bbtkError(MESSAGE)
#define bbtkGlobalError(MESSAGE)
#define bbtkInternalError(MESSAGE)
#endif
//===========================================================

//===========================================================
#define bbtkendl std::endl
//===========================================================


namespace bbtk 
{

  class BBTK_EXPORT MessageManager
  {
  public:
    ///
    MessageManager();
    ///
    ~MessageManager();
    ///
    static MessageManager* GetInstance();
    ///
    static bool RegisterMessageType(std::string key, 
                                    std::string help,
                                    unsigned char default_level = 9);
    ///
    static void SetMessageLevel(std::string key, unsigned char level);
    ///
    static int GetMessageLevel(std::string key);
    ///  
    static std::string& GetTab() { static std::string s; return s; }
    ///
    static std::string GetSpace(int n) { 
      std::string s; s.insert(0,"                ",n); return s; }
    ///
    static void IncTab() { GetTab() += std::string(" "); }
    ///
    static void DecTab() { GetTab() = GetTab().substr(0,GetTab().length()-1); }
    ///
    static void ResetTab() { GetTab() = std::string(""); }
    ///
    static void PrintInfo();

    static std::string FormatKey(const std::string& key, int value);

  private:
    std::map<std::string,int> mMessageLevel;
    std::map<std::string,std::string> mMessageHelp;  
    unsigned int mMaxMessageLength;
  };
  //===========================================================
  
}

#include "bbtkException.h"

#endif
