/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkObject.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.9 $
=========================================================================*/



/**
 *\file
 *\brief  Object : the top class of bbtk class hierarchy 
 * 
 */

#ifndef __bbtkObject_h__
#define __bbtkObject_h__

#include <bbtkSystem.h>
#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>
#include <set>

namespace bbtk
{

  // The top class of bbtk class hierarchy 
  class BBTK_EXPORT Object 
  {
  public:
    typedef boost::shared_ptr<Object> Pointer;
    typedef boost::weak_ptr<Object> WeakPointer;

    Object();
    virtual ~Object(); 
    
    virtual std::string GetObjectName() const;
    virtual std::string GetObjectInfo() const;
    virtual size_t GetObjectSize() const { return sizeof(Object); }
    virtual size_t GetObjectInternalSize() const { return sizeof(Object); }
    virtual size_t GetObjectRecursiveSize() const { return sizeof(Object); }
    long GetUseCount() { return mThisPointer.use_count(); }
    
    static void InsertInObjectList(Pointer);
    static void RemoveFromObjectList(WeakPointer);

    static void InsertInPackageList(Pointer);
    static void ReleasePackages();

    static void PrintObjectListInfo(const std::string& name);
    //    static void PrintObjectInfo(const std::string& name);
    static void PrintObjectInfo(const Pointer& o); 

    static long GetObjectsCount() { return mgObjectList.size(); }

    /// Default objects deleter : removes object from list on deletion
    struct BBTK_EXPORT Deleter 
    { 
      Deleter() : mPointer() {}
      virtual ~Deleter() {}
      virtual void operator() (Object* p); 
      virtual int Delete(Object* p) { delete p; return 0; }
      WeakPointer mPointer;
    };

  protected:
    void LockThis() { mThisPointerLocked = mThisPointer.lock(); }	
    void UnLockThis() { mThisPointerLocked = Pointer(); }
    //    Object::Pointer GetThis() const { return mThisPointer.lock(); } 
    template <class U>
    boost::shared_ptr<U> GetThisPointer() const 
    {
      return boost::dynamic_pointer_cast<U>(mThisPointer.lock());
    }
    template <class U>
    static boost::shared_ptr<U> MakePointer(U* s, bool lock = false)
    {									
      if (s->mThisPointer.lock())					
	{								
//EED 2018-07-11 minwg64 
	  boost::shared_ptr<U> p = s->template GetThisPointer<U>();
	  if (!lock) s->mThisPointerLocked.reset();
	  return p;
	}								
      boost::shared_ptr<U> p = boost::shared_ptr<U>(s,Object::Deleter());
      static_cast<Object::Deleter*>					
	(p._internal_get_deleter(typeid(Object::Deleter)))		
	->mPointer = p;							
      s->mThisPointer = p;						
      Object::InsertInObjectList(p);					
      if (lock) s->LockThis();						
      return p;							
    }									
    template <class U, class D>
     static boost::shared_ptr<U> MakePointer(U* s, 
					     const D& del,
					     bool lock = false)
    {									
      if (s->mThisPointer.lock())					
	{			
//EED 2018-07-11 minwg64 
	  boost::shared_ptr<U> p = s->template GetThisPointer<U>();
	  if (!lock) s->mThisPointerLocked.reset();
	  return p;
	}								
      boost::shared_ptr<U> p = boost::shared_ptr<U>(s,del);
      static_cast<D*>					
	(p._internal_get_deleter(typeid(D)))		
	->mPointer = p;							
      s->mThisPointer = p;						
      Object::InsertInObjectList(p);					
      if (lock) s->LockThis();						
      return p;							
    }									
     
  private:
    typedef std::set<boost::weak_ptr<Object> > ObjectListType;
    static ObjectListType mgObjectList;
    static ObjectListType mgPackageList;
    WeakPointer mThisPointer;						
    Pointer mThisPointerLocked;						

  }; 
  
#define BBTK_OBJECT_DEFINE_SELF(CLASS) public : typedef CLASS Self; 
  
#define BBTK_FORWARD_DECLARE_POINTER(CLASS)			\
  typedef boost::shared_ptr<CLASS> CLASS ## Pointer;		\
    typedef boost::weak_ptr<CLASS> CLASS ## WeakPointer;


#define BBTK_OBJECT_MINIMAL_INTERFACE					\
  public:								\
  typedef boost::shared_ptr<Self> Pointer;				\
    typedef boost::weak_ptr<Self> WeakPointer;				\
    friend struct Object::Deleter;					
  //private:								

  // does not work : why ?
  //      boost::get_deleter<Deleter,Pointer>(pt)->mPointer = pt; 
  //
  
#define BBTK_OBJECT_MINIMAL_INTERFACE_WITH_SELF(CLASS)	\
  public : typedef CLASS Self;				\
    BBTK_OBJECT_MINIMAL_INTERFACE;		

 
#define BBTK_OBJECT_INTERFACE(CLASS)				    \
  BBTK_OBJECT_MINIMAL_INTERFACE_WITH_SELF(CLASS);		    \
public:								    \
  std::string GetObjectName() const;				    \
    std::string GetObjectInfo() const ;				    \
    size_t GetObjectSize() const ;				    \
    size_t GetObjectInternalSize() const ;			    \
    size_t GetObjectRecursiveSize() const ;			    \
protected:							    \
    CLASS();							    \
    CLASS(const CLASS&);						    \
    ~CLASS();					    
  
#define BBTK_OBJECT_INTERFACE_NO_CONDES(CLASS)				    \
  BBTK_OBJECT_MINIMAL_INTERFACE_WITH_SELF(CLASS);		    \
public:								    \
  std::string GetObjectName() const;				    \
    std::string GetObjectInfo() const ;				    \
    size_t GetObjectSize() const ;				    \
    size_t GetObjectInternalSize() const ;			    \
    size_t GetObjectRecursiveSize() const ;			    

#define BBTK_ABSTRACT_OBJECT_INTERFACE(CLASS)				\
  public : typedef CLASS Self;						\
    BBTK_OBJECT_MINIMAL_INTERFACE;					\
protected:								\
    CLASS();								\
    CLASS(const CLASS&);							\
    virtual ~CLASS();					    
  
  //=======================================================================
  // A struct with one static instance 
  // just to print object list info after main
  class BBTK_EXPORT StaticInitTime
  {
  public:
    StaticInitTime();
    ~StaticInitTime();


    static bool PrintObjectListInfo;
  private:
    static bbtk::Object mObject;
  };
  
  
  /*
  template <class T, class U>
  inline T BruteForceDownCastPointer(class U p)
  {
    
  }
  */

}// namespace bbtk

#endif

