/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkTranscriptor.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.16 $
=========================================================================*/



/**
 *  \file 
 *  \brief class Transcriptor: level 0 of script C++ translation (header)
 */
/**
 *  \class bbtk::Transcriptor 
 *  \brief class Transcriptor: level 0 of script C++ translation 
 */
 
#ifndef __bbtkTranscriptor_h__
#define __bbtkTranscriptor_h__

#include "bbtkVirtualExec.h"

#include "bbtkSystem.h"
#include <iostream>

namespace bbtk
{

  class /*BBTK_EXPORT*/ Transcriptor : public VirtualExec
  {
    BBTK_OBJECT_INTERFACE(Transcriptor);
    typedef VirtualExec Superclass;
  public:
    static Pointer New(const std::string& filename);

   
    /// Sets the inputs of the workspace : 
    void SetInputs(const std::map<std::string,std::string>& m) { mInputs = m; }

    /// Puts the executer in "no exec" mode, 
    /// which creates but does not execute pipelines 

    //void SetNoExecMode(bool b) { mNoExecMode = b; }
    void SetNoExecMode(bool b)
    {
       if (b)
          m_Fp << "  e->SetNoExecMode(true);" << std::endl;
       else
          m_Fp << "  e->SetNoExecMode(false);" << std::endl;       
    }

    void SetNoErrorMode(bool b)
    {
       if (b)
          m_Fp << "  e->SetNoErrorMode(true);" << std::endl;
       else
          m_Fp << "  e->SetNoErrorMode(false);" << std::endl;       
    }

    bool GetNoExecMode() const { return false; }
    bool GetNoErrorMode() const { return false; }

    /// Sets the mode of dialog of the executer for Root inputs 
    void SetDialogMode(DialogModeType t) { mDialogMode = t; }

    /// Loads a package
    void LoadPackage(const std::string &name );

    /// Unloads a package
    void UnLoadPackage(const std::string &name );

    /// Starts a package block	
    void BeginPackage (const std::string &name );

    /// Ends a package block
    void EndPackage ();

    /// Starts the definition of a new ComplexBlackBox in package pack
    /// scriptfilename is the file from which the def is read
    void Define (const std::string &name,
		 const std::string& pack,
		 const std::string &scriptfilename);

    /// Sets the file name to use for the current definition
    /// (Used to set it after the Define command)
    void SetCurrentFileName (const std::string &name );

	void SetTypeOfScript_Application();
	  
    /// End the definition of a ComplexBlackBox
    void EndDefine ();

    /// Sets the kind of the currently defined ComplexBlackBox
    void Kind(const std::string& kind);

    /// Creates a new black box in current complex box
    void Create ( const std::string& boxType, const std::string& boxName);

    /// Destroys a black box
    void Destroy (const std::string &boxName);

    /// Clears the currently defined ComplexBlackBox
    void Clear();

    /// Connects the output boxOutput to the input boxInput
    void Connect (const std::string &boxfrom,
		  const std::string &output,
		  const std::string &boxto,
		  const std::string &input);

    /// Updates the box 
    void Execute(const std::string &box);

    /// Defines an input of the current complex box
    void DefineInput (const std::string &name,
		      const std::string &box,
		      const std::string &input,
		      const std::string &help);

    /// Defines an output of the current complex box
    void DefineOutput (const std::string &name,
		       const std::string &box,
		       const std::string &output,
		       const std::string &help);     
  
    /// sets the input of the box with the value
    void Set (const std::string &box, 
	      const std::string &input, 
	      const std::string &value);

    /// gets the output of the box
    std::string Get (const std::string &box, 
		     const std::string &output);

    /// changes the workspace name
    void SetWorkspaceName( const std::string& n );

    ///Adds the authorName to the Box author list
    void Author(const std::string &authorName);

    ///Adds the Categories to the Box category list
    void Category(const std::string &category);

    /// The description string which explains what does the ComplexBox
    void Description(const std::string & d);

    /// prints the list off the boxes of the current box
    void PrintHelpListBoxes();

   /// Generate a HTML with a gif file with the actual pipeline (Graphviz-dot needed). Returns the file path
    std::string ShowGraph(const std::string &nameblackbox, 
			  const std::string &detailStr, 
			  const std::string &levelStr,
			  const std::string &output_file,
			  const std::string &custom_header,
			  const std::string &custom_title,
			  bool system_display = true);

   /// Generate a HTML with a gif file with the actual pipeline (Graphviz-dot needed). Returns the file path
    std::string ShowGraphInstances(const std::string &nameblackbox, int detail, int level, bool system_display=true);

    /// Prints help on a black box
    void PrintHelpBlackBox(const std::string &nameblackbox, const std::string &detailStr, const std::string &levelStr);

      void Reset();
 
  /// Sets the level of message for kind
    void SetMessageLevel(const std::string &kind,
                         int level);

   /// Prints help on the messages
    void HelpMessages();
    
    /// Prints the string ... MORE : TO EXPLAIN 
    void Print(const std::string & message);

  //  static const std::string& GetObjectDescription()
  //  { static std::string s("Executer"); return s; }

  protected:

  private:
    /// Constructor
    Transcriptor(const std::string& filename);

    /// The input values of the Root ComplexBlackBox
    std::map<std::string,std::string> mInputs;
    
    /// Dialog mode
    DialogModeType mDialogMode;
    
    /// File Pointer, to hold generated C++ code.
    // std::ofstream *m_Fp;
    std::ofstream m_Fp;
  };
}
#endif
