/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkVirtualExec.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.21 $
=========================================================================*/



/**
 *  \file 
 *  \brief class VirtualExec: level 0 of script execution (header)
 */
/**
 *  \class bbtk::VirtualExec 
 *  \brief class VirtualExec: level 0 of script execution 
 */
  
#ifndef __bbtkVirtualExec_h__
#define __bbtkVirtualExec_h__

#include "bbtkSystem.h"
#include "bbtkComplexBlackBoxDescriptor.h"
#include "bbtkFactory.h"

namespace bbtk
{

  class Interpreter;
  BBTK_FORWARD_DECLARE_POINTER(Interpreter);

  class BBTK_EXPORT VirtualExec : public Object
  {
    BBTK_ABSTRACT_OBJECT_INTERFACE(VirtualExec);
  public:
    typedef enum
    {
       NoDialog,
       TextDialog,
       GraphicalDialog
    }
    DialogModeType;


    /// Sets the interpreter who uses it 
    void SetInterpreter(InterpreterPointer i) { mInterpreter = i; }
    /// Gets the interpreter who uses it 
    InterpreterPointer GetInterpreter(); // { return mInterpreter.lock(); }
    /// Gets the interpreter who uses it (const)
    //    const Interpreter::Pointer GetInterpreter() const { return mInterpreter; }

    /// Gets the factory used by the executer
    virtual Factory::Pointer GetFactory() { return Factory::Pointer(); }
    /// Gets the factory used by the executer (const)
    // virtual const Factory* GetFactory() const { return 0; }
    
    /// Sets the inputs of the workspace : 
    virtual void SetInputs(const std::map<std::string,std::string>& m) = 0;

    /// Puts the executer in "no exec" mode, 
    /// which creates but does not execute pipelines 
    virtual void SetNoExecMode(bool b) = 0;

    virtual bool GetNoExecMode() const = 0;

    /// Sets the mode of dialog of the executer for Root inputs 
    virtual void SetDialogMode(DialogModeType t) = 0;

    /// Puts the executer in "no error" mode, 
    /// Errors do not stop execution (but warnings are produced)
    virtual void SetNoErrorMode(bool b) = 0;
    virtual bool GetNoErrorMode() const = 0;
    //=================================================================
   /// Loads a package
    virtual void LoadPackage(const std::string &name ) = 0;

    /// Unloads a package
    virtual void UnLoadPackage(const std::string &name ) = 0;

    /// Starts a package block	
    virtual void BeginPackage (const std::string &name ) = 0;

    /// Ends a package block
    virtual void EndPackage () = 0;

    /// Starts the definition of a new ComplexBlackBox in package pack
    /// scriptfilename is the file from which the def is read
    virtual void Define (const std::string &name,
		 const std::string& pack,
		 const std::string &scriptfilename) = 0;
 
    /// Sets the file name to use for the current definition
    /// (Used to set it after the Define command)
    virtual void SetCurrentFileName (const std::string &name ) = 0;
	  
	  virtual void SetTypeOfScript_Application( ) = 0;
	  
    /// End the definition of a ComplexBlackBox
    virtual void EndDefine () = 0;

    /// Sets the kind of the currently defined ComplexBlackBox
    virtual void Kind(const std::string& kind) = 0;

    /// Creates a new black box in current complex box
    virtual void Create ( const std::string& boxType, const std::string&
    boxName) = 0;

    /// Destroys a black box
    virtual void Destroy (const std::string &boxName) = 0;

    /// Clears the currently defined ComplexBlackBox
    virtual void Clear() = 0;

    /// Connects the output boxOutput to the input boxInput
    virtual void Connect (const std::string &boxfrom,
		  const std::string &output,
		  const std::string &boxto,
		  const std::string &input) = 0;

    /// Executes the box 
    virtual void Execute (const std::string &box) = 0;

    /// Defines an input of the current complex box
    virtual void DefineInput (const std::string &name,
		      const std::string &box,
		      const std::string &input,
		      const std::string &help) = 0;

    /// Defines an output of the current complex box
    virtual void DefineOutput (const std::string &name,
		       const std::string &box,
		       const std::string &output,
		       const std::string &help) = 0;     
  
    /// sets the input of the box with the value
    virtual void Set (const std::string &box, 
	      const std::string &input, 
	      const std::string &value) = 0;

    /// gets the output of the box
    virtual std::string Get (const std::string &box, 
		     const std::string &output) = 0;

    /// changes the workspace name
    virtual void SetWorkspaceName( const std::string& n ) = 0;

    ///Adds the authorName to the Box author list
    virtual void Author(const std::string &authorName) = 0;

    ///Adds the categories to the Box categoris list
    virtual void Category(const std::string &categories) = 0;

    /// The description string which explains what does the ComplexBox
    virtual void Description(const std::string &d) = 0;

    /// prints the list of the boxes of the current box
    virtual void PrintHelpListBoxes() = 0;

   /// Generate a HTML file with a gif file with the actual pipeline (Graphviz-dot needed). Returns the file path
    virtual std::string ShowGraph(const std::string &nameblackbox, 
			  const std::string &detailStr, 
			  const std::string &levelStr,
			  const std::string &output_file,
			  const std::string &custom_header,
			  const std::string &custom_title,
			  bool system_display = true) = 0;

   /// Generate a HTML with a gif file with the actual pipeline (Graphviz-dot needed). Returns the file path
    virtual std::string ShowGraphInstances(const std::string &nameblackbox, int detail, int level, bool system_display=true) = 0;

    /// Prints help on a black box
    virtual void PrintHelpBlackBox(const std::string &nameblackbox, const std::string &detailStr, const std::string &levelStr) = 0;
    
    virtual void Reset() = 0;
    
    /// Sets the level of message for kind
    virtual void SetMessageLevel(const std::string &kind, int level) = 0;
    
    /// Prints help on the messages
    virtual void HelpMessages() = 0;

    /// Prints the string ... MORE : TO EXPLAIN 
    virtual void Print(const std::string & message) = 0;

    virtual ComplexBlackBoxDescriptor::Pointer GetWorkspace() 
    { return  ComplexBlackBoxDescriptor::Pointer(); }
  
  //  static const std::string& GetObjectDescription() = 0;
  //  { static std::string s("VirtualExec"); return s; }
  protected:

  private:

   /// The interpreter which uses it (0 if none)
    InterpreterWeakPointer mInterpreter;

  };
}
#endif
