/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkWxStreamRedirector.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.5 $
=========================================================================*/




/**
 * \brief Short description in one line
 * 
 * Long description which 
 * can span multiple lines
 */
/**
 * \file 
 * \brief 
 */
/**
 * \class bbtk::
 * \brief 
 */


#ifdef USE_WXWIDGETS

	
#ifndef __bbtkWxStreamRedirector_h__
#define __bbtkWxStreamRedirector_h__


#include "bbtkWx.h"


namespace bbtk
{


// On Windows when compiling a dll, wx prevents the compilation
// of the class wxStreamToTextRedirector (why ? it is a nightmare...)
// The blocking symbol is wxHAS_TEXT_WINDOW_STREAM.
// Note also that wxStreamToTextRedirector use the fact that wx is 
// compiled with the option WX_USE_STD_STREAMS in which case 
// wxTextCtrl inherits from std::streambuf and the redirection 
// can be done simply by setting the std::cout buffer to the 
// one of the wxTextCtrl. 
// So on windows, we have to redirect manually std::cout to mwxTextHistory.  
// Finally, on all systems we made our redirection class to redirect both to
// the WxConsole and to printf in order to get a console trace when 
// the appli crashes (we could also imagine to log in a file...)
// This is why we finally wrote our own redirection which is crossplatform
// (drawback : not optimal on Unix platform; we could think of 
// a particular implementation...).

  //================================================================
  /// Redirects std::cout to a wxTextCtrl and optionally to printf also
  class WxStreamRedirector   : public std::streambuf
  {	  

  public:

    WxStreamRedirector(std::ostream& redirect,
				 wxTextCtrl *text, 
				 const wxColour& colour = *wxBLACK,
				 bool doprintf=true,
				 int bufferSize=1000)
      : mText(text),
	mPrintf(doprintf),
	m_ostr(redirect),
	mColour(colour)
    {
      if (bufferSize)
        {
	  char *ptr = new char[bufferSize];
	  setp(ptr, ptr + bufferSize);
        }
      else
	setp(0, 0);
      
      m_sbufOld = m_ostr.rdbuf();
      m_ostr.rdbuf(this);
    }
    
    ~WxStreamRedirector()
    {
      sync();
      delete[] pbase();
      m_ostr.rdbuf(m_sbufOld);
    }

   virtual void writeString(const std::string &str) 
    {
      const wxTextAttr& style = mText->GetDefaultStyle();
      mText->SetDefaultStyle(mColour);
      mText->AppendText(std2wx(str));
      mText->SetDefaultStyle(style);
     
      if (mPrintf) 
	{
	  printf("%s",str.c_str());
	}
    }

  private:
    wxTextCtrl* mText;
    // 
    bool mPrintf;
    // the stream we're redirecting
    std::ostream&   m_ostr;
    // the old streambuf (before we changed it)
    std::streambuf *m_sbufOld;
    //
    wxColour mColour;

  private:
    int	overflow(int c)
    {
      sync();
      
      if (c != EOF)
        {
	  if (pbase() == epptr())
            {
	      std::string temp;
	      temp += char(c);
	      writeString(temp);
            }
	  else
	    sputc(c);
        }
      
      return 0;
    }

    int	sync()
    {
      if (pbase() != pptr())
        {
	  int len = int(pptr() - pbase());
	  std::string temp(pbase(), len);
	  writeString(temp);
	  setp(pbase(), epptr());
        }
      return 0;
    }
  };

  //================================================================

} // namespace bbtk

#endif // __bbtkWxStreamRedirector_h__

#endif //USE_WXWIDGETS
