/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#ifndef __OTULINE_MODEL_MANAGER__
#define __OTULINE_MODEL_MANAGER__


//------------------------------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <iostream> 
#include <sstream>
#include <map>
#include <vector>
#include <string>


#include "ContourThing.h"
#include "AxeThing.h"
#include "ImageSourceThing.h"
#include "ImageSectionThing.h"
#include "InstantMembersNameList.h"
#include "OutlineGroup.h"
#include "ContourWorkspace.h"
#include "CommandsHandler.h"
#include "InstantMembersNameList.h"
#include "PrefixMaxKeyGenerator.h"

//#include "../kernel_Environment/InstantHandler.h" //Is not working
//#include "../kernel_Environment/Instant.h" //Is included in the SomeEnvironment template
//#include "../kernel_Environment/SomeEnvironment.h"
#include "SomeEnvironment.h"

class ContourWorkspace; 

class OutlineModelManager{

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------
public:

	/*
	* Creates the outline manager
	*/
	OutlineModelManager( SomeEnvironment<ImageSourceThing *>* imSourceEnv, SomeEnvironment<ImageSectionThing *>* imSectionEnv, SomeEnvironment<AxeThing *>* axesEnv, SomeEnvironment<ContourThing *>* contourEnv );


	/*
	* Destroys the outline manager
	*/
	~OutlineModelManager();

//------------------------------------------------------------------------------------------------------------
// Methods
//------------------------------------------------------------------------------------------------------------

	/*
	* Creates and sets a workSpace object
	* @return Retourns the created workSpace
	*/
	ContourWorkspace * createWorkSpace();

	/*
	* Sets the workSpace object
	* @param  aWorkSpace The workSpace to set 
	*/
	void setWorkSpace( ContourWorkspace * aWorkSpace );

	/*
	* Executes a command over an outline object
	* @param imaKName Is the key name of the outline
	* @param theCommand Is the command to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/
	bool executeCommand_OutlineModel(std::string outKName, CommandObject * theCommand, bool fromRegistration);

	/*
	* Executes a command identifying which actions have to realize before executing it.
	* @param theCommand Is the command to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/
	bool executeCommand(CommandObject * theCommand, bool fromRegistration=false);

	/*
	* Executes a command queue identifying which actions have to realize before executing it, starting in from the front of it.
	* @param executionQueue Is the command queue to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/
	bool executeCommandsQueue(std::deque<CommandObject *> executionQueue, bool fromRegistration=false);

	/*
	* Sets the concepts of the environments and includes the concepts managed by the program 
	* @conceptsScript Is the script for with the concepts descrition
	* @return Returns true if successful insert of the given concepts, false otherwise
	*/
	bool setUserConcepts(std::string conceptsScript);

	/*
	* Add a concept to all the environments 
	* @param theConceptName Is the name of the new concept
	* @param conceptSize Is the size of the concept, that represent the ammount of concept posible instances 
	* @return Returns true if successful insert of concept, false otherwise
	*/		
	bool addUserConcept( std::string theConceptName, int conceptSize );
	
	/**
	* Gets the outlines at an instant form the wrapping reference
	* @param anInstant The instant wondered to get outlines at
	* @param ifAnnotate Indicates if it is needed to annotate the searched outlines
	* @return The vector to the outlines at the given instance
	*/	
	std::vector<NameWrapper *> getActualInstantOutlines ();

	/**
	* Gets the outlines at an instant form the outline's environment
	* @param anInstant The instant wondered to get outlines at
	* @param ifAnnotate Indicates if it is needed to annotate the searched outlines
	* @return The vector to the outlines at the given instance
	*/
	std::vector<ContourThing**> getOutlinesAtInstant(Instant * anInstant,  bool ifAnnotate = true);

	/**
	* Gets the name of all outlines 
	* @return The vector of strings of all outlines
	*/
	std::vector<std::string> GetLstNameThings();

	std::vector<std::string> GetLstNameThingsStatic();


	/**
	* Gets the outlines at a specific form the outline's environment
	* @param aGroupName The name of the group containing the outlines names to get
	* @return The vector to the outlineGroups at the given group
	*/ 
	std::vector<ContourThing*> getOutlinesFromGroup(std::string aGroupName);

	/**
	* Adds to storing at (outlineGroups) an outline group given its name and the group
	* @param theOutlineName The name of the group containing the outlines names to add
	* @param theGroup The outlines group to add
	* @return Returns true
	*/ 
	bool addOutlinesGroup( std::string theOutlineName, OutlineGroup * theGroup );

	/*
	* Gets the an outline given the keyName used to be indexed in the outlines's environment
	* @param outKName Is the outline keyName to search
	* @return  The corresponding unique outline with the given key name
	*/
	ContourThing* getOutlineByKeyName(std::string outKName);

	/*
	* Gets the an imageSourceThing given the keyName used to be indexed in the ImageSourceThing's environment
	* @param outKName Is the imageSourceThing keyName to search
	* @return  The corresponding unique outline with the given key name
	*/
	ImageSourceThing* getImageSourceThingByKeyName(std::string iSeourceKName);

	/*
	* Gets the an imageSectionThing given the keyName used to be indexed in the ImageSectionThing's environment
	* @param outISectionName Is the imageSectionThing keyName to search
	* @return  The corresponding unique outline with the given key name
	*/
	ImageSectionThing* getImageSectionThingByKeyName(std::string iSectionName);

	/*
	* Creates an outlineThing with a given name, if no name is given it would have an automatic
	* @param aName The name for the outlineThing
	* @return Returns the key name of the created outline, or ""if it wasn't created
	*/
	//bool createOutline(std::string aName, ContourThing * &theOutline);
	std::string createOutline(manualBaseModel * model, std::vector<int>  theInstantData, std::string aName = "" );

	/*
	* Creates an axeThing with a given name
	* @param aDescription The description for the axeThing
	* @return Returns true if the creation of the axe was successful
	*/
	//bool createAxe(std::string aDescription,AxeThing * &theAxe);
	bool createAxe(std::string aDescription, Instant * theInstantData);

	/*
	* Creates an imageSourceThing with a given name
	* @param aName The name for the imageSourceThing
	* @return Returns true if the creation of the imageSource was successful
	*/
	//bool createImageSource(std::string aName, ImageSourceThing * &imageSource);
	bool createImageSource(std::string aSource, Instant * theInstantData);

	/*
	* Creates an imageSectionThing with a given name
	* @param aSecImageData The name for the imageSectionThing
	* @return Returns true if the creation of the imageSection was successful
	*/
	//bool createImageSection(std::string aSecImageData, ImageSectionThing * &imageSection);
	bool createImageSection(std::string aSecImageData, Instant * theInstantData);

	/*
	* Adds an outlineThing 
	* @param theOutline The outline/contour (thing)
	* @param theInstantData Is the instant for the outline to add
	* @return Returns the key name of the created outline, or ""if it wasn't created
	*/
	std::string addOutline(ContourThing * theOutline, Instant * theInstantData);

	/*
	* Remove an outlineThing 
	* @param theOutline The outline/contour (thing)
	*/
	void removeOutline( std::string ss );

	/*
	* Adds an axeThing 
	* @param thaAxe The axe (thing)
	* @param theInstantData Is the instant for the axe to add
	* @return Returns true if the addition of the axe was successful
	*/
	bool addAxe( AxeThing * thaAxe, Instant * theInstantData);

	/*
	* Adds an imageSourceThing 
	* @param imgageSource The image source (thing)
	* @param theInstantData Is the instant for the source to add
	* @return Returns true if the addition of the imageSource was successful
	*/
	bool addImageSource(ImageSourceThing * imageSource, Instant * theInstantData);

	/*
	* Adds an imageSectionThing with a given name, if no name is given it would have an automatic
	* @param aName The imageSection (thing)
	* @param theInstantData Is the instant for the imageSection to add
	* @return Returns true if the addition of the imageSection was successful
	*/
	bool addImageSection(ImageSectionThing * imageSection, Instant * theInstantData);

	/*
	* Annotates an outline keyname at the actual instant
	* @param kOutlineName The key name to annotate
	* @param theRealName The real name asigned to the outline
	*/
	void annotateOutlineWrap(std::string kOutlineName, std::string theRealName);

	/*
	* Annotates a set of outline keynames-real names wrapping at the actual instant
	* @param kNamesVector The key names vector to annotate
	* @param theOutlinesVector The outlines pointers vector to get the real names from
	*/
	void annotateOutlinesWrap(std::vector<std::string> keyNames, std::vector<ContourThing **> theOutlinesVector);

	/*
	* Annotates the actual outline keyName-real name wrapping at the actual instant
	* @param actualKeyOutline The key name to annotate
	* @param theRealName The real name asigned to the actual outline
	*/
	void annotateActualOutlineWrap(std::string actualKeyOutline, std::string theRealName);

	/*
	* Annotate the actual axe keyName-real name wrapping at the actual instant
	* @param actualKeyAxe The key name to annotate
	* @param theRealName The real name asigned to the actual axe
	*/
	void annotateActualAxeWrap(std::string actualKeyAxe, std::string theRealName);

	/*
	* Annotate the annotateActualSection image keyName-real name wrapping at the actual instant
	* @param actualKeyImage The key name to annotate
	*/
	void annotateActualSectionImageWrap(std::string actualKeyImage, std::string theRealName);//---BORRAR...

	/*
	* Annotate the annotateActualSource image keyName-real name wrapping at the actual instant
	* @param actualKeyImage The key name to annotate
	*/
	void annotateActualSourceImageWrap(std::string actualKeyImage, std::string theRealName);//---BORRAR...
	/*
	* Sets the actual instant and manage the search of the corresponding elements with the specified instant in all the enviornments
	* @param newActualInstantData Is the instant data
	*/
	void setInstant(Instant * newActualInstantData);

	/*
	* Gets the an axe with a given keyName
	* @axeKName The axe keyName for searching in the axes environment
	* @return The corresponding axe
	*/
	AxeThing * getAxeByKeyName(std::string axeKName);

	/*
	* Gets the instants of a specific outline
	* @param thekName Is the name of the outline
	* @return The instants set 
	*/
	std::vector<Instant *> getOutlineInstantsByName(std::string thekName);

	/*
	* Gets all instants outlines
	* @return The instants set 
	*/
	std::vector<Instant *> getOutlineInstants();

	/*
	* Includes an instant to the specified axe
	* @param outKName Is outline key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool includeOutlineInstant(std::string outKName,Instant * anInstantData);

	/*
	* Includes an instant to the specified axe
	* @param axeKName Is axe key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool includeAxeInstant(std::string axeKName,Instant * anInstantData);

	/*
	* Includes an instant to the specified image section
	* @param imaKName Is the image section key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool includeImageSectionInstant(std::string imaKName,Instant * anInstantData);

	/*
	* Includes an instant to the specified image source
	* @param imaKName Is the image section key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool includeImageSourceInstant(std::string imaKName,Instant * anInstantData);

	/*
	*	Method that retorns the name of each concept and the size of it.
	*	@param conceptNameVect, Vector in which is disposed to be setted the name for each of the included concepts
	*	@param conceptSizeVect, Vector in which is disposed to be setted the size for each of the included concepts
	*/
	void getConceptsInformation(std::vector<std::string>& conceptNameVect, std::vector<int>& conceptSizeVect);
	
	/*
	* Gets the contourWorspace 
	* @return Returns the workspace
	*/
	ContourWorkspace * getContourWorkspace();

	/*
	* Gets the number of groups in the model management 
	* @return Returns the size of 
	*/
	int getGroupsCount();
	
	/*
	* Cleans the outline model manager and its dependencies
	*/
	void clean();	

	void removeAllOutlines();
	std::string createCopyContourOf( std::string anExistingKName, std::vector<int> &instantNoTouchData );

	void SaveThingName( FILE *pFile, FILE *pFileData, std::string nameThing );

	void ChangeContourOfList(std::string keyName, Instant *instant);
	int IsPartOfStaticList(std::string ss);

	Instant* getInstant();

private : 

	/*
	* Update the registered objects in the InstantMemebersNameList, is the one that really changes the instant in the model
	*/
	void updateToActualInstant();
	
	/*
	* Sets the automatic managed concepts including them in the environments. That are at the beginning of the instant vector for the corresponding environments.
	* @return Returns true if successful insert of the automatic concepts, false otherwise
	*/
	bool setAutomaticConcepts();
	

	

	
//------------------------------------------------------------------------------------------------------------
// Constants
//------------------------------------------------------------------------------------------------------------

	

//------------------------------------------------------------------------------------------------------------
// Attributes
//------------------------------------------------------------------------------------------------------------

	
private: 	

	/*
	* Represents the keyName of the actual gruop in the map of gruops
	*/
	std::string actualGruopID;

	/*
	* Represents the environment instant iterator of axes
	*/
//	InstantHandler<AxeThing *> * axesITER;
	
	/*
	* Represents the environment of axes
	*/
	SomeEnvironment<AxeThing *>* axesEnvironment;

	/*
	* Represents the environment instant iterator of section images 
	*/
//	InstantHandler<ImageSectionThing *> * sectionITER;

	/*
	*  Represents the environment of section images 
	*/
	SomeEnvironment<ImageSectionThing *>* imagesSectionEnvironment;

	/*
	* Represents the environment instant iterator of source's images (Generally a complete volume data per image) 
	*/
//	InstantHandler<ImageSourceThing *> * imageITER;

	/*
	* Represents the environment of source images (Generally a complete volume data per image) 
	*/
	SomeEnvironment<ImageSourceThing *>*  imageSourceEnvironment;

	/*
	* Represents the environment instant iterator of outlines (outlines)
	*/
//	InstantHandler<ContourThing *> * outlinesITER;

	/*
	* Represents the environment of outlines (outlines)
	*/
	SomeEnvironment<ContourThing *>*  outlinesEnvironment;

	/*
	* Represents the table of outlines groups
	*/
	std::map< std::string,OutlineGroup * > outlineGroups;

	/*
	* Represents the workspace
	*/
	ContourWorkspace * workSpace;

	/*
	* Represents the actual instant members list wrrapping (keyName, realName) of objects
	*/
	InstantMembersNameList * actualInstantWrapping;

	/*
	* Represents a flag indicating if the actualSourceImage has to be changed when changed instant
	*/
	bool changeSourceImage;

	int counterIdKey;
	PrefixMaxKeyGenerator keyGenerator;	


	/*const*/std::string axeConcept /*= "Axe"*/;
	/*const*/std::string axeDepthConcept /*= "Axe Depth"*/;

	Instant * actualInstant;
	
	//EED
	std::vector<ContourThing **> staticContourLst;
	
};
#endif
