/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */

/*=========================================================================

  Program:  instantHandler
  Module:    $RCSfile: InstantHandler.h,v $
  Language:  C++
  Date:      $Date: 2012/11/15 12:12:26 $
  Version:   $Revision: 1.2 $
  Objective: Handles the change of instant among the concept already inscribed
  Authot:	Monica Maria Lozano Romero

  Copyright: (c) 2007
  License:

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.


=========================================================================*/


//	MACRO
#ifndef __INSTANTHANDLER_H_INCLUDED__
#define __INSTANTHANDLER_H_INCLUDED__

// SYSTEM INCLUDES

#include <iostream>
#include <map>
#include <vector>
#include <string>

// PROJECT INCLUDES

#include "SomeEnvironment.h"
#include "ConceptInstantHandler.h"
#include "ViewerNDimensionsException.h"

// LOCAL INCLUDES

// FORWARD REFERENCES



/*
*	COMMENT
*	@param
*	@return
*/

template <class P>

class InstantHandler
	{
				public:
						//==========	CONSTANTS =======
						
						//====== LIFECYCLE ========
						InstantHandler(SomeEnvironment<P>* environment)throw (ViewerNDimensionsException);
						~InstantHandler();
													
						//====== OPERATIONS =======
						
						/*
						*	Adds a concept to handled in the instant, it means, when 
						*	the next instant is used we are going to change this concepts in the instant
						*	PRECONDITION
						*	the calls to this functions are made in order of position, where the last position (it means
						*	the higher position defined) is the concept thats is going to handled the real time
						*	THE LAST POSITION HANDLEDS THE REAL TIME DOESNT MATTER THE CONCEPT
						*	EXAMPLE
						*	The user wants to reproduce doctors, pacients and time, and positions given are:
						*	0=reproduce doctors, 1= reproduce pacients, 2= time, then  1,2 are going to be reproduced
						*	by PLUS PLUS, and time by REAL TIME
						*	@param nameConcept, name of the concept
						*	@param mode, mode of change
						*	@param position, position of reproduction
						*	@return true if succesful, false otherwise
						*/
						bool addConceptToHandled(std::string nameConcept,int mode,int position)throw (ViewerNDimensionsException);
						
						
						/*
						*	Next instant of the concepts to change
						*	@param 
						*	@return
						*/
						void nextInstant()throw (ViewerNDimensionsException);

						
						/*
						*	Previous instant of the actual concept
						*	@param 
						*	@return
						*/
						void previousInstant()throw (ViewerNDimensionsException);
						
						/*
						* Removes all the concept handled
						*/
						void removeAllConcepts();
						
						/*
						*	Removes the concept with the name given
						*	@param nameConcept, name of the concept to remove
						*	WARNING the user know wich concept is erasing, also knows the
						*	mode of other concepts, if the user is deleting the concept that
						*	handled the real time, should change the mode to other concept that exist
						*/
						void removeConceptToHandled(std::string nameConcept);

						/*
						* sets all the concept to handled in 0
						*/
						void resetConceptsHandled();
						
						/*
						* Pause
						*/
						//void pause();
						
						//====== INQUIRY =========
						
						/*
						* get the actual instant for the concept identified by the name
						*/
						Instant* getActualInstant()throw (ViewerNDimensionsException);
						
						
						/*
						* If the actual concept has next instant, it means, if the index of the concept
						* in the  instant is less that the maximum value
						* @return true if IS NOT in the maximum, false otherwise 
						*/
						bool hasNextInstant();			
						
						/*
						* If the concept has previous instant, it means, if the index of the concept
						* in the  instant is positive
						* @return true if IS NOT in zero, false otherwise 
						*/
						bool hasPreviousInstant();
						
						/*
						* returns the number con conceptsToHandled
						*/
						int getSizeConceptsToChange();

						/*
						* returns the number of possibles instants of the handler
						* it depends of the concepts added in conceptsTochange
						*/
						int getNumOfInstants();
						
						/*
						*	Returns the names of the concepts handled and its modes
						*	DEPRECATED
						*/
						void getConceptsInfo(std::vector<std::string>& namesConcepts,std::vector<int>& modes);
						/*
						*	Returns the name of the concept that handled the real time
						*	DEPRECATED
						*/
						void getTimeConcept(std::string& nameConcept);
						//====== ACCESS ==========
						/*
						*	Sets the environment that is going to be handled
						*	@param environment, a pointer to the environmen
						*	@return
						*/
						void setEnvironment(SomeEnvironment<P>* environment);
						/*
						*	Sets the actual instant of the instant handler
						*	@param instant
						*/
						void setActualInstant(Instant* instant);
						/*
						*	Sets the  time of reproduction to the concept
						*	that depends of the real time of the reproduction
						*	(miliseconds)
						*	@param  time
						*/
						void setTimeReproduction(double time);
						/*
						*	Sets the actual time of the reproduction,
						*	is the real time (seconds)
						*	@param actualTime
						*/
						void setActualTime(double actualTime);
						/*
						* Set the mode of a concept that's being handled
						*/
						void setModeOfConcept(std::string nameConcept,int mode);
						/*
						* Set handler
						*/
						void set();
						 /*
						* Set the increase Value for changing the instant
						*/
						void setIncreaseValue(int increase);
						/*
						* Set the decrease Value for changing the instant
						*/
						void setDecreaseValue(int decrease);
						
						
						
				private:
					
					//======= ATTRIBUTES=======
					
					/*
					*	actual instant of the InstantHandler
					*/
					Instant*	actualInstant;

					/*
					*	Environment that's being handled
					*/
					SomeEnvironment<P>* environment;

					/*
					* concepts to change in the actual instant
					*/
					std::vector<ConceptInstantHandler*> conceptsToChange;
					
					/*
					*	actual concept being manipulated
					*/
					ConceptInstantHandler* actualConcept;

					/*
					* if the actual concept has next
					*/
					bool hasNext;
					/*
					* if the actual concept has previous
					*/
					bool hasPrevious;

					/*
					*	actual time of the reproduction
					*/
					double actualTime;

					/*
					* time of pause
					*/
					double pauseTime;

					//====== PRIVATE METHODS=========
					/*
					*  Sets if the instant has next or not
					*/
						
					void setHasNext();
					/*
					*  Sets if the instant has previous or not
					*/
						
					void setHasPrevious();
					/*
					* actualize the actual index of the concepts handled
					*/
					void setConceptsToChange();
					/*
					*	resets all the next concepts to zero
					*/
					void resetNextConcepts();
					/*
					*	Sets the next index of concepts to handled to the maximum value
					*/
					void setNextConcepts();
					/*
					* this method locates the new actual concept
					* @param next, if has to locate the actual concept to the first concept that has next
					* next=true, else search for the first concept that has previous
					*/
					void toLocateActualConcept(bool next);

							
	};
//include the implementation of the template
#include "InstantHandler_Txx.h"
#endif
