/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#ifndef __creaImageIOTreeNode_h_INCLUDED__
#define __creaImageIOTreeNode_h_INCLUDED__

#include <creaImageIOTreeLevelDescriptor.h>
#include <creaImageIOTreeComparators.h>
#include<boost/filesystem/operations.hpp>
#include <vector>
#include <map>


namespace creaImageIO
{

  namespace tree
  {
	  /**
	* \ingroup Tree
	*/
    //=====================================================================
    /// Forward declaration of Tree
    class Tree;
    //=====================================================================  
    
    //=====================================================================
    /// Abstract class to store user data on a Tree node
    struct NodeData
    { 
      NodeData() {}
      virtual ~NodeData() {}
    };
    //=====================================================================


    //=====================================================================
    /// Node of an attributed Tree structure
    class Node
    {
    public:
      typedef std::map<std::string,std::string> AttributeMapType;

      /// Ctor with parent
      Node(Node* parent);
      /// Ctor with parent and attributes map 
      Node(Node* parent, const AttributeMapType& );
      /// Virtual destructor
      virtual ~Node();

      /// Initializes the attribute map i.e. creates the entries
      void InitializeAttributeMap();
 
      /// Returns the level descriptor of the node
      const LevelDescriptor& GetLevelDescriptor() const;


      /// Returns the tree to which the node belongs
      virtual Tree* GetTree() { return mParent->GetTree(); }
      /// Returns the tree to which the node belongs
      virtual const Tree* GetTree() const { return mParent->GetTree(); }
      /// Returns the level of the node in the tree
      virtual int GetLevel() const { return mParent->GetLevel()+1; }

      /// Returns the parent of the node
      Node* GetParent() const { return mParent; }

      /// Returns the number of children of the node.
      /// Warning : if the children are not loaded then might return 0
      ///           even if the node has children !
      ///           see TreeHandler::GetNumberOfChildren 
      unsigned int GetNumberOfChildren() const { return (int)mChildren.size(); }

      /// Returns true iff the node's children are loaded
      bool GetChildrenLoaded() const { return mChildrenLoaded; }

      /// Sets the node's children 
      void SetChildrenLoaded(bool l) { mChildrenLoaded = l; }

      /// The type of children container
      typedef std::vector<Node*> ChildrenListType;
      /// Returns the list of children
      ChildrenListType& GetChildrenList() { return mChildren; }
      /// Returns the list of children (const)
      const ChildrenListType& GetChildrenList() const { return mChildren; }

      /// Remove the given children from the children list
      int RemoveChildrenFromList(Node*);
	
	  /// Get the Attributes Map
      AttributeMapType& GetAttributeMap() { return mAttributeMap; }

	  /// Get the Attributes Map
      const AttributeMapType& GetAttributeMap() const { return mAttributeMap; }

	  /// Get the Attribute for a specific key
      const std::string& GetAttribute(const std::string& k) const;
      
	  /// Get the Attribute for a specific key without OS dependance (not implemented)
	  // TODO : backslash OS uniformity
	  const std::string& GetCleanAttribute(const std::string& k) const;

	  /// Set an Attribute for a specific key
      void SetAttribute(const std::string& k, const std::string& v);

	  /// Set an Attribute for a specific key(unsafe mode)
      void UnsafeSetAttribute(const std::string& k, const std::string& v)
      { mAttributeMap[k] = v; }
    
	  /// Get Descriptor for an Attribute
      const AttributeDescriptor& GetAttributeDescriptor(const std::string& k)const;
      
      /// Returns true if the KEY attributes of the node match those of the map provided
      bool Matches( const AttributeMapType& ) const;

      /// Returns the node data casted into the type T
      template<class T> T GetData() const 
      { if (mData!=0) return dynamic_cast<T>(mData); return 0; }

      /// Sets the node data. Deletes existing data if any.
	  void SetData(boost::shared_ptr<NodeData> d) {mData.reset(); mData = d; }//{ if (mData) delete mData; mData = d; }

      /// Sorts the children of the node 
      void SortChildren(const LexicographicalComparator&);

	  /// Print the node
      virtual void Print() const;

	  /// Get the Label of the node
      std::string GetLabel() const;
 

    private:
      /// The parent of the node
      Node* mParent;
      /// The list of children
      ChildrenListType mChildren;
      /// The map of attributes
      AttributeMapType mAttributeMap;
      /// User data
      boost::shared_ptr<NodeData> mData;
      /// Are the children loaded ?
      bool mChildrenLoaded;
      /// The number of children
      // int mNumberOfChildren;

    }; // class Node
    //=====================================================================

  } // namespace tree


} // namespace creaImageIO



#endif // #ifndef __creaImageIOTreeNode_h_INCLUDED__
