/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#ifndef __creaImageIOWxTreeView_h_INCLUDED__
#define __creaImageIOWxTreeView_h_INCLUDED__

#ifdef USE_WXWIDGETS

#include <creaImageIOTreeView.h>
#include <creaWx.h>

#include <wx/listctrl.h>
#include <wx/splitter.h>
//#include <vector>
const std::string empty_string("");
namespace creaImageIO
{

	  //=====================================================================
  /// Data stored by the list items
  struct ItemData
  {
    ItemData() : node(0), id(-1), attr(&empty_string) {}
    // The corresponding Node
    tree::Node* node;
    // The id ?
    int id;
    // The pointer on the current attribute string to sort on
    const std::string* attr;
    // Was the item previously selected ?
    // Useful for reselecting the item after sort
    bool selected;
  };
  /**
   * \ingroup View
   */
  //=====================================================================
  
  //=====================================================================
  /// Abstract class that handles the view of a Tree through its TreeHandler
  class WxTreeView : public wxPanel, virtual public TreeView
    {
    public:
      /// Ctor
      WxTreeView(TreeHandler*, GimmickView*, 
		 wxWindow* parent, const wxWindowID id);
      /// Virtual destructor
      virtual ~WxTreeView();

      
      /// Updates the view of a level given the selected items of upper level
      virtual void UpdateLevel( int );

      ///Removes selected nodes on given level
	  virtual void RemoveSelected(std::string &i_save);

	  ///Returns the last selected level
	  virtual unsigned int GetLastSelectedLevel(){return mLastLevel;}

      /// Callback for item selection
      void OnItemSelected(wxListEvent& event);

      /// Callback for item deselection
      void OnItemDeSelected(wxListEvent& event);
      
      /// Callback for column click
      void OnColClick(wxListEvent& event);
      
      /// Callback when the user needs the items sorted
      void OnPopupSort(wxCommandEvent& event);
      
      ///Callback when the user need the items filtered
      void OnPopupFilter(wxCommandEvent& event);

	  ///Callback when the user needs the item copied to the local disk
      void OnLocalCopy(wxCommandEvent& event);


  	  ///Callback when the user needs the item copied to the local disk
      void OnAnonymizer(wxCommandEvent& event);


	  ///Callback when the user needs to edit a field
      void OnEditField(wxCommandEvent& event);

	  ///Callback when the user needs to display alll dicom tags for a file
	  void OnDumpTags(wxCommandEvent &event);
      
	  ///Callback when the user needs to transfer data from storage to storage
	  void OnExportToStorage(wxCommandEvent &event);
      
      ///Callback on mouse click
      void OnMouseClick(wxMouseEvent& event);

	  /// Displays a menu for items
	  void OnItemMenu(wxListEvent &event);
	  
	  /// Gets the attributes that are being shown and the ones that have been blocked on a specific level
	  void GetAttributes(std::vector<std::string>& areShown, std::vector<std::string>& notShown, int level);

	  ///Sets the non visible attributes and refreshes the GUI
	  void SetNonVisibleAttributes(const std::vector<std::string>& notShown, int level);

	  ///Creates a new listctrl
	  void CreateCtrl(std::vector<std::string>& notShown, int nlevel);
  
	  ///Returns true if the attribute passed as a parameter is visible or not
	  bool IsAttributeVisible(const std::string& val, int level);
      
      /// Actual processing of item selection/deselection 
      /// Called by OnItemSelected and OnItemDeSelected
      //      void ProcessItem
    private:
		wxBoxSizer	*msizer;
      /// The struct holding the data for one level
      /// Holds the wx control and other data
      /// such as the vector of attribute keys corresponding to the columns
      struct LevelType
      {
			// The List Control
			wxListCtrl					*wxCtrl;
			wxSplitterWindow			*wxSplitter;
			std::vector<std::string> 	key;
			// The vector of currently selected nodes of the level
			std::vector<tree::Node*> 	Selected;
			// True iff the vector Selected is up to date
			bool SelectedUpToDate;
			// The column used for sorting
			unsigned int SortColumn;
			///Boolean that defines the direction of the sort
			///True is ascending order and false is descending
			bool SortAscending;
			//The vector of not shown attributes
			std::vector<std::string> notShownAtts; 
      };
      /// The vector of levels : one for each level of the tree      
      std::vector<LevelType> mLevelList;
      
      /// return the wxListCtrl of one level
      wxListCtrl* GetCtrl(int l) { return mLevelList[l].wxCtrl; }
      /// return the wxSplitter of one level
      wxSplitterWindow* GetSplitter(int l) { return mLevelList[l].wxSplitter; }
      //Returns the maximum number of levels
      int GetNumberOfLevels(){ return (int)mLevelList.size(); }
      /// Gets the user selected data from the level passed as a parameter
      /// Updates the vector if necessary
      const std::vector<tree::Node*>& GetSelected(int level);
      /// Set the bool SelectedUpToDate for level l
      void SetSelectedUpToDate(int l, bool v) { mLevelList[l].SelectedUpToDate = v; }
      /// Get the bool SelectedUpToDate for level l
      bool GetSelectedUpToDate(int l) { return mLevelList[l].SelectedUpToDate; }
       ///Validates the selected images
      void ValidateSelectedImages(bool isSelection);   
      ///Gets selected filenames
      void GetSelectedAsString(std::vector<std::string>&s);
	  ///Gets the filenames of the given nodes and returns them on the given vector. Is recursive.
	  void GetFilenamesAsString(const std::vector<tree::Node*>& nodes, std::vector<std::string>&s);
      /// Gets the next nodes on the list, be it up(true) or down(false).
      void GetNodes(std::vector<tree::Node*>& nodes, bool direction);
      /// Updates the view of a level given the selected items of upper level
      /// Recursive method
      virtual void RecursiveUpdateLevel( int );
      ///Selects all the elements of a level 
      void SelectAll(int level);
      ///UnSelects all the elements of a level 
      void UnSelectAll(int level);

	  void OnKeyDown(wxListEvent &event);
      /// Sorts the level
      void SortLevel(int level);
      
      /// Currently Selected Column
      int mColumnSelected;
      ///The last selected item on the list (left click)
      long mLastSelected;

	  ///The last selected item on the list (right click)
	  long mLastRightSelected;

	  ///The last selected level (by right click)
	  int mLastRightLevel;
      ///The color map
      typedef std::map<tree::Node*,wxColour> ColorMap;
      typedef std::pair<tree::Node*,wxColour> NodeColorPair;
      ColorMap mColorMap;
      ///Initial color palette
      std::vector<std::string> mColorPalette;

      wxMenu		*menu;
	  
      wxObject		*senderCtrl;
      int mAscendingID;
      int mDescendingID;
      int mFilterID;
      unsigned int mLastLevel;
      
	  wxMenu	*menuItem;
	  wxMenu 	*subExportMenu;
	  int mAnonymizingID;
	  int mLocalCopyID;
	  int mEditFieldID;
	  int mDumpID;
	  int mExportID;
	  int mExport2StorageID;
	  
      // If set to true then OnSelectedChanged returns immediately.
      // Used to do avoid useless process during multiple selections 
      // or sorting
      bool mIgnoreSelectedChanged;

      DECLARE_EVENT_TABLE()
    };
    // EO class WxTreeView
    //=====================================================================

	class RemoveAlertDlg : public wxDialog
	{
	public:
		RemoveAlertDlg(wxWindow *parent, 
					       wxString title,    
						   const wxSize& size);
		~RemoveAlertDlg();
		bool isChecked();

	private :
		void onCheck(wxCommandEvent &Event);
		bool mSave;
		wxCheckBox *mcheck;

	};


} // EO namespace creaImageIO


#endif // USE_WIDGETS
// EOF
#endif  
