/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

////////////////////////////////////////////////////////////////////////////////
// vector.h
// Creation : 24/02/2000
// Author   : Leonardo FLOREZ VALENCIA
//               l-florez@uniandes.edu.co
//               lflorez@creatis.insa-lyon.fr
// Copyright (C) 2000-2002 Leonardo FLOREZ VALENCIA
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
////////////////////////////////////////////////////////////////////////////////

#ifndef GTMLIB__MATH__VECTOR__HXX
#define GTMLIB__MATH__VECTOR__HXX

#include "mathdefs.h"
#include "vmfuncs.h"

/// Vector type IDs
//@{
#define ROW_VECTOR 0x01
#define COL_VECTOR 0x02
//@}

namespace gtm
{
    template< class T >
    class TMatrix;

    /** TVector class.
     *
     *  This class defines a C++ template to use mathematical vectors
     *  in a N space.
     *  @see TMatrix
     */
    template< class T >
    class TVector
    {
    public:

	/** Contructors.
	 *
	 *  @param N Cardinality.
	 *  @param data Initial data.
	 *  @param type Vector type (ROW_VECTOR/COL_VECTOR).
	 *  @param r Right object (vector or matrix).
	 *  @param block ANSI array.
	 *  @param copy Make a copy of given array?
	 */
	//@{
	/// Default constructor.
	TVector( uint32_t N = 3, T data = ( T )0, int type = COL_VECTOR  );
	/// Copy constructor.
	TVector( const TVector< T >& r );
	/// Use this to treat an ANSI array as a TVector.
	TVector( T* block, uint32_t N, bool copy = true, int type = COL_VECTOR );
	//@}

	/// Destructor.
	~TVector( ) {
	    if( _myMemory ) VectorFreeMemory< T >( _vector );
	};

	/// Size change
	void SetN( uint32_t N );

	/** Assignation operators.
	 *
	 *  @param r Right object (vector or matrix).
	 */
	//@{
	/// Vector assignation.
	TVector< T >& operator=( const TVector< T >& r );
	/// Matrix assignation (defined in class TMatrix).
	TVector< T >& operator=( TMatrix< T >& r );
	/// ANSI assignation (r size must be, at least, N).
	TVector< T >& operator=( T* r ) {
	    VectorAssignMemory< T >( _vector, r, _N );
	    return( *this );
	};
	/// Scalar assignation.
	TVector< T >& operator=( T r ) {
	    VectorAssignScalar< T >( _vector, r, _N );
	    return( *this );
	};
	//@}

	/** Comparation operators.
	 *
	 *  @param r Right object.
	 */
	//@{
	/// Equality.
	bool operator==( const TVector< T >& r );
	/// Inequality.
	bool operator!=( const TVector< T >& r );
	//@}

	/// Reference operator.
	T& operator()( uint32_t i ) {
	    return( _vector[ i ] );
	};
	/// ANSI (C/C++ array) reference.
	T* GetAnsiRef( ) {
	    return( _vector );
	};
	/// Vector's cardinality.
	uint32_t GetN( ) {
	    return( _N );
	};
	/// Vector's type (ROW_VECTOR/COL_VECTOR).
	int GetType( ) {
	    return( _type );
	};
	/// Mathematical norm (L2).
	T GetNorm( ) {
	    return( VectorNorm< T >( _vector, _N ) );
	};
	/// Normalizes the vector.
	void Normalize( ) {
	    VectorNormalize< T >( _vector, _vector, _N );
	};
	/// Dot product.
	T Dot( const TVector< T >& r ) {
	    return( VectorDotProduct< T >( _vector, r._vector, GTM_MIN( _N, r._N ) ) );
	};

	/** Binary operators.
	 *
	 *  @param r Right object (vector, matrix or scalar).
	 */
	//@{
	/// Addition.
	TVector< T > operator+( const TVector< T >& r );
	/// Substraction.
	TVector< T > operator-( const TVector< T >& r );
	/// Cross product.
	TVector< T > operator*( const TVector< T >& r );
	/// Matrix product (defined in class TMatrix).
	TMatrix< T > operator*( TMatrix< T >& r );
	/// Scalar product.
	TVector< T > operator*( T r );
	//@}

	/** Self-assignation binary operators.
	 *
	 *  @param r Right object (vector, matrix or scalar).
	 */
	//@{
	/// Addition.
	TVector< T >& operator+=( const TVector< T >& r ) {
	    *this = *this + r;
	    return( *this );
	};
	/// Substraction.
	TVector< T >& operator-=( const TVector< T >& r ) {
	    *this = *this - r;
	    return( *this );
	};
	/// Cross product.
	TVector< T >& operator*=( const TVector< T >& r ) {
	    *this = *this * r;
	    return( *this );
	};
	/// Scalar product.
	TVector< T >& operator*=( T r ) {
	    *this = *this * r;
	    return( *this );
	};
	//@}

	/// Unary operators.
	//@{
	/// Additive inverse.
	TVector< T > operator-( );
	/// Normalized vector.
	TVector< T > operator!( );
	/// Transposed vector.
	TVector< T > operator~( );
	//@}

    private:

	/// Vector's internal state.
	//@{
	/// Memory block.
	T* _vector;
	/// Cardinality.
	uint32_t _N;
	/// Type (ROW_VECTOR/COL_VECTOR).
	int _type;
	/// Have I created _vector?
	bool _myMemory;
	//@}

    };

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T >::TVector( uint32_t N, T data, int type )
    {
	_N        = N;
	_type     = type;
	_vector   = VectorAllocateMemory< T >( _N );
	_myMemory = true;
	VectorAssignScalar< T >( _vector, data, _N );

    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T >::TVector( const TVector< T >& r )
    {
	_N        = r._N;
	_type     = r._type;
	_myMemory = true;
	_vector   = VectorCopyMemory< T >( r._vector, _N );

    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T >::TVector( T* block, uint32_t N, bool copy, int type )
    {
	_N        = N;
	_type     = type;
	_myMemory = copy;
	_vector   = ( copy )? VectorCopyMemory< T >( block, _N ): block;

    }

// -----------------------------------------------------------------------------
    template< class T >
    void TVector< T >::SetN( uint32_t N )
    {
	if( _myMemory ) VectorFreeMemory< T >( _vector );
	_N        = N;
	_vector   = VectorAllocateMemory< T >( _N );
	_myMemory = true;
	VectorAssignScalar< T >( _vector, ( T )0, _N );

    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T >& TVector< T >::operator=( const TVector< T >& r )
    {
	/*
	 * Only assigns the minimum cardinality (WARNING WITH NON-LOCAL MEMORY).
	 */
	_type = r._type;
	VectorAssignMemory< T >( _vector, r._vector, GTM_MIN( _N, r._N ) );
	return( *this );

    }

// -----------------------------------------------------------------------------
    template< class T >
    bool TVector< T >::operator==( const TVector< T >& r )
    {
	uint32_t i;
	bool ret;

	for(
	    i = 0, ret = ( _N == r._N );
	    i < _N && ret;
	    ret &= ( _vector[ i ] == r._vector[ i ] ), i++
	    );
	return( ret );

    }

// -----------------------------------------------------------------------------
    template< class T >
    bool TVector< T >::operator!=( const TVector< T >& r )
    {
	uint32_t i;
	bool ret;

	for(
	    i = 0, ret = ( _N != r._N );
	    i < _N && !ret;
	    ret |= ( _vector[ i ] != r._vector[ i ] ), i++
	    );
	return( ret );

    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T > TVector< T >::operator+( const TVector< T >& r )
    {
	TVector< T > ret( GTM_MIN( _N, r._N ) );
  
	VectorAdd< T >( ret._vector, _vector, r._vector, GTM_MIN( _N, r._N ) );
	return( ret );
  
    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T > TVector< T >::operator-( const TVector< T >& r )
    {
	TVector< T > ret( GTM_MIN( _N, r._N ) );
  
	VectorSubtract< T >( ret._vector, _vector, r._vector, GTM_MIN( _N, r._N ) );
	return( ret );
  
    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T > TVector< T >::operator*( const TVector< T >& r )
    {
	TVector< T > ret( GTM_MIN( _N, r._N ) );
  
	VectorCrossProduct< T >( ret._vector, _vector, r._vector );
	return( ret );
  
    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T > TVector< T >::operator*( T r )
    {
	TVector< T > ret( _N );
  
	VectorScalarProduct< T >( ret._vector, _vector, r, _N );
	return( ret );
  
    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T > TVector< T >::operator-( )
    {
	TVector< T > ret( _N );
	uint32_t i;
  
	for( i = 0; i < _N; ret._vector[ i ] = ( T )0 - _vector[ i ], i++ );
	return( ret );
  
    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T > TVector< T >::operator!( )
    {
	TVector< T > ret( _N );
  
	VectorNormalize< T >( ret._vector, _vector, _N );
	return( ret );
  
    }

// -----------------------------------------------------------------------------
    template< class T >
    TVector< T > TVector< T >::operator~( )
    {
	TVector< T > ret = *this;
  
	ret._type = ( _type == COL_VECTOR )? ROW_VECTOR: COL_VECTOR;
	return( ret );
  
    }

} // namespace

#endif // GTMLIB__MATH__VECTOR__HXX

// EOF - vector.h
