/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sante)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/
#include <vtkPolyData.h>
#include <deque>
#include <memory>

template<class StateType>
class HistoryHandler{
public:
	HistoryHandler(int maxElements);
	~HistoryHandler();
	void 		CleanHistory();
	StateType*	Undo(StateType* State);
	StateType*	UndoKeepCurrent();
	StateType*	Redo(StateType* State);
	StateType*	RedoKeepCurrent();
	void 		Save(StateType* State);
	StateType*	GetPrevious();
	StateType*	GetNext();
	int 		UndoSize();
	int 		RedoSize();
	
private:
	std::deque<StateType*> 	undoStack;
	std::deque<StateType*> 	redoStack;
	int 					maxElements;
};

class MeshModel{

public:
	MeshModel(int id);
	MeshModel(vtkPolyData* mesh, int id);
	MeshModel(MeshModel* meshModel);
	~MeshModel();
	
	void 			SetMeshBase(vtkPolyData* mesh);
    void 			SetMeshMemoryMode(vtkPolyData* mesh);
    void 			CopySetMeshBase(vtkPolyData* mesh);
    void 			ResetMeshTemp();
    void 			ResetMeshTemp_();
    vtkPolyData* 	GetMeshBase();
    vtkPolyData* 	GetMeshTemp();
    int 			GetId();
    std::string 	GetName();
    

protected:

private:
	vtkPolyData 	*_meshBase;
    vtkPolyData 	*_meshTemp;
    int				_meshId;
    std::string		_name;
};

#ifndef _MESHMANAGERMODEL_H_
#define _MESHMANAGERMODEL_H_

//---------------------------------------------
// Class Name: MeshManagerModel
// [classdescription]
//---------------------------------------------

class MeshManagerModel
{

//---------------------------------------------
//Methods and attributes exposed to other classes
//---------------------------------------------
public :
  MeshManagerModel();
  MeshManagerModel(std::vector<vtkPolyData*> meshList);
  ~MeshManagerModel();

	void SetHistory(int maxCapacity);
	void ResetHistory();
	void ResetAll();
	
	void SetReferencePoint(std::vector<double> point);
	std::vector<double> GetReferencePoint();
	void SetReferenceNormal(std::vector<double> normal);
	std::vector<double> GetReferenceNormal();
	
    void SetMeshBase(vtkPolyData* mesh);
    void SetMeshMemoryMode(vtkPolyData* mesh);
	void MeshMemoryModeOn();
    void MeshMemoryModeOff();
    void CopySetMeshBase(vtkPolyData* mesh);
    void ResetMeshTemp();
    //void ResetMeshTemp_();
    vtkPolyData* GetMeshBase();
    vtkPolyData* GetMeshTemp();
    
    virtual void RefreshOutputs(bool signalBox);
    
    std::shared_ptr<MeshModel> GetMeshModel();
    int GetNumberOfMeshes();
    
    void AddMesh_(vtkPolyData* mesh);
    void AddMesh(vtkPolyData* mesh);
    
    void AddEmptyMesh_();
    void AddEmptyMesh();
    
    void AddMeshes(std::vector<vtkPolyData*> meshList);
    void AddMeshes_(std::vector<vtkPolyData*> meshList);
    
    void InsertMeshesAtCurrent_(std::vector<vtkPolyData*> meshList);
    void InsertMeshesAtCurrent(std::vector<vtkPolyData*> meshList);
    
    void InsertMeshModels_(std::vector<MeshModel*> meshModelList);
	void InsertMeshModels(std::vector<MeshModel*> meshModelList);
    
    void SelectMesh(int i);
    void SelectMeshByName(std::string meshName);
    
    void DeleteMesh_(int position);
    void DeleteMesh(int position);
    void DeleteMeshByName(std::string meshName);
    void DeleteCurrentMesh();
    
    void DeleteAll_();
    void DeleteAll();
    
    void ReplaceMesh(std::vector<vtkPolyData*> meshList);
    
    void NextMesh();
    void PreviousMesh();
    
    void SaveMemoryMode();
    void Undo();
    void Redo();
    
    std::vector<std::string> GetMeshNames();
    std::vector<vtkPolyData*> GetAllPolyDatas();
    int  GetCurrentMesh();

//--Method template----------------------------
//  void FunctionName(int& parameterA);

//---------------------------------------------
//Methods and attributes exposed only to classes
//that are derived from this class
//---------------------------------------------
protected:

//---------------------------------------------
//Methods and attributes only visible by this class
//---------------------------------------------
private:
    
    std::vector<std::shared_ptr<MeshModel>> _meshes;
    int 									currentMesh;
    int										meshId;
    int										lastModified;
    bool									memoryMode;
    std::vector<double>						referencePoint;
    std::vector<double>						referenceNormal;
    
    class ManagerState{
    	public:
    		ManagerState(std::vector<std::shared_ptr<MeshModel>> meshesToSave, int meshId, int modifiedId);
    		ManagerState(std::vector<std::shared_ptr<MeshModel>> meshesToSave, int meshId, int modifiedId, std::vector<double> refPoint, std::vector<double> refNormal);
    		~ManagerState();
    		std::vector<std::shared_ptr<MeshModel>>& GetMeshes();
    		int GetMeshId();
    		int GetModifiedPos();
    		std::vector<double>& GetReferencePoint();

    		std::vector<std::shared_ptr<MeshModel>> savedMeshes;
    		int savedModifiedPos;
    		int savedId;
    		std::vector<double> referencePoint;
    		std::vector<double> referenceNormal;
    };
    
    void 							Save();
    void 							RestoreState(ManagerState* state);
    void 							RestoreStateMemoryMode(ManagerState* state);
    
    HistoryHandler<ManagerState>	*history;
    
    
};

//-end of _MESHMANAGERMODEL_H_------------------------------------------------------
#endif
